/*
 * Decompiled with CFR 0.152.
 */
package org.biojava.nbio.structure.xtal;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import javax.vecmath.Matrix4d;
import javax.vecmath.Point3i;
import javax.vecmath.Tuple3d;
import javax.vecmath.Vector3d;
import org.biojava.nbio.structure.Calc;
import org.biojava.nbio.structure.Chain;
import org.biojava.nbio.structure.Group;
import org.biojava.nbio.structure.PDBCrystallographicInfo;
import org.biojava.nbio.structure.Structure;
import org.biojava.nbio.structure.StructureTools;
import org.biojava.nbio.structure.contact.AtomContactSet;
import org.biojava.nbio.structure.contact.StructureInterface;
import org.biojava.nbio.structure.contact.StructureInterfaceList;
import org.biojava.nbio.structure.xtal.CrystalTransform;
import org.biojava.nbio.structure.xtal.UnitCellBoundingBox;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CrystalBuilder {
    public static final String NCS_CHAINID_SUFFIX_CHAR = "n";
    public static final int DEF_NUM_CELLS = 20;
    public static final double DEFAULT_INTERFACE_DISTANCE_CUTOFF = 5.5;
    public static final Matrix4d IDENTITY = new Matrix4d(1.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0, 0.0, 0.0, 0.0, 1.0);
    private static final boolean INCLUDE_HETATOMS = true;
    private Structure structure;
    private PDBCrystallographicInfo crystallographicInfo;
    private int numPolyChainsAu;
    private int numOperatorsSg;
    private Map<String, Matrix4d> chainNcsOps = null;
    private Map<String, String> chainOrigNames = null;
    private static final Logger logger = LoggerFactory.getLogger(CrystalBuilder.class);
    private int numCells;
    private ArrayList<CrystalTransform> visitedCrystalTransforms;
    private Map<String, Map<Matrix4d, StructureInterface>> visitedNcsChainPairs = null;
    private boolean searchBeyondAU;
    private Matrix4d[] ops;

    public CrystalBuilder(Structure structure, Map<String, String> chainOrigNames, Map<String, Matrix4d> chainNcsOps) {
        this(structure);
        this.chainOrigNames = chainOrigNames;
        this.chainNcsOps = chainNcsOps;
    }

    public CrystalBuilder(Structure structure) {
        this.structure = structure;
        this.crystallographicInfo = structure.getCrystallographicInfo();
        this.numPolyChainsAu = structure.getPolyChains().size();
        this.searchBeyondAU = false;
        if (structure.isCrystallographic()) {
            this.searchBeyondAU = true;
            if (this.crystallographicInfo.isNonStandardSg()) {
                logger.warn("Space group is non-standard, will only calculate asymmetric unit interfaces.");
                this.searchBeyondAU = false;
            }
            if (this.crystallographicInfo.getSpaceGroup() == null) {
                logger.warn("Space group is null, will only calculate asymmetric unit interfaces.");
                this.searchBeyondAU = false;
            }
            if (this.crystallographicInfo.getCrystalCell() == null) {
                logger.warn("Could not find a crystal cell definition, will only calculate asymmetric unit interfaces.");
                this.searchBeyondAU = false;
            }
            if (this.crystallographicInfo.isNonStandardCoordFrameConvention()) {
                logger.warn("Non-standard coordinate frame convention, will only calculate asymmetric unit interfaces.");
                this.searchBeyondAU = false;
            }
        }
        if (this.searchBeyondAU) {
            this.numOperatorsSg = this.crystallographicInfo.getSpaceGroup().getMultiplicity();
            this.ops = this.crystallographicInfo.getTransformationsOrthonormal();
        } else {
            this.numOperatorsSg = 1;
            this.ops = new Matrix4d[1];
            this.ops[0] = new Matrix4d(IDENTITY);
        }
        this.numCells = 20;
    }

    public boolean hasNcsOps() {
        return this.chainNcsOps != null;
    }

    public void setNumCells(int numCells) {
        this.numCells = numCells;
    }

    private void initialiseVisited() {
        this.visitedCrystalTransforms = new ArrayList();
        if (this.hasNcsOps()) {
            this.visitedNcsChainPairs = new HashMap<String, Map<Matrix4d, StructureInterface>>();
        }
    }

    public StructureInterfaceList getUniqueInterfaces() {
        return this.getUniqueInterfaces(5.5);
    }

    public StructureInterfaceList getUniqueInterfaces(double cutoff) {
        StructureInterfaceList set = new StructureInterfaceList();
        if (this.numPolyChainsAu == 0) {
            logger.warn("No chains present in the structure! No interfaces will be calculated");
            return set;
        }
        if (this.chainOrigNames != null) {
            set.setChainOrigNamesMap(this.chainOrigNames);
        }
        this.initialiseVisited();
        this.calcInterfacesCrystal(set, cutoff);
        return set;
    }

    private void calcInterfacesCrystal(StructureInterfaceList set, double cutoff) {
        boolean verbose;
        long start = -1L;
        long end = -1L;
        int trialCount = 0;
        int skippedRedundant = 0;
        int skippedAUsNoOverlap = 0;
        int skippedChainsNoOverlap = 0;
        int skippedSelfEquivalent = 0;
        UnitCellBoundingBox bbGrid = new UnitCellBoundingBox(this.numOperatorsSg, this.numPolyChainsAu);
        bbGrid.setBbs(this.structure, this.ops, true);
        if (!this.searchBeyondAU) {
            this.numCells = 0;
        }
        if (verbose = logger.isDebugEnabled()) {
            trialCount = 0;
            start = System.currentTimeMillis();
            int neighbors = (2 * this.numCells + 1) * (2 * this.numCells + 1) * (2 * this.numCells + 1) - 1;
            int auTrials = this.numPolyChainsAu * (this.numPolyChainsAu - 1) / 2;
            int trials = this.numPolyChainsAu * this.numOperatorsSg * this.numPolyChainsAu * neighbors;
            logger.debug("Chain clash trials within original AU: {}", (Object)auTrials);
            logger.debug("Chain clash trials between the original AU and the neighbouring " + neighbors + " whole unit cells (" + this.numCells + " neighbours)(2x" + this.numPolyChainsAu + "chains x " + this.numOperatorsSg + "AUs x " + neighbors + "cells) : " + trials);
            logger.debug("Total trials: {}", (Object)(auTrials + trials));
        }
        List<Chain> polyChains = this.structure.getPolyChains();
        for (int a = -this.numCells; a <= this.numCells; ++a) {
            for (int b = -this.numCells; b <= this.numCells; ++b) {
                for (int c = -this.numCells; c <= this.numCells; ++c) {
                    Point3i trans = new Point3i(a, b, c);
                    Vector3d transOrth = new Vector3d((double)a, (double)b, (double)c);
                    if (a != 0 || b != 0 || c != 0) {
                        this.crystallographicInfo.getCrystalCell().transfToOrthonormal((Tuple3d)transOrth);
                    }
                    UnitCellBoundingBox bbGridTrans = bbGrid.getTranslatedBbs(transOrth);
                    for (int n = 0; n < this.numOperatorsSg; ++n) {
                        if (!bbGrid.getAuBoundingBox(0).overlaps(bbGridTrans.getAuBoundingBox(n), cutoff)) {
                            ++skippedAUsNoOverlap;
                            continue;
                        }
                        CrystalTransform tt = new CrystalTransform(this.crystallographicInfo.getSpaceGroup(), n);
                        tt.translate(trans);
                        if (this.isRedundantTransform(tt)) {
                            ++skippedRedundant;
                            continue;
                        }
                        this.addVisitedTransform(tt);
                        boolean selfEquivalent = false;
                        if (tt.isEquivalent(tt)) {
                            logger.debug("Transform {} is equivalent to itself, will skip half of i-chains to j-chains comparisons", (Object)tt.toString());
                            selfEquivalent = true;
                        }
                        StringBuilder builder = null;
                        if (verbose) {
                            builder = new StringBuilder(String.valueOf(tt)).append(" ");
                        }
                        int contactsFound = 0;
                        for (int j = 0; j < this.numPolyChainsAu; ++j) {
                            for (int i = 0; i < this.numPolyChainsAu; ++i) {
                                StructureInterface interf;
                                if (selfEquivalent && j > i) {
                                    ++skippedSelfEquivalent;
                                    continue;
                                }
                                if (n == 0 && a == 0 && b == 0 && c == 0 && i == j) continue;
                                if (!bbGrid.getChainBoundingBox(0, i).overlaps(bbGridTrans.getChainBoundingBox(n, j), cutoff)) {
                                    ++skippedChainsNoOverlap;
                                    if (!verbose) continue;
                                    builder.append(".");
                                    continue;
                                }
                                ++trialCount;
                                Chain chaini = polyChains.get(i);
                                Chain chainj = polyChains.get(j);
                                if (n != 0 || a != 0 || b != 0 || c != 0) {
                                    Matrix4d mJCryst = new Matrix4d(this.ops[n]);
                                    this.translate(mJCryst, transOrth);
                                    chainj = (Chain)chainj.clone();
                                    Calc.transform(chainj, mJCryst);
                                }
                                if ((interf = this.calcContacts(chaini, chainj, cutoff, tt, builder)) == null) continue;
                                ++contactsFound;
                                if (this.hasNcsOps()) {
                                    StructureInterface interfNcsRef = this.findNcsRef(interf);
                                    set.addNcsEquivalent(interf, interfNcsRef);
                                    continue;
                                }
                                set.add(interf);
                            }
                        }
                        if (!verbose) continue;
                        if (a == 0 && b == 0 && c == 0 && n == 0) {
                            builder.append(" " + contactsFound + "(" + this.numPolyChainsAu * (this.numPolyChainsAu - 1) / 2 + ")");
                        } else if (selfEquivalent) {
                            builder.append(" " + contactsFound + "(" + this.numPolyChainsAu * (this.numPolyChainsAu + 1) / 2 + ")");
                        } else {
                            builder.append(" " + contactsFound + "(" + this.numPolyChainsAu * this.numPolyChainsAu + ")");
                        }
                        logger.debug(builder.toString());
                    }
                }
            }
        }
        end = System.currentTimeMillis();
        logger.debug("\n{} chain-chain clash trials done. Time {}{}s", new Object[]{trialCount, end - start, 1000});
        logger.debug("  skipped (not overlapping AUs)       : {}", (Object)skippedAUsNoOverlap);
        logger.debug("  skipped (not overlapping chains)    : {}", (Object)skippedChainsNoOverlap);
        logger.debug("  skipped (sym redundant op pairs)    : {}", (Object)skippedRedundant);
        logger.debug("  skipped (sym redundant self op)     : {}", (Object)skippedSelfEquivalent);
        logger.debug("Found {} interfaces.", (Object)set.size());
    }

    private StructureInterface findNcsRef(StructureInterface interf) {
        Matrix4d mJCryst;
        if (!this.hasNcsOps()) {
            return null;
        }
        String chainIName = interf.getMoleculeIds().getFirst();
        String iOrigName = this.chainOrigNames.get(chainIName);
        String chainJName = interf.getMoleculeIds().getSecond();
        String jOrigName = this.chainOrigNames.get(chainJName);
        if (this.searchBeyondAU) {
            mJCryst = interf.getTransforms().getSecond().getMatTransform();
            mJCryst = this.crystallographicInfo.getCrystalCell().transfToOrthonormal(mJCryst);
        } else {
            mJCryst = IDENTITY;
        }
        Matrix4d mChainIInv = new Matrix4d(this.chainNcsOps.get(chainIName));
        mChainIInv.invert();
        Matrix4d mJNcs = new Matrix4d(this.chainNcsOps.get(chainJName));
        Matrix4d j2iNcsOrigin = new Matrix4d(mChainIInv);
        j2iNcsOrigin.mul(mJCryst);
        j2iNcsOrigin.mul(mJNcs);
        Matrix4d i2jNcsOrigin = new Matrix4d(j2iNcsOrigin);
        i2jNcsOrigin.invert();
        String matchChainIdsIJ = iOrigName + jOrigName;
        String matchChainIdsJI = jOrigName + iOrigName;
        Optional<Matrix4d> matchDirect = this.visitedNcsChainPairs.computeIfAbsent(matchChainIdsIJ, k -> new HashMap()).entrySet().stream().map(r -> (Matrix4d)r.getKey()).filter(r -> r.epsilonEquals(j2iNcsOrigin, 0.01)).findFirst();
        Matrix4d matchMatrix = matchDirect.orElse(null);
        String matchChainIds = matchChainIdsIJ;
        if (matchMatrix == null) {
            Optional<Matrix4d> matchInverse = this.visitedNcsChainPairs.computeIfAbsent(matchChainIdsJI, k -> new HashMap()).entrySet().stream().map(r -> (Matrix4d)r.getKey()).filter(r -> r.epsilonEquals(i2jNcsOrigin, 0.01)).findFirst();
            matchMatrix = matchInverse.orElse(null);
            matchChainIds = matchChainIdsJI;
        }
        StructureInterface matchInterface = null;
        if (matchMatrix == null) {
            this.visitedNcsChainPairs.get(matchChainIdsIJ).put(j2iNcsOrigin, interf);
        } else {
            matchInterface = this.visitedNcsChainPairs.get(matchChainIds).get(matchMatrix);
        }
        return matchInterface;
    }

    private StructureInterface calcContacts(Chain chaini, Chain chainj, double cutoff, CrystalTransform tt, StringBuilder builder) {
        AtomContactSet graph = StructureTools.getAtomsInContact(chaini, chainj, cutoff, true);
        if (graph.size() > 0) {
            if (builder != null) {
                builder.append("x");
            }
            CrystalTransform transf = new CrystalTransform(this.crystallographicInfo.getSpaceGroup());
            StructureInterface interf = new StructureInterface(StructureTools.getAllAtomArray(chaini), StructureTools.getAllAtomArray(chainj), chaini.getName(), chainj.getName(), graph, transf, tt);
            return interf;
        }
        if (builder != null) {
            builder.append("o");
        }
        return null;
    }

    private void addVisitedTransform(CrystalTransform tt) {
        this.visitedCrystalTransforms.add(tt);
    }

    private boolean isRedundantTransform(CrystalTransform tt) {
        Iterator<CrystalTransform> it = this.visitedCrystalTransforms.iterator();
        while (it.hasNext()) {
            CrystalTransform v = it.next();
            if (!tt.isEquivalent(v)) continue;
            logger.debug("Skipping redundant transformation: " + String.valueOf(tt) + ", equivalent to " + String.valueOf(v));
            it.remove();
            return true;
        }
        return false;
    }

    public void translate(Matrix4d m, Vector3d translation) {
        m.m03 += translation.x;
        m.m13 += translation.y;
        m.m23 += translation.z;
    }

    public static void expandNcsOps(Structure structure, Map<String, String> chainOrigNames, Map<String, Matrix4d> chainNcsOps) {
        PDBCrystallographicInfo xtalInfo = structure.getCrystallographicInfo();
        if (xtalInfo == null) {
            return;
        }
        if (xtalInfo.getNcsOperators() == null || xtalInfo.getNcsOperators().length == 0) {
            return;
        }
        ArrayList<Chain> chainsToAdd = new ArrayList<Chain>();
        Matrix4d identity = new Matrix4d();
        identity.setIdentity();
        Matrix4d[] ncsOps = xtalInfo.getNcsOperators();
        for (Chain c : structure.getChains()) {
            String cOrigId = c.getId();
            String cOrigName = c.getName();
            for (int iOperator = 0; iOperator < ncsOps.length; ++iOperator) {
                Matrix4d m = ncsOps[iOperator];
                Chain clonedChain = (Chain)c.clone();
                String newChainId = cOrigId + (iOperator + 1) + NCS_CHAINID_SUFFIX_CHAR;
                String newChainName = cOrigName + (iOperator + 1) + NCS_CHAINID_SUFFIX_CHAR;
                clonedChain.setId(newChainId);
                clonedChain.setName(newChainName);
                CrystalBuilder.setChainIdsInResidueNumbers(clonedChain, newChainName);
                Calc.transform(clonedChain, m);
                chainsToAdd.add(clonedChain);
                c.getEntityInfo().addChain(clonedChain);
                chainOrigNames.put(newChainName, cOrigName);
                chainNcsOps.put(newChainName, m);
            }
            chainNcsOps.put(cOrigName, identity);
            chainOrigNames.put(cOrigName, cOrigName);
        }
        chainsToAdd.forEach(structure::addChain);
    }

    private static void setChainIdsInResidueNumbers(Chain c, String newChainName) {
        for (Group g : c.getAtomGroups()) {
            g.setResidueNumber(newChainName, g.getResidueNumber().getSeqNum(), g.getResidueNumber().getInsCode());
        }
        for (Group g : c.getSeqResGroups()) {
            if (g.getResidueNumber() == null) continue;
            g.setResidueNumber(newChainName, g.getResidueNumber().getSeqNum(), g.getResidueNumber().getInsCode());
        }
    }
}

