package dev.shortloop.agent;

import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.AllArgsConstructor;
import org.apache.http.HeaderElement;
import org.apache.http.HeaderElementIterator;
import org.apache.http.HeaderIterator;
import org.apache.http.client.HttpRequestRetryHandler;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.conn.ConnectionKeepAliveStrategy;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.DefaultHttpRequestRetryHandler;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.message.BasicHeaderElementIterator;
import org.apache.http.protocol.HTTP;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.web.client.RestTemplateBuilder;
import org.springframework.http.*;
import org.springframework.http.client.BufferingClientHttpRequestFactory;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.stereotype.Component;
import org.springframework.util.ObjectUtils;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.DefaultUriBuilderFactory;
import org.springframework.web.util.UriComponentsBuilder;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Map;
import java.util.concurrent.TimeUnit;


public class ShortloopHttpClient {
    ShortloopHttpClientConfig shortloopHttpClientConfig;

    private ConnectionKeepAliveStrategy connectionKeepAliveStrategy() {
        return (httpResponse, httpContext) -> {
            HeaderIterator headerIterator = httpResponse.headerIterator(HTTP.CONN_KEEP_ALIVE);
            HeaderElementIterator elementIterator = new BasicHeaderElementIterator(headerIterator);

            while (elementIterator.hasNext()) {
                HeaderElement element = elementIterator.nextElement();
                String param = element.getName();
                String value = element.getValue();
                if (value != null && param.equalsIgnoreCase("timeout")) {
                    return Long.parseLong(value) * 1000;
                }
            }
            return shortloopHttpClientConfig.getDefaultKeepAliveTime();
        };
    }

    private CloseableHttpClient getCloseableHttpClient() {
        RequestConfig requestConfig = RequestConfig.
                custom().
                setConnectTimeout(shortloopHttpClientConfig.getConnectTimeout()).
                setConnectionRequestTimeout(shortloopHttpClientConfig.getConnectionRequestTimeout()).
                setSocketTimeout(shortloopHttpClientConfig.getSocketTimeout()).
                build();
        HttpRequestRetryHandler retryHandler = new DefaultHttpRequestRetryHandler(shortloopHttpClientConfig.getRetryCount(), false);
        HttpClientBuilder builder = HttpClientBuilder.create();
        builder.setDefaultRequestConfig(requestConfig);
        builder.setKeepAliveStrategy(connectionKeepAliveStrategy());
        builder.setMaxConnPerRoute(shortloopHttpClientConfig.getMaxConnPerRoute());
        builder.setMaxConnTotal(shortloopHttpClientConfig.getMaxConnTotal());
        builder.setRetryHandler(retryHandler);
        builder.evictIdleConnections(shortloopHttpClientConfig.getIdleConnectionWaitTime(), TimeUnit.MILLISECONDS);
        return builder.build();
    }


    private RestTemplate restTemplate;
    private final ObjectMapper objectMapper;

    public ShortloopHttpClient(ShortloopHttpClientConfig shortloopHttpClientConfig) {
        this.shortloopHttpClientConfig = shortloopHttpClientConfig;
        this.objectMapper = new ObjectMapper();
        HttpComponentsClientHttpRequestFactory clientHttpRequestFactory =
                new HttpComponentsClientHttpRequestFactory();
        clientHttpRequestFactory.setHttpClient(getCloseableHttpClient());

        DefaultUriBuilderFactory defaultUriBuilderFactory = new DefaultUriBuilderFactory();
        defaultUriBuilderFactory.setEncodingMode(DefaultUriBuilderFactory.EncodingMode.NONE);

        this.restTemplate = new RestTemplateBuilder().
                requestFactory(() -> new BufferingClientHttpRequestFactory(clientHttpRequestFactory)).
                uriTemplateHandler(defaultUriBuilderFactory).
                build();
    }

    public <IN, OUT> ResponseEntity<OUT> getRequest(HttpRequest<IN, OUT> requestDetail) throws HttpException {
        return request(requestDetail, HttpMethod.GET);
    }

    public <IN, OUT> ResponseEntity<OUT> postRequest(HttpRequest<IN, OUT> requestDetail) throws HttpException {
        return request(requestDetail, HttpMethod.POST);
    }

    private <IN, OUT> ResponseEntity<OUT> request(HttpRequest<IN, OUT> requestInfo, HttpMethod method) throws HttpException {
        try {
            String url = requestInfo.getUrl();
            if (requestInfo.getQueryParams() != null) {
                url = url + getQueryParams(requestInfo.getQueryParams());;
            }
            HttpHeaders httpHeaders = getHttpHeader(requestInfo.getRequestHeaders());
            ResponseEntity<String> response = restTemplate.exchange(url, method, new HttpEntity<>(requestInfo.getRequestBody(), httpHeaders), String.class);
            OUT res = ObjectUtils.isEmpty(response.getBody()) ? null : objectMapper.readValue(response.getBody(), requestInfo.getResponseType());
            return new ResponseEntity<>(res, response.getHeaders(), response.getStatusCode());
        } catch (Exception e) {
            throw new HttpException(e.getMessage());
        }
    }

    private HttpHeaders getHttpHeader(Map<String, String> headerMap) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        if (!ObjectUtils.isEmpty(headerMap)) {
            for (String key : headerMap.keySet()) {
                headers.set(key, headerMap.get(key));
            }
        }
        return headers;
    }

    public static String getQueryParams(Map<String, String> params) throws UnsupportedEncodingException {
        if (ObjectUtils.isEmpty(params)) {
            return "";
        }
        UriComponentsBuilder componentsBuilder = UriComponentsBuilder.newInstance();
        for (Map.Entry<String, String> entry : params.entrySet()) {
            componentsBuilder.queryParam(entry.getKey(), URLEncoder.encode(entry.getValue(), StandardCharsets.UTF_8.toString()));
        }

        return componentsBuilder.
                build().
                toString();
    }
}

