package dev.shortloop.agent.buffer;

import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import dev.shortloop.agent.ShortloopHttpClient;
import dev.shortloop.agent.HttpRequest;
import dev.shortloop.agent.SDKLogger;
import dev.shortloop.agent.commons.SimpleSuccessResponse;
import dev.shortloop.agent.config.ConfigManager;
import dev.shortloop.common.models.data.APISample;
import dev.shortloop.common.models.data.AgentConfig;
import lombok.NonNull;

import java.util.List;
import java.util.Map;
import java.util.concurrent.Semaphore;
public class DiscoveredApiBufferManager extends AbstractBufferManager {
    private final ShortloopHttpClient shortloopHttpClient;
    private final String ctUrl;
    private final SDKLogger logger;
    public DiscoveredApiBufferManager(ConfigManager configManager, ShortloopHttpClient shortloopHttpClient, String ctUrl, SDKLogger logger) {
        super(configManager, logger);
        this.shortloopHttpClient = shortloopHttpClient;
        this.ctUrl = ctUrl;
        this.logger = logger;
    }

    BufferManagerWorker<ApiBufferKey> createWorker(AgentConfig newConfig) {
        return new DiscoveredApiBufferManagerWorker(newConfig, shortloopHttpClient, logger);
    }

    public class DiscoveredApiBufferManagerWorker extends BufferManagerWorker<ApiBufferKey> {

        private final ShortloopHttpClient shortloopHttpClient;
        private final Semaphore semaphore;

        public DiscoveredApiBufferManagerWorker(@NonNull AgentConfig config, ShortloopHttpClient shortloopHttpClient, SDKLogger logger) {
            super(config, ctUrl, logger);
            semaphore = new Semaphore(config.getDiscoveryBufferSize());
            this.shortloopHttpClient = shortloopHttpClient;
        }

        @Override
        public boolean init() {
            return true;
        }

        public boolean offer(ApiBufferKey apiBufferKey, APISample apiSample) {
            bufferMap.computeIfAbsent(apiBufferKey, (key) -> new SimpleBuffer(getOperatingConfig().getDiscoveryBufferSizePerApi()));
            Buffer buffer = bufferMap.get(apiBufferKey);
            if (buffer != null) {
                return buffer.offer(apiSample);
            } else {
                logger.error("Buffer found null for " + apiBufferKey.getUri());
            }
            return false;
        }

        public boolean canOffer(ApiBufferKey apiBufferKey) {
            if (semaphore.tryAcquire()) {
                boolean canOffer = false;
                if (getOperatingConfig().getCaptureApiSample()) {
                    Buffer buffer = bufferMap.get(apiBufferKey);
                    if (null != buffer) {
                        canOffer = buffer.canOffer();
                    } else {
                        canOffer = bufferMap.size() < getOperatingConfig().getDiscoveryBufferSize();
                    }
                }
                semaphore.release();
                return canOffer;
            }
            return false;
        }
        void syncForKey(ApiBufferKey key) {
            try {
                Buffer buffer = bufferMap.get(key);
                if (null == buffer) {
                    logger.error("Buffer is null for " + key.getUri());
                    return;
                }
                int iterations = buffer.getContentCount();
                if (iterations == 0) {
                    bufferMap.remove(key);
                    return;
                }
                List<APISample> contents = Lists.newArrayList();
                while (iterations -- > 0) {
                    APISample apiSample = buffer.poll();
                    if (null == apiSample) {
                        bufferMap.remove(key);
                        break;
                    }
                    contents.add(apiSample);
                }
                Map<String, String> requestHeaders = Maps.newHashMap();
                requestHeaders.put("Content-Type", "application/json");
                HttpRequest<String, SimpleSuccessResponse> httpRequestDetail =
                        new HttpRequest(ctUrl + getUri(), contents, null, requestHeaders, new TypeReference<SimpleSuccessResponse>() {});
                SimpleSuccessResponse simpleSuccessResponse = (SimpleSuccessResponse)this.shortloopHttpClient.postRequest(httpRequestDetail).getBody();
            } catch (Exception exception) {
                logger.error("Error inside syncForKey for key " + key.getUri(), exception);
            }
        }
    }
}
