package dev.shortloop.agent.buffer;


import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import dev.shortloop.agent.ShortloopHttpClient;
import dev.shortloop.agent.HttpRequest;
import dev.shortloop.agent.SDKLogger;
import dev.shortloop.agent.commons.SimpleSuccessResponse;
import dev.shortloop.agent.config.ConfigManager;
import dev.shortloop.common.models.data.APISample;
import dev.shortloop.common.models.data.AgentConfig;
import lombok.NonNull;
import org.apache.commons.collections4.CollectionUtils;

import java.util.concurrent.Semaphore;

public class RegisteredApiBufferManager extends AbstractBufferManager {

    private final ShortloopHttpClient shortloopHttpClient;
    private final String ctUrl;
    private final SDKLogger logger;
    public RegisteredApiBufferManager(ConfigManager configManager, ShortloopHttpClient shortloopHttpClient, String ctUrl, SDKLogger logger) {
        super(configManager, logger);
        this.shortloopHttpClient = shortloopHttpClient;
        this.ctUrl = ctUrl;
        this.logger = logger;
    }

    BufferManagerWorker<ApiBufferKey> createWorker(AgentConfig newConfig) {
        return new RegisteredApiBufferManagerWorker(newConfig, shortloopHttpClient, logger);
    }

    public class RegisteredApiBufferManagerWorker extends BufferManagerWorker<ApiBufferKey> {

        private final ShortloopHttpClient shortloopHttpClient;
        private final Semaphore semaphore;


        public RegisteredApiBufferManagerWorker(@NonNull AgentConfig config, ShortloopHttpClient shortloopHttpClient, SDKLogger logger) {
            super(config, ctUrl, logger);
            this.shortloopHttpClient = shortloopHttpClient;
            semaphore = new Semaphore(getRegisteredApiCountToCapture());
        }

        @Override
        public boolean init() {
            return true;
        }

        public boolean offer(ApiBufferKey apiBufferKey, APISample apiSample) {
            bufferMap.computeIfAbsent(apiBufferKey, (key) -> new SimpleBuffer(getRegisteredApiBufferSize(apiBufferKey)));
            Buffer buffer = bufferMap.get(apiBufferKey);
            if (buffer != null) {
                return buffer.offer(apiSample);
            } else {
                logger.error("Buffer found null for " + apiBufferKey.getUri());
            }
            return false;
        }

        public boolean canOffer(ApiBufferKey apiBufferKey) {
            if (!getOperatingConfig().getCaptureApiSample()) {
                return false;
            }
            int bufferSize = getRegisteredApiBufferSize(apiBufferKey);
            if (bufferSize == 0) {
                return false;
            }
            Buffer buffer = bufferMap.get(apiBufferKey);
            if (semaphore.tryAcquire()) {
                boolean canOffer = false;
                if (null == buffer) {
                    canOffer = true;
                } else {
                    canOffer = buffer.canOffer();
                }
                semaphore.release();
                return canOffer;
            }
            return false;
        }

        public int getRegisteredApiBufferSize(ApiBufferKey apiBufferKey) {
            AgentConfig agentConfig = getOperatingConfig();
            if (CollectionUtils.isEmpty(agentConfig.getRegisteredApiConfigs())) {
                return 0;
            }
            for (AgentConfig.ApiConfig apiConfig : agentConfig.getRegisteredApiConfigs()) {
                if (apiConfig.getMethod() == apiBufferKey.getMethod() && apiConfig.getUri().equals(apiBufferKey.getUri())) {
                    return apiConfig.getBufferSize();
                }
            }
            return 0;
        }

        public int getRegisteredApiCountToCapture() {
            AgentConfig agentConfig = getOperatingConfig();
            if (CollectionUtils.isEmpty(agentConfig.getRegisteredApiConfigs())) {
                return 0;
            }
            int totalApis = 0;
            for (AgentConfig.ApiConfig apiConfig : agentConfig.getRegisteredApiConfigs()) {
                if (null != apiConfig.getBufferSize()) {
                    totalApis += apiConfig.getBufferSize();
                }
            }
            return totalApis;

        }
        void syncForKey(ApiBufferKey key) {
            try {
                Buffer buffer = bufferMap.get(key);
                if (null == buffer) {
                    logger.error("Buffer is null for " + key.getUri());
                    return;
                }
                int iterations = buffer.getContentCount();
                while (iterations -- > 0) {
                    APISample apiSample = buffer.poll();
                    if (null == apiSample) {
                        break;
                    }
                    HttpRequest<String, SimpleSuccessResponse> httpRequestDetail =
                            new HttpRequest(ctUrl + getUri(), Lists.newArrayList(apiSample), null, null, new TypeReference<SimpleSuccessResponse>() {});
                    this.shortloopHttpClient.postRequest(httpRequestDetail);
                }
            } catch (Exception exception) {
                logger.error("Error inside syncForKey for key " + key.getUri(), exception);
            }
        }
    }

}
