/*
 * Decompiled with CFR 0.152.
 */
package dev.vanengine.core;

import com.fasterxml.jackson.databind.ObjectMapper;
import dev.vanengine.core.NativeBinaryResolver;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class VanCompiler
implements AutoCloseable {
    private static final Logger log = LoggerFactory.getLogger(VanCompiler.class);
    private final ObjectMapper objectMapper = new ObjectMapper();
    private final ConcurrentHashMap<String, CacheEntry> cache = new ConcurrentHashMap();
    private Process process;
    private BufferedWriter processStdin;
    private BufferedReader processStdout;

    public void init() throws Exception {
        Path binary = new NativeBinaryResolver().resolve();
        ProcessBuilder pb = new ProcessBuilder(binary.toString(), "--daemon");
        pb.redirectErrorStream(false);
        this.process = pb.start();
        this.processStdin = new BufferedWriter(new OutputStreamWriter(this.process.getOutputStream(), StandardCharsets.UTF_8));
        this.processStdout = new BufferedReader(new InputStreamReader(this.process.getInputStream(), StandardCharsets.UTF_8));
        log.info("van-compiler daemon started (pid {})", (Object)this.process.pid());
    }

    @Override
    public void close() {
        if (this.process != null && this.process.isAlive()) {
            log.info("Stopping van-compiler daemon (pid {})", (Object)this.process.pid());
            try {
                this.processStdin.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
            this.process.destroy();
        }
    }

    public CompiledResult compile(Path vanFile, Path basePath) throws IOException {
        String key = vanFile.toAbsolutePath().toString();
        long mtime = Files.getLastModifiedTime(vanFile, new LinkOption[0]).toMillis();
        CacheEntry entry = this.cache.get(key);
        if (entry != null && entry.mtime == mtime) {
            return entry.result;
        }
        Map<String, String> files = this.readVanFiles(vanFile, basePath);
        String entryPath = basePath.relativize(vanFile.toAbsolutePath()).toString().replace(File.separatorChar, '/');
        CompiledResult result = this.doCompile(entryPath, files);
        this.cache.put(key, new CacheEntry(result, mtime));
        return result;
    }

    public CompiledResult compile(String entryPath, Map<String, String> files) throws IOException {
        return this.doCompile(entryPath, files);
    }

    private synchronized CompiledResult doCompile(String entryPath, Map<String, String> files) throws IOException {
        this.ensureProcessAlive();
        HashMap<String, Object> request = new HashMap<String, Object>();
        request.put("entry_path", entryPath);
        request.put("files", files);
        request.put("mock_data_json", "{}");
        String jsonLine = this.objectMapper.writeValueAsString(request);
        this.processStdin.write(jsonLine);
        this.processStdin.newLine();
        this.processStdin.flush();
        String responseLine = this.processStdout.readLine();
        if (responseLine == null) {
            throw new IOException("van-compiler daemon terminated unexpectedly");
        }
        Map response = (Map)this.objectMapper.readValue(responseLine, Map.class);
        boolean ok = Boolean.TRUE.equals(response.get("ok"));
        if (!ok) {
            String error = (String)response.get("error");
            throw new IOException("van-compiler error: " + error);
        }
        String html = (String)response.get("html");
        Map assets = (Map)response.get("assets");
        return new CompiledResult(html, assets);
    }

    private void ensureProcessAlive() throws IOException {
        if (this.process == null || !this.process.isAlive()) {
            log.warn("van-compiler daemon is not running, restarting...");
            try {
                this.init();
            }
            catch (Exception e) {
                throw new IOException("Failed to restart van-compiler daemon", e);
            }
        }
    }

    private Map<String, String> readVanFiles(Path entryFile, Path basePath) throws IOException {
        HashMap<String, String> files = new HashMap<String, String>();
        this.collectVanFiles(entryFile, basePath, files);
        return files;
    }

    private void collectVanFiles(Path file, Path basePath, Map<String, String> files) throws IOException {
        String relativePath = basePath.relativize(file.toAbsolutePath()).toString().replace(File.separatorChar, '/');
        if (files.containsKey(relativePath)) {
            return;
        }
        String content = Files.readString(file, StandardCharsets.UTF_8);
        files.put(relativePath, content);
        for (String importPath : VanCompiler.parseImportPaths(content)) {
            Path importFile = file.getParent().resolve(importPath).normalize();
            if (!Files.isRegularFile(importFile, new LinkOption[0])) continue;
            this.collectVanFiles(importFile, basePath, files);
        }
    }

    public static List<String> parseImportPaths(String content) {
        ArrayList<String> paths = new ArrayList<String>();
        int scriptStart = content.indexOf("<script setup>");
        if (scriptStart == -1) {
            return paths;
        }
        int scriptEnd = content.indexOf("</script>", scriptStart);
        if (scriptEnd == -1) {
            return paths;
        }
        String script = content.substring(scriptStart + "<script setup>".length(), scriptEnd);
        Pattern pattern = Pattern.compile("import\\s+\\w+\\s+from\\s+['\"]([^'\"]+\\.van)['\"]");
        Matcher matcher = pattern.matcher(script);
        while (matcher.find()) {
            paths.add(matcher.group(1));
        }
        return paths;
    }

    record CacheEntry(CompiledResult result, long mtime) {
    }

    public record CompiledResult(String html, Map<String, String> assets) {
    }
}

