package fi.evolver.script.app;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;

import fi.evolver.script.FileUtils;
import fi.evolver.script.Shell;
import fi.evolver.script.Step;

/**
 * Note that there is no ARM Linux version of Chrome available.
 * Chromium is available e.g. through Debian repositories.
 */
public class Chromium {

	private static final String EXTERNAL_EXTENSION_JSON = """
			{
				"external_update_url": "https://clients2.google.com/service/update2/crx"
			}
			""";
	private static final Path CONFIG_DIR = Shell.HOME.resolve(".config/chromium");

	public static final String EXTENSION_REACT_DEVELOPER_TOOLS = "fmkadmapgofadopljbjfkapdkoienihi";
	public static final String EXTENSION_REDUX_DEV_TOOLS = "lmhkpmbekcpmknklioeibfkpmmfibljd";

	public static void install() {
		try (Step step = Step.start("Chromium: Install")) {
			Apt.install("chromium");
		}
	}

	public static void addExtensions(List<String> extensionIds) throws IOException {
		try (Step step = Step.start("Chromium: Add extensions")) {
			Path externalExtensionsDir = CONFIG_DIR.resolve("External Extensions");
			if (Files.exists(externalExtensionsDir)) {
				step.skip("External extensions already configured");
				return;
			}

			Files.createDirectories(externalExtensionsDir);

			for (String extensionId : extensionIds) {
				Path extensionFile = externalExtensionsDir.resolve(extensionId + ".json");
				FileUtils.write(extensionFile, EXTERNAL_EXTENSION_JSON);
			}
		}
	}

	public static void installBookmarks(Path bookmarksSourcePath) throws IOException {
		try (Step step = Step.start("Chromium: Install Bookmarks")) {
			Path defaultProfileDir = CONFIG_DIR.resolve("Default");
			Path targetFile = defaultProfileDir.resolve("Bookmarks");

			if (Files.exists(targetFile)) {
				step.skip("Bookmarks already exist");
				return;
			}

			Files.createDirectories(defaultProfileDir);
			Files.copy(
					bookmarksSourcePath,
					targetFile
			);
		}
	}

}

