/*
 * Decompiled with CFR 0.152.
 */
package org.biojava.nbio.structure.secstruc;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.biojava.nbio.structure.Atom;
import org.biojava.nbio.structure.Calc;
import org.biojava.nbio.structure.Chain;
import org.biojava.nbio.structure.Group;
import org.biojava.nbio.structure.ResidueNumber;
import org.biojava.nbio.structure.Structure;
import org.biojava.nbio.structure.StructureException;
import org.biojava.nbio.structure.StructureTools;
import org.biojava.nbio.structure.contact.AtomContact;
import org.biojava.nbio.structure.contact.AtomContactSet;
import org.biojava.nbio.structure.contact.Grid;
import org.biojava.nbio.structure.contact.Pair;
import org.biojava.nbio.structure.secstruc.BetaBridge;
import org.biojava.nbio.structure.secstruc.BridgeType;
import org.biojava.nbio.structure.secstruc.HBond;
import org.biojava.nbio.structure.secstruc.Ladder;
import org.biojava.nbio.structure.secstruc.SecStrucGroup;
import org.biojava.nbio.structure.secstruc.SecStrucState;
import org.biojava.nbio.structure.secstruc.SecStrucType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SecStrucCalc {
    private static final boolean DSSP_HELICES = true;
    private static final Logger logger = LoggerFactory.getLogger(SecStrucCalc.class);
    public static final double MINDIST = 0.5;
    public static final double CA_MIN_DIST = 9.0;
    public static final double MAX_PEPTIDE_BOND_LENGTH = 2.5;
    public static final int HBONDLOWENERGY = -9900;
    public static final double HBONDHIGHENERGY = -500.0;
    public static final double Q = -27888.0;
    private SecStrucGroup[] groups;
    private List<Ladder> ladders = new ArrayList<Ladder>();
    private List<BetaBridge> bridges = new ArrayList<BetaBridge>();
    private Atom[] atoms;
    private AtomContactSet contactSet;
    private Map<ResidueNumber, Integer> indResMap;

    public List<SecStrucState> calculate(Structure s, boolean assign) throws StructureException {
        ArrayList<SecStrucState> secstruc = new ArrayList<SecStrucState>();
        for (int i = 0; i < s.nrModels(); ++i) {
            this.ladders = new ArrayList<Ladder>();
            this.bridges = new ArrayList<BetaBridge>();
            this.groups = SecStrucCalc.initGroupArray(s, i);
            this.initContactSet();
            if (this.groups.length < 5) {
                throw new StructureException("Not enough backbone groups in the Structure to calculate the secondary structure (" + this.groups.length + " given, minimum 5)");
            }
            this.calculateHAtoms();
            this.calculateHBonds();
            this.calculateDihedralAngles();
            this.calculateTurns();
            this.buildHelices();
            this.detectBends();
            this.detectStrands();
            for (SecStrucGroup sg : this.groups) {
                SecStrucState ss = (SecStrucState)sg.getProperty("secstruc");
                secstruc.add(ss);
                if (!assign) continue;
                sg.getOriginal().setProperty("secstruc", ss);
            }
        }
        return secstruc;
    }

    private void initContactSet() {
        this.atoms = new Atom[this.groups.length];
        this.indResMap = new HashMap<ResidueNumber, Integer>();
        for (int i = 0; i < this.groups.length; ++i) {
            SecStrucGroup one = this.groups[i];
            this.indResMap.put(one.getResidueNumber(), i);
            this.atoms[i] = one.getCA();
        }
        Grid grid = new Grid(9.0);
        if (this.atoms.length == 0) {
            this.contactSet = new AtomContactSet(9.0);
        } else {
            grid.addAtoms(this.atoms);
            this.contactSet = grid.getAtomContacts();
        }
    }

    private void detectStrands() {
        this.findBridges();
        this.createLadders();
        this.connectLadders();
        this.updateSheets();
    }

    private void createLadders() {
        for (BetaBridge b : this.bridges) {
            boolean found = false;
            for (Ladder ladder : this.ladders) {
                if (!this.shouldExtendLadder(ladder, b)) continue;
                found = true;
                ++ladder.to;
                switch (b.type) {
                    case parallel: {
                        ++ladder.lto;
                        break;
                    }
                    case antiparallel: {
                        --ladder.lfrom;
                    }
                }
                break;
            }
            if (found) continue;
            Ladder l = new Ladder();
            l.from = b.partner1;
            l.to = b.partner1;
            l.lfrom = b.partner2;
            l.lto = b.partner2;
            l.btype = b.type;
            this.ladders.add(l);
        }
    }

    private void updateSheets() {
        logger.debug(" got " + this.ladders.size() + "  ladders!");
        for (Ladder ladder : this.ladders) {
            int lcount;
            logger.debug(ladder.toString());
            for (int lcount2 = ladder.from; lcount2 <= ladder.to; ++lcount2) {
                SecStrucState state = this.getSecStrucState(lcount2);
                SecStrucType stype = state.getType();
                int diff = ladder.from - lcount2;
                int l2count = ladder.lfrom - diff;
                SecStrucState state2 = this.getSecStrucState(l2count);
                SecStrucType stype2 = state2.getType();
                if (ladder.from != ladder.to) {
                    this.setSecStrucType(lcount2, SecStrucType.extended);
                    this.setSecStrucType(l2count, SecStrucType.extended);
                    continue;
                }
                if (!stype.isHelixType() && !stype.equals((Object)SecStrucType.extended)) {
                    this.setSecStrucType(lcount2, SecStrucType.bridge);
                }
                if (stype2.isHelixType() || stype2.equals((Object)SecStrucType.extended)) continue;
                this.setSecStrucType(l2count, SecStrucType.bridge);
            }
            if (ladder.connectedTo == 0) continue;
            Ladder conladder = this.ladders.get(ladder.connectedTo);
            if (ladder.btype.equals((Object)BridgeType.antiparallel)) {
                for (lcount = ladder.from; lcount <= conladder.to; ++lcount) {
                    this.setSecStrucType(lcount, SecStrucType.extended);
                }
                for (lcount = conladder.lto; lcount <= ladder.lfrom; ++lcount) {
                    this.setSecStrucType(lcount, SecStrucType.extended);
                }
                continue;
            }
            for (lcount = ladder.from; lcount <= conladder.to; ++lcount) {
                this.setSecStrucType(lcount, SecStrucType.extended);
            }
            for (lcount = ladder.lfrom; lcount <= conladder.lto; ++lcount) {
                this.setSecStrucType(lcount, SecStrucType.extended);
            }
        }
    }

    private void connectLadders() {
        for (int i = 0; i < this.ladders.size(); ++i) {
            for (int j = i; j < this.ladders.size(); ++j) {
                Ladder l2;
                Ladder l1 = this.ladders.get(i);
                if (!this.hasBulge(l1, l2 = this.ladders.get(j))) continue;
                l1.connectedTo = j;
                l2.connectedFrom = i;
                logger.debug("Bulge from " + i + " to " + j);
            }
        }
    }

    private boolean hasBulge(Ladder l1, Ladder l2) {
        boolean bulge;
        boolean bl = bulge = l1.btype.equals((Object)l2.btype) && l2.from - l1.to < 6 && l1.to < l2.from && l2.connectedTo == 0;
        if (!bulge) {
            return bulge;
        }
        switch (l1.btype) {
            case parallel: {
                bulge = l2.lfrom - l1.lto > 0 && (l2.lfrom - l1.lto < 6 && l2.from - l1.to < 3 || l2.lfrom - l1.lto < 3);
                break;
            }
            case antiparallel: {
                bulge = l1.lfrom - l2.lto > 0 && (l1.lfrom - l2.lto < 6 && l2.from - l1.to < 3 || l1.lfrom - l2.lto < 3);
            }
        }
        return bulge;
    }

    private void registerBridge(int i, int j, BridgeType btype) {
        BetaBridge bridge = new BetaBridge(i, j, btype);
        boolean b1 = this.getSecStrucState(i).addBridge(bridge);
        boolean b2 = this.getSecStrucState(j).addBridge(bridge);
        if (!b1 && !b2) {
            logger.warn("Ignoring Bridge between residues" + i + " and " + j + ". DSSP assignment might differ.");
        }
        this.bridges.add(bridge);
    }

    private boolean shouldExtendLadder(Ladder ladder, BetaBridge b) {
        boolean sequential;
        boolean sameType = b.type.equals((Object)ladder.btype);
        if (!sameType) {
            return false;
        }
        boolean bl = sequential = b.partner1 == ladder.to + 1;
        if (!sequential) {
            return false;
        }
        switch (b.type) {
            case parallel: {
                if (b.partner2 != ladder.lto + 1) break;
                return true;
            }
            case antiparallel: {
                if (b.partner2 != ladder.lfrom - 1) break;
                return true;
            }
        }
        return false;
    }

    private void findBridges() {
        Iterator<AtomContact> myIter = this.contactSet.iterator();
        ArrayList<Pair<Integer>> outList = new ArrayList<Pair<Integer>>();
        while (myIter.hasNext()) {
            int j;
            AtomContact ac = myIter.next();
            Group group = ac.getPair().getFirst().getGroup();
            Group g2 = ac.getPair().getSecond().getGroup();
            int i = this.indResMap.get(group.getResidueNumber());
            if (i > (j = this.indResMap.get(g2.getResidueNumber()).intValue())) {
                int old = i;
                i = j;
                j = old;
            }
            if (j < i + 3 || i == 0 || j == 0 || i == this.groups.length - 1 || j == this.groups.length - 1) continue;
            Pair<Integer> thisPair = new Pair<Integer>(i, j);
            outList.add(thisPair);
        }
        Collections.sort(outList, new Comparator<Pair<Integer>>(){

            @Override
            public int compare(Pair<Integer> o1, Pair<Integer> o2) {
                if (o1.getFirst() < o2.getFirst()) {
                    return -1;
                }
                if (o1.getFirst() > o2.getFirst()) {
                    return 1;
                }
                if (o1.getSecond() < o2.getSecond()) {
                    return -1;
                }
                if (o1.getSecond() > o2.getSecond()) {
                    return 1;
                }
                return 0;
            }
        });
        for (Pair pair : outList) {
            int i = (Integer)pair.getFirst();
            int j = (Integer)pair.getSecond();
            BridgeType btype = null;
            if (this.isBonded(i - 1, j) && this.isBonded(j, i + 1) || this.isBonded(j - 1, i) && this.isBonded(i, j + 1)) {
                btype = BridgeType.parallel;
            } else if (this.isBonded(i, j) && this.isBonded(j, i) || this.isBonded(i - 1, j + 1) && this.isBonded(j - 1, i + 1)) {
                btype = BridgeType.antiparallel;
            }
            if (btype == null) continue;
            this.registerBridge(i, j, btype);
        }
    }

    private void detectBends() {
        for (int i = 2; i < this.groups.length - 2; ++i) {
            boolean bonded = true;
            for (int k = 0; k < 4; ++k) {
                Atom N;
                int index = i + k - 2;
                Atom C = this.groups[index].getC();
                if (!(Calc.getDistance(C, N = this.groups[index + 1].getN()) > 2.5)) continue;
                bonded = false;
                break;
            }
            if (!bonded) continue;
            SecStrucGroup im2 = this.groups[i - 2];
            SecStrucGroup g = this.groups[i];
            SecStrucGroup ip2 = this.groups[i + 2];
            Atom caim2 = im2.getCA();
            Atom cag = g.getCA();
            Atom caip2 = ip2.getCA();
            Atom caminus2 = Calc.subtract(caim2, cag);
            Atom caplus2 = Calc.subtract(cag, caip2);
            double angle = Calc.angle(caminus2, caplus2);
            SecStrucState state = this.getSecStrucState(i);
            state.setKappa((float)angle);
            if (!(angle > 70.0) || !(angle < 359.99)) continue;
            this.setSecStrucType(i, SecStrucType.bend);
            state.setBend(true);
        }
    }

    private void calculateDihedralAngles() {
        for (int i = 0; i < this.groups.length - 1; ++i) {
            SecStrucGroup a = this.groups[i];
            SecStrucGroup b = this.groups[i + 1];
            Atom a_N = a.getN();
            Atom a_CA = a.getCA();
            Atom a_C = a.getC();
            Atom b_N = b.getN();
            Atom b_CA = b.getCA();
            Atom b_C = b.getC();
            double phi = Calc.torsionAngle(a_C, b_N, b_CA, b_C);
            double psi = Calc.torsionAngle(a_N, a_CA, a_C, b_N);
            double omega = Calc.torsionAngle(a_CA, a_C, b_N, b_CA);
            SecStrucState state1 = (SecStrucState)a.getProperty("secstruc");
            SecStrucState state2 = (SecStrucState)b.getProperty("secstruc");
            state2.setPhi(phi);
            state1.setPsi(psi);
            state1.setOmega(omega);
        }
    }

    public String toString() {
        return this.printDSSP();
    }

    public String printDSSP() {
        StringBuffer buf = new StringBuffer();
        String nl = System.getProperty("line.separator");
        buf.append("==== Secondary Structure Definition by BioJava DSSP implementation, Version October 2015 ====" + nl);
        buf.append("  #  RESIDUE AA STRUCTURE BP1 BP2  ACC     N-H-->O    O-->H-N    N-H-->O    O-->H-N    TCO  KAPPA ALPHA  PHI    PSI    X-CA   Y-CA   Z-CA ");
        for (int i = 0; i < this.groups.length; ++i) {
            buf.append(nl);
            SecStrucState ss = this.getSecStrucState(i);
            buf.append(ss.printDSSPline(i));
        }
        return buf.toString();
    }

    public String printHelixSummary() {
        StringBuffer g = new StringBuffer();
        StringBuffer h = new StringBuffer();
        StringBuffer i = new StringBuffer();
        StringBuffer ss = new StringBuffer();
        StringBuffer aa = new StringBuffer();
        String nl = System.getProperty("line.separator");
        g.append("3 turn: ");
        h.append("4 turn: ");
        i.append("5 turn: ");
        ss.append("SS:     ");
        aa.append("AA:     ");
        for (int k = 0; k < this.groups.length; ++k) {
            SecStrucState state = this.getSecStrucState(k);
            g.append(state.getTurn()[0]);
            h.append(state.getTurn()[1]);
            i.append(state.getTurn()[2]);
            ss.append((Object)state.getType());
            aa.append(StructureTools.get1LetterCode(this.groups[k].getPDBName()));
        }
        return g.toString() + nl + h.toString() + nl + i.toString() + nl + ss.toString() + nl + aa.toString();
    }

    public String printFASTA() {
        StringBuffer buf = new StringBuffer();
        String nl = System.getProperty("line.separator");
        buf.append(">" + this.groups[0].getChain().getStructure().getIdentifier() + nl);
        for (int g = 0; g < this.groups.length; ++g) {
            buf.append((Object)this.getSecStrucState(g).getType());
        }
        return buf.toString();
    }

    public int hashCode() {
        int prime = 31;
        int result = 1;
        result = 31 * result + Arrays.hashCode(this.atoms);
        return result;
    }

    public boolean equals(Object o) {
        if (!(o instanceof SecStrucCalc)) {
            return false;
        }
        SecStrucCalc ss = (SecStrucCalc)o;
        if (this.groups.length != ss.groups.length) {
            return false;
        }
        for (int g = 0; g < this.groups.length; ++g) {
            SecStrucState g2;
            SecStrucState g1 = this.getSecStrucState(g);
            if (g1.equals(g2 = ss.getSecStrucState(g))) continue;
            return false;
        }
        return true;
    }

    private static SecStrucGroup[] initGroupArray(Structure s, int modelId) {
        ArrayList<SecStrucGroup> groupList = new ArrayList<SecStrucGroup>();
        for (Chain c : s.getChains(modelId)) {
            for (Group g : c.getAtomGroups()) {
                if (!g.hasAminoAtoms()) continue;
                SecStrucGroup sg = new SecStrucGroup();
                sg.setResidueNumber(g.getResidueNumber());
                sg.setPDBFlag(true);
                sg.setPDBName(g.getPDBName());
                sg.setChain(g.getChain());
                Atom N = g.getAtom("N");
                Atom CA = g.getAtom("CA");
                Atom C = g.getAtom("C");
                Atom O = g.getAtom("O");
                if (N == null || CA == null || C == null || O == null) continue;
                sg.setN((Atom)N.clone());
                sg.setCA((Atom)CA.clone());
                sg.setC((Atom)C.clone());
                sg.setO((Atom)O.clone());
                sg.setOriginal(g);
                SecStrucState state = new SecStrucState(sg, "BIOJAVA_ASSIGNMENT", SecStrucType.coil);
                sg.setProperty("secstruc", state);
                groupList.add(sg);
            }
        }
        return groupList.toArray(new SecStrucGroup[groupList.size()]);
    }

    private void calculateHAtoms() throws StructureException {
        for (int i = 0; i < this.groups.length - 1; ++i) {
            SecStrucGroup a = this.groups[i];
            SecStrucGroup b = this.groups[i + 1];
            if (b.hasAtom("H")) continue;
            Atom H = SecStrucCalc.calcSimple_H(a.getC(), a.getO(), b.getN());
            b.setH(H);
        }
    }

    private void calculateHBonds() {
        if (this.groups.length < 5) {
            return;
        }
        for (AtomContact ac : this.contactSet) {
            Pair<Atom> pair = ac.getPair();
            Group g1 = pair.getFirst().getGroup();
            Group g2 = pair.getSecond().getGroup();
            int i = this.indResMap.get(g1.getResidueNumber());
            int j = this.indResMap.get(g2.getResidueNumber());
            this.checkAddHBond(i, j);
            if (j == i + 1) continue;
            this.checkAddHBond(j, i);
        }
    }

    private void checkAddHBond(int i, int j) {
        SecStrucGroup one = this.groups[i];
        if (one.getPDBName().equals("PRO")) {
            logger.debug("Ignore: PRO " + one.getResidueNumber());
            return;
        }
        if (!one.hasAtom("H")) {
            logger.debug("Residue " + one.getResidueNumber() + " has no H");
            return;
        }
        SecStrucGroup two = this.groups[j];
        double energy = 0.0;
        try {
            energy = SecStrucCalc.calculateHBondEnergy(one, two);
        }
        catch (Exception e) {
            logger.warn("Energy calculation failed", (Throwable)e);
            return;
        }
        logger.debug("Energy between positions (" + i + "," + j + "): " + energy);
        this.trackHBondEnergy(i, j, energy);
    }

    private static double calculateHBondEnergy(SecStrucGroup one, SecStrucGroup two) {
        Atom N = one.getN();
        Atom H = one.getH();
        Atom O = two.getO();
        Atom C = two.getC();
        double dno = Calc.getDistance(O, N);
        double dhc = Calc.getDistance(C, H);
        double dho = Calc.getDistance(O, H);
        double dnc = Calc.getDistance(C, N);
        logger.debug("     cccc: " + one.getResidueNumber() + " " + one.getPDBName() + " " + two.getResidueNumber() + " " + two.getPDBName() + String.format(" O (" + O.getPDBserial() + ")..N (" + N.getPDBserial() + "):%4.1f  |  ho:%4.1f - hc:%4.1f + nc:%4.1f - no:%4.1f ", dno, dho, dhc, dnc, dno));
        if (dno < 0.5 || dhc < 0.5 || dnc < 0.5 || dno < 0.5) {
            return -9900.0;
        }
        double e1 = -27888.0 / dho - -27888.0 / dhc;
        double e2 = -27888.0 / dnc - -27888.0 / dno;
        double energy = e1 + e2;
        logger.debug(String.format("      N (%d) O(%d): %4.1f : %4.2f ", N.getPDBserial(), O.getPDBserial(), Float.valueOf((float)dno), energy));
        if (energy > -9900.0) {
            return energy;
        }
        return -9900.0;
    }

    private void trackHBondEnergy(int i, int j, double energy) {
        HBond bond;
        if (this.groups[i].getPDBName().equals("PRO")) {
            logger.debug("Ignore: PRO " + this.groups[i].getResidueNumber());
            return;
        }
        SecStrucState stateOne = this.getSecStrucState(i);
        SecStrucState stateTwo = this.getSecStrucState(j);
        double acc1e = stateOne.getAccept1().getEnergy();
        double acc2e = stateOne.getAccept2().getEnergy();
        double don1e = stateTwo.getDonor1().getEnergy();
        double don2e = stateTwo.getDonor2().getEnergy();
        if (energy < acc1e) {
            logger.debug(energy + "<" + acc1e);
            stateOne.setAccept2(stateOne.getAccept1());
            bond = new HBond();
            bond.setEnergy(energy);
            bond.setPartner(j);
            stateOne.setAccept1(bond);
        } else if (energy < acc2e) {
            logger.debug(energy + "<" + acc2e);
            bond = new HBond();
            bond.setEnergy(energy);
            bond.setPartner(j);
            stateOne.setAccept2(bond);
        }
        if (energy < don1e) {
            logger.debug(energy + "<" + don1e);
            stateTwo.setDonor2(stateTwo.getDonor1());
            bond = new HBond();
            bond.setEnergy(energy);
            bond.setPartner(i);
            stateTwo.setDonor1(bond);
        } else if (energy < don2e) {
            logger.debug(energy + "<" + don2e);
            bond = new HBond();
            bond.setEnergy(energy);
            bond.setPartner(i);
            stateTwo.setDonor2(bond);
        }
    }

    private void calculateTurns() {
        for (int i = 0; i < this.groups.length; ++i) {
            for (int turn = 3; turn <= 5; ++turn) {
                if (i + turn >= this.groups.length || !this.isBonded(i, i + turn)) continue;
                logger.debug("Turn at (" + i + "," + (i + turn) + ") turn " + turn);
                this.getSecStrucState(i).setTurn('>', turn);
                this.getSecStrucState(i + turn).setTurn('<', turn);
                for (int j = i + 1; j < i + turn; ++j) {
                    Integer t = turn;
                    char helix = t.toString().charAt(0);
                    this.getSecStrucState(j).setTurn(helix, turn);
                }
            }
        }
    }

    private boolean isBonded(int i, int j) {
        boolean hbond;
        SecStrucState one = this.getSecStrucState(i);
        SecStrucState two = this.getSecStrucState(j);
        double don1e = one.getDonor1().getEnergy();
        double don2e = one.getDonor2().getEnergy();
        double acc1e = two.getAccept1().getEnergy();
        double acc2e = two.getAccept2().getEnergy();
        int don1p = one.getDonor1().getPartner();
        int don2p = one.getDonor2().getPartner();
        int acc1p = two.getAccept1().getPartner();
        int acc2p = two.getAccept2().getPartner();
        boolean bl = hbond = don1p == j && don1e < -500.0 || don2p == j && don2e < -500.0 || acc1p == i && acc1e < -500.0 || acc2p == i && acc2e < -500.0;
        if (hbond) {
            logger.debug("*** H-bond from CO of " + i + " to NH of " + j);
            return true;
        }
        return false;
    }

    private static Atom calc_H(Atom C, Atom N, Atom CA) throws StructureException {
        Atom nc = Calc.subtract(N, C);
        Atom nca = Calc.subtract(N, CA);
        Atom u_nc = Calc.unitVector(nc);
        Atom u_nca = Calc.unitVector(nca);
        Atom added = Calc.add(u_nc, u_nca);
        Atom U = Calc.unitVector(added);
        Atom H = Calc.add(N, U);
        H.setName("H");
        return H;
    }

    private static Atom calcSimple_H(Atom c, Atom o, Atom n) {
        Atom h = Calc.subtract(c, o);
        double dist = Calc.getDistance(o, c);
        double x = n.getX() + h.getX() / dist;
        double y = n.getY() + h.getY() / dist;
        double z = n.getZ() + h.getZ() / dist;
        h.setX(x);
        h.setY(y);
        h.setZ(z);
        h.setName("H");
        return h;
    }

    private void buildHelices() {
        this.checkSetHelix(4, SecStrucType.helix4);
        this.checkSetHelix(3, SecStrucType.helix3);
        this.checkSetHelix(5, SecStrucType.helix5);
        this.checkSetTurns();
    }

    private void checkSetTurns() {
        SecStrucType type = SecStrucType.turn;
        for (int idx = 0; idx < 3; ++idx) {
            for (int i = 0; i < this.groups.length - 1; ++i) {
                SecStrucState state = this.getSecStrucState(i);
                char[] turn = state.getTurn();
                if (turn[idx] != '>' && turn[idx] != 'X') continue;
                for (int k = 1; k < idx + 3; ++k) {
                    this.setSecStrucType(i + k, type);
                }
            }
        }
    }

    private void checkSetHelix(int n, SecStrucType type) {
        int idx = n - 3;
        logger.debug("Set helix " + (Object)((Object)type) + " " + n + " " + idx);
        for (int i = 1; i < this.groups.length - n; ++i) {
            SecStrucState state = this.getSecStrucState(i);
            SecStrucState previousState = this.getSecStrucState(i - 1);
            if (state.getType().compareTo(type) < 0 || this.getSecStrucState(i + 1).getType().compareTo(type) < 0) continue;
            char turn = state.getTurn()[idx];
            char pturn = previousState.getTurn()[idx];
            if (turn != '>' && turn != 'X' || pturn != '>' && pturn != 'X') continue;
            for (int k = 0; k < n; ++k) {
                this.setSecStrucType(i + k, type);
            }
        }
    }

    private void setSecStrucType(int pos, SecStrucType type) {
        SecStrucState ss = this.getSecStrucState(pos);
        if (type.compareTo(ss.getType()) < 0) {
            ss.setType(type);
        }
    }

    private SecStrucState getSecStrucState(int pos) {
        SecStrucGroup g = this.groups[pos];
        SecStrucState state = (SecStrucState)g.getProperty("secstruc");
        return state;
    }
}

