/*
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package br.com.objectos.code.java.declaration;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.element.AbstractCodeElement;
import br.com.objectos.code.java.expression.LambdaParameter;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.type.ArrayTypeName;
import br.com.objectos.code.java.type.ClassName;
import br.com.objectos.code.java.type.TypeName;
import java.util.List;
import java.util.Objects;
import javax.lang.model.element.Element;
import javax.lang.model.element.ExecutableElement;
import javax.lang.model.element.VariableElement;
import javax.lang.model.type.TypeMirror;
import javax.lang.model.util.SimpleElementVisitor8;

public class ParameterCode extends AbstractCodeElement
    implements
    LambdaParameter {

  private final ParameterTypeName typeName;
  private final String name;

  ParameterCode(ParameterTypeName typeName, String name) {
    this.typeName = typeName;
    this.name = name;
  }

  public static ParameterCode of(VariableElement element) {
    checkNotNull(element, "element == null");
    Element enclosingElement = element.getEnclosingElement();
    return enclosingElement.accept(FromVariableElementFactory.INSTANCE, element);
  }

  public static ParameterCode of(Class<?> type, String name) {
    ClassName typeName = ClassName.of(type);
    checkNotNull(name, "name == null");
    return new ParameterCode(typeName, name);
  }

  public static ParameterCode of(ParameterTypeName typeName, String name) {
    checkNotNull(typeName, "typeName == null");
    checkNotNull(name, "name == null");
    return new ParameterCode(typeName, name);
  }

  static ParameterCode ofUnchecked(ParameterTypeName typeName, String name) {
    return new ParameterCode(typeName, name);
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter writer) {
    return writer.writeCodeElement(typeName).writeWord(name);
  }

  @Override
  public final boolean equals(Object obj) {
    if (obj == this) {
      return true;
    }
    if (!(obj instanceof ParameterCode)) {
      return false;
    }
    ParameterCode that = (ParameterCode) obj;
    return Objects.equals(typeName, that.typeName)
        && Objects.equals(name, that.name);
  }

  @Override
  public final int hashCode() {
    return Objects.hash(typeName, name);
  }

  public final boolean isVarArgs() {
    return typeName.isVarArgs();
  }

  public final String name() {
    return name;
  }

  public final ParameterTypeName typeName() {
    return typeName;
  }

  private static class FromVariableElementFactory
      extends SimpleElementVisitor8<ParameterCode, VariableElement> {

    static FromVariableElementFactory INSTANCE = new FromVariableElementFactory();

    @Override
    public final ParameterCode visitExecutable(ExecutableElement e, VariableElement p) {
      return e.isVarArgs() ? visitExecutableVarArgs(e, p) : visitExecutableStandard(e, p);
    }

    private ParameterCode visitExecutableStandard(ExecutableElement e, VariableElement p) {
      TypeMirror type = p.asType();
      return new ParameterCode(
          TypeName.ofUnchecked(type),
          p.getSimpleName().toString()
      );
    }

    private ParameterCode visitExecutableVarArgs(ExecutableElement e, VariableElement p) {
      List<? extends VariableElement> parameters = e.getParameters();
      VariableElement last = parameters.get(parameters.size() - 1);

      if (!last.equals(p)) {
        return visitExecutableStandard(e, p);
      } else {
        TypeMirror type = p.asType();
        ArrayTypeName typeName = (ArrayTypeName) TypeName.ofUnchecked(type);
        return new ParameterCode(
            VarArgs.of(typeName),
            p.getSimpleName().toString()
        );
      }
    }

    @Override
    protected final ParameterCode defaultAction(Element e, VariableElement p) {
      throw new UnsupportedOperationException(
          "Cannot instantiate ParameterCode from " + e.getKind());
    }

  }

}