/*
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package br.com.objectos.code.java.io;

import br.com.objectos.code.java.declaration.PackageName;
import br.com.objectos.code.java.type.ClassName;
import br.com.objectos.code.java.type.TypeName;
import br.com.objectos.comuns.collections.GrowableSet;
import br.com.objectos.comuns.collections.StreamIterator;

public class JavaFileImportSet extends ImportSet {

  private final GrowableSet<ClassName> qualifiedNameSet = GrowableSet.newSet();
  private final GrowableSet<String> simpleNameSet = GrowableSet.newSet();

  private final PackageName packageName;

  JavaFileImportSet(PackageName packageName) {
    this.packageName = packageName;
  }

  @Override
  public final boolean contains(ClassName qualifiedName) {
    return qualifiedNameSet.contains(qualifiedName);
  }

  @Override
  public final String get(TypeName typeName) {
    return typeName.acceptJavaFileImportSet(this);
  }

  @Override
  public final boolean isEmpty() {
    return size() == 0 && packageName.isUnnamed();
  }

  @Override
  public final int size() {
    return qualifiedNameSet.size();
  }

  @Override
  public final String toString() {
    return packageName.isUnnamed() ? toStringUnnamed() : toStringRegular();
  }

  public final boolean addQualifiedName(ClassName qualifiedName) {
    return qualifiedNameSet.add(qualifiedName);
  }

  public final boolean addSimpleName(String simpleName) {
    return simpleNameSet.add(simpleName);
  }

  public final boolean canSkipImport(PackageName otherPackageName) {
    return otherPackageName.is("java.lang") || packageName.equals(otherPackageName);
  }

  private String toStringRegular() {
    CodeWriter w = CodeWriter.forToString();

    w.writeWord("package");
    w.writeWord(packageName.toString());
    w.write(';');

    if (!qualifiedNameSet.isEmpty()) {
      w.nextLine();
    }

    for (ClassName className : qualifiedNameSet.sorted()) {
      w.nextLine();
      writeImport(w, className);
    }

    return w.toString();
  }

  private String toStringUnnamed() {
    CodeWriter w = CodeWriter.forToString();

    StreamIterator<ClassName> iterator = qualifiedNameSet.sorted().iterator();
    if (iterator.hasNext()) {
      writeImport(w, iterator.next());
      while (iterator.hasNext()) {
        w.nextLine();
        writeImport(w, iterator.next());
      }
    }

    return w.toString();
  }

  private void writeImport(CodeWriter w, ClassName className) {
    w.writeWord("import");
    w.writeWord(className.toString());
    w.write(';');
  }

}