/*
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package br.com.objectos.code.java.statement;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.expression.ExpressionCode;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.statement.Block.Semicolon;

public class IfThenStatement extends AbstractStatement {

  private final ExpressionCode expression;
  private final Statement statement;

  private IfThenStatement(ExpressionCode expression, Statement statement) {
    this.expression = expression;
    this.statement = statement;
  }

  static Builder builder(ExpressionCode condition) {
    checkNotNull(condition, "condition == null");
    return new Builder(condition);
  }

  public final IfThenElseStatement _else(BlockElement... elements) {
    Block block = Block.of(elements);
    return new IfThenElseStatement(this, block);
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    return w
        .writeControl("if", expression)
        .writeCodeElement(statement);
  }

  @Override
  public final void acceptSemicolon(Semicolon semicolon) {
    statement.acceptSemicolon(semicolon);
  }

  public static class Builder {

    private final ExpressionCode condition;

    private Builder(ExpressionCode condition) {
      this.condition = condition;
    }

    public final IfThenStatement then(BlockElement... elements) {
      return block(elements);
    }

    public final IfThenStatement block(BlockElement... elements) {
      return withBlock(Block.of(elements));
    }

    public final IfThenStatement withBlock(Block block) {
      checkNotNull(block, "block == null");
      return new IfThenStatement(condition, block);
    }

  }

}