/*
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package br.com.objectos.code.java.statement;

import br.com.objectos.code.java.expression.Identifier;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.code.java.type.TypeName;
import br.com.objectos.comuns.collections.ImmutableList;
import br.com.objectos.comuns.collections.StreamIterator;

public class TryCatchStatement extends AbstractStatement implements TryStatement {

  private final TryElement tryElement;
  private final ImmutableList<TypeName> exceptionTypes;
  private final Identifier id;
  private final Block block;

  private TryCatchStatement(TryElement tryElement,
                            ImmutableList<TypeName> exceptionTypes,
                            Identifier id,
                            Block block) {
    this.tryElement = tryElement;
    this.exceptionTypes = exceptionTypes;
    this.id = id;
    this.block = block;
  }

  static Builder builder(
      TryElement tryElement, ImmutableList<TypeName> exceptionTypes, Identifier id) {
    return new Builder(tryElement, exceptionTypes, id);
  }

  public final TryCatchFinallyStatement _finally(BlockStatement... statements) {
    Block block = Block.of(statements);
    return new TryCatchFinallyStatement(this, block);
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    w.writeCodeElement(tryElement).spaceOn();
    w.writeWord("catch").writeWord('(').spaceOff();

    StreamIterator<TypeName> exIter = exceptionTypes.iterator();
    if (exIter.hasNext()) {
      w.writeTypeNameAsWord(exIter.next());
      while (exIter.hasNext()) {
        w.writeWord('|');
        w.writeTypeNameAsWord(exIter.next());
      }
    }

    w.writeCodeElement(id).spaceOff().writeWord(')');
    w.writeCodeElement(block);

    return w;
  }

  public static class Builder {

    private final TryElement tryElement;
    private final ImmutableList<TypeName> exceptionTypes;
    private final Identifier id;

    private Builder(TryElement tryElement,
                    ImmutableList<TypeName> exceptionTypes,
                    Identifier id) {
      this.tryElement = tryElement;
      this.exceptionTypes = exceptionTypes;
      this.id = id;
    }

    public final TryCatchStatement block(BlockElement... elements) {
      Block block = Block.of(elements);
      return new TryCatchStatement(tryElement, exceptionTypes, id, block);
    }

  }

}