/*
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package br.com.objectos.code.java.statement;

import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.comuns.collections.ImmutableList;
import java.util.Iterator;

public class TryWithStatement extends TryElement implements TryStatement {

  private final ImmutableList<Resource> resources;
  private final Block block;

  private TryWithStatement(ImmutableList<Resource> resources,
                           Block block) {
    this.resources = resources;
    this.block = block;
  }

  static Builder builder(Resource... resources) {
    ImmutableList<Resource> list = ImmutableList.newListWithAll(resources);
    return new Builder(list);
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    w.writeWord("try").writeWord('(').spaceOff();

    Iterator<Resource> it = resources.iterator();
    if (it.hasNext()) {
      w.writeCodeElement(it.next());
      while (it.hasNext()) {
        w.spaceOff().writeWord(';');
        w.writeCodeElement(it.next());
      }
    }

    return w.spaceOff().writeWord(')').writeCodeElement(block);
  }

  @Override
  public final void acceptStatementOrBlockBuilder(StatementOrBlockBuilder builder) {
    builder.withStatement(this);
  }

  public static class Builder {

    private final ImmutableList<Resource> resources;

    private Builder(ImmutableList<Resource> resources) {
      this.resources = resources;
    }

    public final TryWithStatement block(Statement... statements) {
      Block block = Block.of(statements);
      return new TryWithStatement(resources, block);
    }

  }

}
