/*
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package br.com.objectos.code.java.expression;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.CodeJava;
import br.com.objectos.code.java.element.CodeElement;
import br.com.objectos.code.java.element.NewLine;
import br.com.objectos.code.java.statement.VariableInitializer;
import br.com.objectos.comuns.collections.GrowableList;

public interface ExpressionCode extends CodeElement, LambdaBody, VariableInitializer {

  static Builder builder() {
    return new Builder();
  }

  static Builder builder(ExpressionCode startingExpression) {
    return builder().withExpression(startingExpression);
  }

  static class Builder {

    private final GrowableList<CodeElement> elements = GrowableList.newList();
    private boolean chain;

    private Builder() {}

    public final Expression build() {
      return new Expression(elements.toImmutableList());
    }

    public final Builder and(InclusiveOrExpression rhs) {
      return add(Operator.AND).add(rhs).setChain();
    }

    public final Builder id(String name) {
      return add(CodeJava.id(name)).setChain();
    }

    public final Builder invoke(String methodName, Argument... args) {
      checkNotNull(methodName, "methodName == null");
      checkNotNull(args, "args == null");
      return invoke0(methodName, Arguments.of(args));
    }

    public final Builder invoke(String methodName, Iterable<? extends Argument> args) {
      checkNotNull(methodName, "methodName == null");
      checkNotNull(args, "args == null");
      return invoke0(methodName, Arguments.of(args));
    }

    public final Builder nl() {
      return add(NewLine.nextLine());
    }

    final Builder withExpression(ExpressionCode expression) {
      return add(expression).setChain();
    }

    private Builder add(CodeElement element) {
      elements.add(element);
      return this;
    }

    private Builder invoke0(String methodName, Arguments args) {
      CodeElement invocation
          = chain
              ? ChainedMethodInvocation.ofUnchecked(methodName, args)
              : MethodInvocationImpl.invoke1(methodName, args);
      return add(invocation).setChain();
    }

    private Builder setChain() {
      chain = true;
      return this;
    }

  }

}