/*
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package br.com.objectos.code.java.statement;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.expression.ExpressionCode;
import br.com.objectos.code.java.io.CodeWriter;
import br.com.objectos.comuns.collections.ImmutableList;

public class SwitchStatement extends AbstractStatement {

  private final ExpressionCode expression;
  private final ImmutableList<SwitchElement> body;

  private SwitchStatement(ExpressionCode expression, ImmutableList<SwitchElement> body) {
    this.expression = expression;
    this.body = body;
  }

  public static SwitchStatement.Builder _switch(ExpressionCode expression) {
    return SwitchStatement.builder(expression);
  }

  static Builder builder(ExpressionCode expression) {
    checkNotNull(expression, "expression == null");
    return new Builder(expression);
  }

  @Override
  public final CodeWriter acceptCodeWriter(CodeWriter w) {
    w.writeWord("switch");
    w.writeWord('(').spaceOff();
    w.writeCodeElement(expression);
    w.spaceOff().writeWord(')');
    w.writeBlock(body);
    return w;
  }

  public static class Builder {

    private final ExpressionCode expression;

    private Builder(ExpressionCode expression) {
      this.expression = expression;
    }

    public final SwitchStatement block(SwitchElement... elements) {
      ImmutableList<SwitchElement> body = ImmutableList.newListWithAll(elements);
      return new SwitchStatement(expression, body);
    }

  }

}
