/*
 * Copyright (C) 2014-2019 Objectos Software LTDA.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package br.com.objectos.code.java.type;

import static br.com.objectos.comuns.lang.Preconditions.checkNotNull;

import br.com.objectos.code.java.declaration.PackageName;
import br.com.objectos.code.java.expression.Argument;
import br.com.objectos.code.java.expression.Callee;
import br.com.objectos.code.java.expression.CastExpression;
import br.com.objectos.code.java.expression.ExpressionName;
import br.com.objectos.code.java.expression.Expressions;
import br.com.objectos.code.java.expression.Identifier;
import br.com.objectos.code.java.expression.MethodInvocation;
import br.com.objectos.code.java.expression.MethodReference;
import br.com.objectos.code.java.expression.TypeWitness;
import br.com.objectos.code.java.expression.UnaryExpressionNotPlusMinus;
import br.com.objectos.code.java.io.JavaFileImportSet;
import br.com.objectos.comuns.lang.Preconditions;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import javax.lang.model.SourceVersion;
import javax.lang.model.element.Element;
import javax.lang.model.element.ElementKind;
import javax.lang.model.element.PackageElement;
import javax.lang.model.element.TypeElement;

public class ClassName extends AbstractTypeName
    implements
    ClassNameOrParameterizedTypeName,
    Comparable<ClassName>,
    Callee,
    EnclosingElement {

  private static final ClassName OBJECT = PackageName.named("java.lang").nestedClass("Object");

  private final EnclosingElement enclosingElement;
  private final String simpleName;

  private final String toString;

  ClassName(EnclosingElement enclosingElement, String simpleName) {
    this.enclosingElement = enclosingElement;
    this.simpleName = simpleName;

    toString = enclosingElement.toString(simpleName);
  }

  public static ClassName object() {
    return OBJECT;
  }

  public static ClassName of(Class<?> type) {
    checkNotNull(type, "type == null");
    return ofUnchecked(type);
  }

  public static ClassName of(TypeElement element) {
    checkNotNull(element, "element == null");
    return ofUnchecked(element);
  }

  public static ClassName ofUnchecked(Class<?> type) {
    List<String> nameList = new ArrayList<>();
    nameList.add(type.getSimpleName());

    Class<?> enclosing = type.getEnclosingClass();
    while (enclosing != null) {
      nameList.add(enclosing.getSimpleName().toString());
      enclosing = enclosing.getEnclosingClass();
    }

    ClassName result = null;
    EnclosingElement currentName = PackageName.named(type.getPackage().getName());
    Collections.reverse(nameList);
    for (String name : nameList) {
      result = currentName.nestedClass(name);
      currentName = result;
    }

    return result;
  }

  public static ClassName ofUnchecked(TypeElement element) {
    List<String> nameList = new ArrayList<>();
    nameList.add(element.getSimpleName().toString());

    Element enclosing = element.getEnclosingElement();
    while (!enclosing.getKind().equals(ElementKind.PACKAGE)) {
      nameList.add(enclosing.getSimpleName().toString());
      enclosing = enclosing.getEnclosingElement();
    }

    ClassName result = null;
    EnclosingElement currentName = PackageName.of((PackageElement) enclosing);
    Collections.reverse(nameList);
    for (String name : nameList) {
      result = currentName.nestedClass(name);
      currentName = result;
    }

    return result;
  }

  @Override
  public final String acceptJavaFileImportSet(JavaFileImportSet set) {
    if (set.contains(this)) {
      return simpleName;
    }

    if (set.canSkipImport(packageName())) {
      set.addSimpleName(simpleName);
      return simpleName;
    }

    if (set.addSimpleName(simpleName)) {
      set.addQualifiedName(this);
      return simpleName;
    }

    return toString();
  }

  @Override
  public final <R, P> R acceptTypeNameVisitor(TypeNameVisitor<R, P> visitor, P p) {
    return visitor.visitClassName(this, p);
  }

  @Override
  public final TypeName arrayCreationTypeName() {
    return this;
  }

  @Override
  public final CastExpression cast(UnaryExpressionNotPlusMinus expression) {
    return Expressions.cast(this, expression);
  }

  @Override
  public final int compareTo(ClassName o) {
    return toString.compareTo(o.toString);
  }

  @Override
  public final boolean equals(Object obj) {
    if (!(obj instanceof ClassName)) {
      return false;
    }
    ClassName that = (ClassName) obj;
    return enclosingElement.equals(that.enclosingElement)
        && simpleName.equals(that.simpleName);
  }

  @Override
  public final int hashCode() {
    return Objects.hash(enclosingElement, simpleName);
  }

  public final ExpressionName id(Identifier id) {
    return Expressions.expressionName(this, id);
  }

  public final ExpressionName id(String id) {
    return Expressions.expressionName(this, id);
  }

  @Override
  public final MethodInvocation invoke(String methodName) {
    return Expressions.invoke(this, methodName);
  }

  @Override
  public final MethodInvocation invoke(String methodName, Argument a1) {
    return Expressions.invoke(this, methodName, a1);
  }

  @Override
  public final MethodInvocation invoke(String methodName, Argument a1, Argument a2) {
    return Expressions.invoke(this, methodName, a1, a2);
  }

  @Override
  public final MethodInvocation invoke(String methodName, Argument a1, Argument a2, Argument a3) {
    return Expressions.invoke(this, methodName, a1, a2, a3);
  }

  @Override
  public final MethodInvocation invoke(
      String methodName, Argument a1, Argument a2, Argument a3, Argument a4) {
    return Expressions.invoke(this, methodName, a1, a2, a3, a4);
  }

  @Override
  public final MethodInvocation invoke(String methodName, Iterable<? extends Argument> args) {
    return Expressions.invoke(this, methodName, args);
  }

  @Override
  public final MethodInvocation invoke(
      TypeWitness witness, String methodName) {
    return Expressions.invoke(this, witness, methodName);
  }

  @Override
  public final MethodInvocation invoke(
      TypeWitness witness, String methodName, Argument a1) {
    return Expressions.invoke(this, witness, methodName, a1);
  }

  @Override
  public final MethodInvocation invoke(
      TypeWitness witness, String methodName, Argument a1, Argument a2) {
    return Expressions.invoke(this, witness, methodName, a1, a2);
  }

  @Override
  public final MethodInvocation invoke(
      TypeWitness witness, String methodName, Argument a1, Argument a2, Argument a3) {
    return Expressions.invoke(this, witness, methodName, a1, a2, a3);
  }

  @Override
  public final MethodInvocation invoke(
      TypeWitness witness, String methodName, Argument a1, Argument a2, Argument a3, Argument a4) {
    return Expressions.invoke(this, witness, methodName, a1, a2, a3, a4);
  }

  @Override
  public final MethodInvocation invoke(
      TypeWitness witness, String methodName, Iterable<? extends Argument> args) {
    return Expressions.invoke(this, witness, methodName, args);
  }

  @Override
  public final boolean isJavaLangObject() {
    return toString.equals("java.lang.Object");
  }

  @Override
  public final PackageName packageName() {
    return enclosingElement.packageName();
  }

  @Override
  public final MethodReference ref(String methodName) {
    return Expressions.ref(this, methodName);
  }

  @Override
  public final MethodReference ref(TypeWitness witness, String methodName) {
    return Expressions.ref(this, witness, methodName);
  }

  public final String simpleName() {
    return simpleName;
  }

  @Override
  public final ArrayTypeName toArrayTypeName() {
    return ArrayTypeName.ofUnchecked(this);
  }

  @Override
  public final Optional<ClassName> toClassName() {
    return Optional.of(this);
  }

  @Override
  public final ClassName toClassNameUnchecked() {
    return this;
  }

  @Override
  public final String toString() {
    return toString;
  }

  public final ClassName withPrefix(String prefix) {
    checkNotNull(prefix, "prefix == null");
    String withPrefix = prefix + simpleName;
    checkTypeName(withPrefix);
    return new ClassName(enclosingElement, withPrefix);
  }

  public final ClassName withSuffix(String suffix) {
    checkNotNull(suffix, "suffix == null");
    String withSuffix = simpleName + suffix;
    checkTypeName(withSuffix);
    return new ClassName(enclosingElement, withSuffix);
  }

  @Override
  public final ParameterizedTypeName withTypeArgument(TypeName type) {
    return ParameterizedTypeName.of(this, type);
  }

  private void checkTypeName(String simpleName) {
    Preconditions.checkArgument(simpleName != null && SourceVersion.isName(simpleName),
        "%s is not a valid type name", simpleName);
  }

}