/*
 * @authors   Rohan Sahay, rohansahay@elear.solutions
 *            Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonParseException;
import com.google.gson.JsonPrimitive;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import com.google.gson.annotations.SerializedName;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * Attribute model class is a structure of the attribute used by a capability of a resource.
 */
public class Attribute {

  private static class DataTypeParser implements JsonSerializer<DataType>,
      JsonDeserializer<DataType> {
    @Override
    public DataType deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context)
        throws
        JsonParseException {
      int dataType = json.getAsInt();
      return DataType.getEnum(dataType);
    }

    @Override
    public JsonElement serialize(DataType dataType, Type typeOfSrc,
                                 JsonSerializationContext context) {
      return new JsonPrimitive(dataType.getInt());
    }
  }

  /**
   * An enum denoting the data type of the given attribute's values.
   */
  public enum DataType {
    BOOLEAN,
    BOOLEAN_ARR,
    STRING,
    STRING_ARR,
    UINT8,
    UINT8_ARR,
    UINT16,
    UINT16_ARR,
    UINT32,
    UINT32_ARR,
    UINT64,
    UINT64_ARR,
    INT8,
    INT8_ARR,
    INT16,
    INT16_ARR,
    INT32,
    INT32_ARR,
    INT64,
    INT64_ARR,
    FLOAT,
    FLOAT_ARR,
    DOUBLE,
    DOUBLE_ARR,
    JSON,
    JSON_ARR;

    static DataType getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    int getInt() {
      return ordinal();
    }

    /**
     * This function summarizes the data type of attribute value.
     *
     * @return SimpleDataType: This enum describes attribute data type used to decide typecasting.
     */
    public DataTypeFormatter.SimpleDataType toSimpleDataType() {
      switch (this) {
        case JSON:
        case STRING:
          return DataTypeFormatter.SimpleDataType.STRING;
        case BOOLEAN:
          return DataTypeFormatter.SimpleDataType.BOOLEAN;
        case BOOLEAN_ARR:
          return DataTypeFormatter.SimpleDataType.BOOLEAN_ARR;
        case JSON_ARR:
        case STRING_ARR:
          return DataTypeFormatter.SimpleDataType.STRING_ARR;
        case UINT8:
        case UINT16:
        case UINT32:
        case UINT64:
        case INT8:
        case INT16:
        case INT32:
        case INT64:
        case FLOAT:
        case DOUBLE:
          return DataTypeFormatter.SimpleDataType.NUMBER;
        case UINT8_ARR:
        case UINT16_ARR:
        case UINT32_ARR:
        case UINT64_ARR:
        case INT8_ARR:
        case INT16_ARR:
        case INT32_ARR:
        case INT64_ARR:
        case FLOAT_ARR:
        case DOUBLE_ARR:
          return DataTypeFormatter.SimpleDataType.NUMBER_ARR;
        default:
          throw new IllegalStateException();
      }
    }
  }

  @SerializedName(Constants.ATTRIBUTE_ID)
  private final int id;
  @SerializedName(Constants.ATTRIBUTE_NAME)
  private String name;
  @SerializedName(Constants.ATTRIBUTE_DESCRIPTION)
  private String description;
  @SerializedName(Constants.ATTRIBUTE_DATA_TYPE)
  private DataType dataType;
  @SerializedName(Constants.ATTRIBUTE_ARRAY_LENGTH)
  private int arrayLength;
  @SerializedName(Constants.ATTRIBUTE_MIN_VALUE)
  private Object minValue;
  @SerializedName(Constants.ATTRIBUTE_MAX_VALUE)
  private Object maxValue;
  @SerializedName(Constants.ATTRIBUTE_DEFAULT_VALUE)
  private Object defaultValue;
  @SerializedName(Constants.ATTRIBUTE_VALUE)
  private Object currentValue;
  @SerializedName(Constants.ATTRIBUTE_MIN_REPORTING_INTERVAL)
  private long minReportingInterval;
  @SerializedName(Constants.ATTRIBUTE_MAX_REPORTING_INTERVAL)
  private long maxReportingInterval;

  private transient boolean isRealTimeUpdate = false;

  private transient boolean ready = false;
  private transient Capability parent;

  /**
   * An initializer for the current class and adds serializers needed for this class.
   */
  static void init() {
    Command.GSON_BUILDER.registerTypeAdapter(DataType.class, new DataTypeParser());
  }

  /**
   * The constructor of the current class.
   *
   * @param id     The unique Id of the attribute.
   * @param parent The parent capability of the attribute
   */
  protected Attribute(int id, Capability parent) {
    this.id = id;
    this.parent = parent;
    this.dataType = null;
  }

  /**
   * This function is used to get the integer Ids of the Attribute.
   *
   * @return int: The Integer Id of the Attribute.
   */
  public final int getMapKey() {
    return this.id;
  }

  /**
   * This function is used to get the AttributeId enum of the Attribute.
   *
   * @return Capability.AttributeId: The return value can be directly compared to
   *         attributeIds present in capability's child classes.
   */
  public final Capability.AttributeId getId() {
    return parent.getId().getAttributeId(id);
  }

  /**
   * This function is used to get the name of the attribute.
   *
   * @return String: The name of the attribute.
   */
  public String getName() {
    return this.name;
  }

  /**
   * This function is used to get the description of the attribute.
   *
   * @return String: The description of the attribute is returned.
   */
  public String getDescription() {
    return this.description;
  }

  /**
   * This function can be used to get the dataType of the attribute.
   *
   * @return DataType: The enum denoting the dataType of the attribute. i.e.., The
   *         datatype to of current, min, max, default values.
   */
  public DataType getDataType() {
    return this.dataType;
  }

  /**
   * This function can be used to get the min value that the attribute can take.
   *
   * @return Object: This can be typecast to different values by using the getDataType().
   */
  public Object getMinValue() {
    return this.minValue;
  }

  /**
   * This function can be used to get the max value that the attribute can take.
   *
   * @return Object: This can be typecast to different values by using the getDataType().
   */
  public Object getMaxValue() {
    return this.maxValue;
  }

  /**
   * This function can be used to get the default value that the attribute can take.
   *
   * @return Object: This can be typecast to different values by using the {@link #getDataType()}.
   */
  public Object getDefaultValue() {
    return this.defaultValue;
  }

  /**
   * This function can be used to get the current value that the attribute can take.
   *
   * @return Object: This can be typecast to different values by using the getDataType().
   */
  public Object getCurrentValue() {
    return this.currentValue;
  }

  /**
   * This function can be used to get the parent capability of the attribute.
   *
   * @return Capability: The parent capability of the attribute.
   */
  public Capability getParent() {
    return this.parent;
  }

  /**
   * This function is used to get the maxReporting time of an attribute.
   *
   * @return long: The max duration between two updates.
   */
  public long getMaxReportingInterval() {
    return maxReportingInterval;
  }

  /**
   * This function is used to get the minReporting time of an attribute.
   *
   * @return long: The min duration between two updates
   */
  public long getMinReportingInterval() {
    return minReportingInterval;
  }

  /**
   * This function can be used to get the length of the array IF AND ONLY IF THE CURRENT
   * {@link #dataType} is an array.
   *
   * @return The length of the array, if dataType is an array
   */
  public int getArrayLength() {
    return this.arrayLength;
  }

  /**
   * A function to help with getting the items of the current value NOTE: Works only when
   * {@link #dataType} is an array.
   *
   * @param position The position of the item.
   * @return The object at the index position
   */
  public <T> T getElement(int position) {
    Object value = null;
    Object currentValue = getCurrentValue();

    if (currentValue instanceof boolean[]) {
      value = ((boolean[]) currentValue)[position];
    }

    if (currentValue instanceof int[]) {
      value = ((int[]) currentValue)[position];
    }

    if (currentValue instanceof long[]) {
      value = ((long[]) currentValue)[position];
    }

    if (currentValue instanceof float[]) {
      value = ((float[]) currentValue)[position];
    }

    if (currentValue instanceof double[]) {
      value = ((double[]) currentValue)[position];
    }

    if (currentValue instanceof String[]) {
      value = ((String[]) currentValue)[position];
    }

    return Utils.castUp(value);
  }

  public final boolean isRealTimeUpdate() {
    return isRealTimeUpdate;
  }

  protected final void internalSetRealtimeUpdate(boolean isRealTimeUpdate) {
    this.isRealTimeUpdate = isRealTimeUpdate;
  }

  protected void internalSetName(String name) {
    this.name = name;
  }

  protected void internalSetDescription(String description) {
    this.description = description;
  }

  protected void internalSetDataType(int dataType) {
    this.dataType = DataType.getEnum(dataType);
  }

  protected void internalSetArrayLength(int arrayLength) {
    this.arrayLength = arrayLength;
  }

  protected void internalSetMinValue(Object minValue) {
    this.minValue = minValue;
  }

  protected void internalSetMaxValue(Object maxValue) {
    this.maxValue = maxValue;
  }

  protected void internalSetCurrentValue(Object currentValue) {
    this.currentValue = currentValue;
  }

  protected void internalSetDefaultValue(Object defaultValue) {
    this.defaultValue = defaultValue;
  }

  protected void internalSetMinReportingInterval(long minReportingInterval) {
    this.minReportingInterval = minReportingInterval;
  }

  protected void internalSetMaxReportingInterval(long maxReportingInterval) {
    this.maxReportingInterval = maxReportingInterval;
  }

  protected void internalMarkAsReady() {
    this.ready = true;
  }

  public String getAsString() {
    Object currentValue = getCurrentValue();
    return currentValue instanceof String ? (String) currentValue : null;
  }

  public boolean isStringValue() {
    return (getCurrentValue() instanceof String);
  }

  public Number getAsNumber() {
    Object currentValue = getCurrentValue();
    return currentValue instanceof Number ? (Number) currentValue : null;
  }

  public boolean isNumberValue() {
    return (getCurrentValue() instanceof Number);
  }

  public boolean isBooleanValue() {
    return (getCurrentValue() instanceof Boolean);
  }

  public Boolean getAsBoolean() {
    Object currentValue = getCurrentValue();
    return currentValue instanceof Boolean ? (Boolean) currentValue : null;
  }

  /**
   * A function with a job to narrow the {@link #currentValue}
   * to Number[] explicitly.
   * NOTE: works only when {@link #dataType} is int[], long[], float[] or double[]
   *
   * @return Number[]: Attribute value as array of Number objects.
   */
  public Number[] getAsNumberArr() {
    List<Number> retVal;
    Object currentValue = getCurrentValue();

    if (null == currentValue) {
      return null;
    }

    retVal = new ArrayList<>(arrayLength);

    if (currentValue instanceof int[]) {
      int[] castVal = (int[]) currentValue;
      for (int val : castVal) {
        retVal.add(val);
      }
    }

    if (currentValue instanceof long[]) {
      long[] castVal = (long[]) currentValue;
      for (long val : castVal) {
        retVal.add(val);
      }
    }

    if (currentValue instanceof float[]) {
      float[] castVal = (float[]) currentValue;
      for (double val : castVal) {
        retVal.add(val);
      }
    }

    if (currentValue instanceof double[]) {
      double[] castVal = (double[]) currentValue;
      for (double val : castVal) {
        retVal.add(val);
      }
    }

    return retVal.toArray(new Number[0]);
  }

  /**
   * A function with a job to narrow the {@link #currentValue}
   * to boolean[] explicitly.
   *
   * @return boolean[]: Attribute value as an array of boolean.
   */
  public boolean[] getAsBooleanArr() {
    Object currentValue = getCurrentValue();

    boolean[] castVal = (currentValue instanceof boolean[]) ? (boolean[]) currentValue : null;

    return (null == castVal) ? null : Arrays.copyOf(castVal, castVal.length);
  }

  /**
   * A function with a job to narrow the {@link #currentValue}
   * to String[] explicitly.
   *
   * @return String[]: Attribute value as an array of string.
   */
  public String[] getAsStringArr() {
    Object currentValue = getCurrentValue();

    String[] castVal = (currentValue instanceof String[]) ? (String[]) currentValue : null;

    return (null == castVal) ? null : Arrays.copyOf(castVal, castVal.length);
  }

  protected void internalSetParent(Capability parent) {
    this.parent = parent;
  }

  @Override
  public String toString() {
    return "Attribute{"
        + "id=" + ((null != getParent()) ? getId() : getMapKey())
        + ", name='" + name + '\''
        + ", description='" + description + '\''
        + ", dataType=" + dataType
        + ", arrayLength=" + arrayLength
        + ", minValue=" + minValue
        + ", maxValue=" + maxValue
        + ", defaultValue=" + defaultValue
        + ", currentValue=" + currentValue
        + ", minReportingInterval=" + minReportingInterval
        + ", maxReportingInterval=" + maxReportingInterval
        + ", ready=" + ready
        + '}';
  }

  /**
   * A function to state if the current attribute is made from callbacks or state retrieval.
   *
   * @return boolean: states if the current attribute is ready or not.
   */
  public boolean isReady() {
    return parent.isReady() && ready;
  }

  @Override
  public final int hashCode() {
    return getMapKey();
  }
}
