/*
 * @authors   Rohan Sahay, rohansahay@elear.solutions
 *            Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import java.util.ArrayList;
import java.util.List;

class CallbackMultiplexer implements CallbacksInterface {
  private static final String TAG = "CallbackMultiplexer";

  private static final CallbackMultiplexer instance = new CallbackMultiplexer();

  // untested internal count calculations
  private int internalCallbackCount = 0;
  private final List<CallbacksInterface> subscribers = new ArrayList<>(2);

  private CallbackMultiplexer() {
  }

  static CallbackMultiplexer getInstance() {
    return instance;
  }

  void internalAddSubscription(CallbacksInterface subscriber) {
    if (null == subscriber) {
      return;
    }

    internalRemoveSubscription(subscriber);

    synchronized (subscribers) {
      subscribers.add(internalCallbackCount++, subscriber);
    }
  }

  void internalRemoveSubscription(CallbacksInterface subscriber) {
    if (null == subscriber) {
      return;
    }

    synchronized (subscribers) {
      if (subscribers.remove(subscriber)) {
        internalCallbackCount--;
      }
    }
  }

  void addSubscription(CallbacksInterface subscriber) {
    if (null == subscriber) {
      return;
    }

    removeSubscription(subscriber);

    synchronized (subscribers) {
      subscribers.add(subscriber);
    }
  }

  void removeSubscription(CallbacksInterface subscriber) {
    if (null == subscriber) {
      return;
    }

    synchronized (subscribers) {
      int size = subscribers.size();
      int index = subscribers.indexOf(subscriber);

      if (0 > index || size <= index) {
        return;
      }

      // if internal callback then ignore
      if (index < internalCallbackCount) {
        Log.w(TAG, "remove called on internal subscription, ignoring");
        return;
      }

      subscribers.remove(subscriber);
    }
  }

  void removeAllExternalSubscribers() {

    synchronized (subscribers) {
      subscribers.subList(internalCallbackCount, subscribers.size()).clear();
    }
  }

  @Override
  public void connectStatusCallback(Network network) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.connectStatusCallback(network);
    }
  }

  @Override
  public void deviceInfoCallback(Device device) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.deviceInfoCallback(device);
    }
  }

  @Override
  public void resourceCallback(Resource resource) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.resourceCallback(resource);
    }
  }

  @Override
  public void resourceCapabilityCallback(Capability capability) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.resourceCapabilityCallback(capability);
    }
  }

  @Override
  public void resourceAttributeCallback(Attribute attribute) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.resourceAttributeCallback(attribute);
    }
  }

  @Override
  public void commandStatusCallback(CommandResponse<?> command) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.commandStatusCallback(command);
    }
  }

  @Override
  public void networkListCallback(ArrayList<Network> networksList) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.networkListCallback(networksList);
    }
  }

  @Override
  public void deviceManagementStatusCallback(Device device, CommandResponse<?> executedCommand,
                                             Resource[] resourcesImpacted) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.deviceManagementStatusCallback(device, executedCommand, resourcesImpacted);
    }
  }

  @Override
  public void infoRequestCallback(InfoRequest infoRequest) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.infoRequestCallback(infoRequest);
    }
  }

  @Override
  public void infoResponseStatusCallback(Command.State state, InfoResponse infoResponse) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.infoResponseStatusCallback(state, infoResponse);
    }
  }

  @Override
  public void messageCallback(String title, String message, MessageType messageType) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.messageCallback(title, message, messageType);
    }
  }

  @Override
  public void advertiseResourceCallback(Resource resource) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.advertiseResourceCallback(resource);
    }
  }

  @Override
  public void infoRequestStatusCallback(Command.State state, InfoRequest infoRequest) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.infoRequestStatusCallback(state, infoRequest);
    }
  }

  @Override
  public void infoResponseCallback(InfoRequest request, InfoResponse infoResponse) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.infoResponseCallback(request, infoResponse);
    }
  }

  @Override
  public void resourceIncludedCallback(ArrayList<Resource> resources) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.resourceIncludedCallback(resources);
    }
  }

  @Override
  public void resourceExcludedCallback(Resource resource) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.resourceExcludedCallback(resource);
    }
  }

  @Override
  public void tunnelStatusCallback(Capability tunnel, long tunnelHandle,
                                   CapabilityTunnel.State state, int port) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.tunnelStatusCallback(tunnel, tunnelHandle, state, port);
    }
  }

  @Override
  public void zoneInfoCallback(Zone zone) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.zoneInfoCallback(zone);
    }
  }

  @Override
  public void zoneDeletedCallback(Network network, int zoneId) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.zoneDeletedCallback(network, zoneId);
    }
  }

  @Override
  public void sceneInfoCallback(Scene scene) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.sceneInfoCallback(scene);
    }
  }

  @Override
  public void sceneDeletedCallback(Network network, int sceneId) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.sceneDeletedCallback(network, sceneId);
    }
  }

  @Override
  public void ruleInfoCallback(Rule rule) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.ruleInfoCallback(rule);
    }
  }

  @Override
  public void ruleDeletedCallback(Network network, int ruleId) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.ruleDeletedCallback(network, ruleId);
    }
  }

  @Override
  public void nodeConnectionStatusCallback(Network network, long nodeId, NodeType nodeType,
                                           boolean isOnline) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.nodeConnectionStatusCallback(network, nodeId, nodeType, isOnline);
    }
  }

  @Override
  public void networkManagementCommandCallback(Network network, CommandResponse<?> response) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.networkManagementCommandCallback(network, response);
    }
  }

  @Override
  public void networkDataCallback(Network network) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.networkDataCallback(network);
    }
  }

  @Override
  public void leaveNetworkStatusCallback(Network network, Command.State status) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.leaveNetworkStatusCallback(network, status);
    }
  }

  @Override
  public void errorCallback(Throwable tr) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.errorCallback(tr);
    }
  }

  @Override
  public void receiveDataCallback(Network network, long sourceNodeId, String data) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.receiveDataCallback(network, sourceNodeId, data);
    }
  }

  @Override
  public void contentInfoCallback(Network network, long sourceNodeId, long contentTime,
                                  String data) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.contentInfoCallback(network, sourceNodeId, contentTime, data);
    }
  }

  @Override
  public void networkMetadataCallback(Network network) {
    for (CallbacksInterface subscriber : subscribers) {
      subscriber.networkMetadataCallback(network);
    }
  }
}
