/*
 * @authors   Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonPrimitive;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import com.google.gson.annotations.SerializedName;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Capability class represents the Capability of a resource
 * This file contains all the attributes, commands supported and capability names.
 */
public abstract class Capability implements Iterable<Attribute> {

  /**
   * Super interface for all CommandIds in all child Capabilities.
   */
  public interface CommandId extends CommandIdInterface {
  }

  /**
   * Super interface for all AttributeIds used by capability of a resource.
   * Interface also assists AttributeId enums with their capabilityIds.
   */
  public interface AttributeId extends CommandIdInterface {
    CapabilityId getCapabilityId();
  }

  /**
   * Enum mapping all coco-standard capabilityIds and their corresponding capability IDProviders.
   *
   * @see IdProvider
   */
  public enum CapabilityId {
    ON_OFF_CONTROL(IdProvider.CAP_ON_OFF_PROVIDER),
    LEVEL_CTRL(IdProvider.CAP_LEVEL_PROVIDER),
    COLOR_CTRL(IdProvider.CAP_COLOR_PROVIDER),
    LOCK_CONTROL(IdProvider.CAP_LOCK_PROVIDER),
    ENERGY_METERING(IdProvider.CAP_ENERGY_PROVIDER),
    MOTION_SENSING(IdProvider.CAP_MOTION_PROVIDER),
    OCCUPANCY_SENSING(IdProvider.CAP_OCCUPANCY_PROVIDER),
    CONTACT_SENSING(IdProvider.CAP_CONTACT_PROVIDER),
    FLUID_LEVEL_MEASUREMENT(IdProvider.CAP_FLUID_PROVIDER),
    FIRE_SENSING(IdProvider.CAP_FIRE_PROVIDER),
    TEMPERATURE_MEASUREMENT(IdProvider.CAP_TEMP_PROVIDER),
    ILLUMINANCE_MEASUREMENT(IdProvider.CAP_LUMINANCE_PROVIDER),
    POWER_LEVEL_MEASUREMENT(IdProvider.CAP_POWER_PROVIDER),
    TUNNEL_CONTROL(IdProvider.CAP_TUNNEL_PROVIDER),
    REL_HUMIDITY_MEASUREMENT(IdProvider.CAP_HUMIDITY_PROVIDER),
    KEY_PRESS_SENSING(IdProvider.CAP_KEY_PRESS_PROVIDER),
    WARNING_DEV_CONTROL(IdProvider.CAP_WARN_PROVIDER),
    NETWORK_CONFIGURATION(IdProvider.CAP_NETWORK_PROVIDER),
    MEDIA_STREAM(IdProvider.CAP_MEDIA_PROVIDER),
    STORAGE_CONFIG(IdProvider.CAP_STORAGE_CONFIG_PROVIDER),
    STORAGE_CONTROL(IdProvider.CAP_STORAGE_CONTROL_PROVIDER),
    MOTOR_CONTROL(IdProvider.CAP_MOTOR_PROVIDER),
    IMAGE_CONTROL(IdProvider.CAP_IMAGE_PROVIDER),
    SNAPSHOT(IdProvider.CAP_SNAP_PROVIDER),
    STATIONARY_POS(IdProvider.CAP_STATIONARY_POS_PROVIDER),
    REAL_TIME_POS(IdProvider.CAP_REAL_TIME_POS_PROVIDER),
    VIBRATION_SENSING(IdProvider.CAP_VIBRATION_PROVIDER),
    AIR_QUALITY_SENSING(IdProvider.CAP_AIR_Q_PROVIDER),
    WINDOW_COVERING(IdProvider.CAP_WINDOW_PROVIDER),
    REMOTE_CONTROL(IdProvider.CAP_REMOTE_PROVIDER),
    HVAC_CONTROL(IdProvider.CAP_HVAC_PROVIDER);   // Currently, not in use. ONLY for REMOTE_CONTROL

    private final IdProvider provider;

    CapabilityId(IdProvider provider) {
      this.provider = provider;
    }

    public static CapabilityId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }

    public AttributeId getAttributeId(int attrId) {
      return provider.getAttributeId(attrId);
    }

    public CommandId getCommandId(int cmdId) {
      return provider.getCommandId(cmdId);
    }
  }

  @SerializedName(Constants.CAPABILITY_ID)
  private final CapabilityId id;
  @SerializedName(Constants.CAPABILITY_NAME)
  private String name;
  @SerializedName(Constants.CAPABILITY_COMMAND_ARRAY)
  private final Set<Integer> standardCommandSet;

  private transient Resource parent;
  private transient boolean ready = false;

  @SerializedName(Constants.CAPABILITY_ATTRIBUTE_MAP)
  private Map<Integer, Attribute> attributeMap;

  private transient CapabilityWrapper capabilityWrapper = null;

  /**
   * Constructor of the current class.
   *
   * @param id     The unique id Of the capability
   * @param parent The parent Resource of the capability.
   */
  protected Capability(int id, Resource parent) {
    this.standardCommandSet = new HashSet<>();
    this.id = CapabilityId.getEnum(id);
    this.parent = parent;
  }

  static void init() {
    Command.GSON_BUILDER.registerTypeAdapter(CapabilityId.class, new CapabilityIdParser());
  }

  protected <T extends CommandId> void interceptCommand(Command<T> command) {
  }

  /**
   * sendCommands to the current capability.
   *
   * @param command  The command which is to be executed.
   * @param listener Callback of this interface will be triggered.
   * @see CapabilityTunnel implements a different approach
   */
  public final <T extends CommandId> void sendResourceCommand(Command<T> command,
                                                              CommandStatusListener<T> listener) {
    interceptCommand(command);
    CocoClient.getInstance().getNativeHandler().sendResourceCommand(this, command, listener);
  }

  /**
   * This function checks if a command is supported or not, pass
   * null to check if capability supports any commands at all.
   * NOTE: Check the overridden function also for a complete view.
   *
   * @param commandId The Id denoting the command to be sent.
   * @return boolean: If the command is supported then True is returned.
   */
  public boolean supports(CommandId commandId) {
    if (!ready) {
      return false;
    }

    if (null == commandId) {
      return (standardCommandSet.size() > 0);
    }

    return standardCommandSet.contains(commandId.getInt());
  }

  /**
   * This function is used to get the unique id of the capability.
   *
   * @return CapabilityId: The enum denoting what is the current capability. NOTE:
   *         This id may be the same across different resources.
   */
  public final CapabilityId getId() {
    return this.id;
  }

  /**
   * This function is used to get the Name of the capability.
   *
   * @return String: The name of the current capability.
   */
  public final String getName() {
    return this.name;
  }

  /**
   * This function is used to get the parent resource of the current capability.
   *
   * @return Resource: The parent Resource of the capability
   */
  public final Resource getParent() {
    return this.parent;
  }

  /**
   * This function is used to return the HashMap that the sdk uses to map
   * the integer id of the attribute to the Attribute Objects.
   * This function can be used to databind the sdk to the application.
   *
   * @return The key being the integer id of attribute
   *         and the value being the Attribute Object. NOTE: This hashmap is for
   *         read-only any misuse may cause the program to misbehave and even crash.
   */
  public Map<Integer, Attribute> getAttributeMap() {
    if (null == this.attributeMap) {
      this.attributeMap = new ConcurrentHashMap<>();
    }

    return this.attributeMap;
  }

  /**
   * This function is used to get the Attribute with the given Id.
   *
   * @return Attribute of the given AttributeId
   */
  public <T extends Attribute> T getAttribute(int attributeId) {
    return Utils.castUp(getAttributeMap().get(attributeId));
  }

  /**
   * This function is used to get the Attribute with the given Id.
   *
   * @return Attribute of the given AttributeId
   */
  public <T extends Attribute> T getAttribute(AttributeId attributeId) {
    // checking if attribute Id is same the intended attributeId
    if (getAttributeId(attributeId.getInt()) != attributeId) {
      throw new IllegalArgumentException("invalid id: " + attributeId);
    }

    return getAttribute(attributeId.getInt());
  }

  protected boolean containsAttribute(int attributeId) {
    return getAttributeMap().containsKey(attributeId);
  }

  public boolean containsAttribute(AttributeId attributeId) {
    return containsAttribute(attributeId.getInt());
  }

  protected void internalAddAttribute(Attribute attribute) {
    CapabilityWrapper wrapper;

    getAttributeMap().put(attribute.getMapKey(), attribute);

    if (null != (wrapper = getCapabilityWrapper())) {
      wrapper.internalAddAttribute(attribute);
    }
  }

  protected void internalRemoveAttribute(int attributeId) {
    CapabilityWrapper wrapper;

    getAttributeMap().remove(attributeId);

    if (null != (wrapper = getCapabilityWrapper())) {
      wrapper.internalRemoveAttribute(attributeId);
    }
  }

  protected void internalSetParent(Resource parent) {
    CapabilityWrapper wrapper;

    this.parent = parent;

    if (null != (wrapper = getCapabilityWrapper())) {
      wrapper.internalSetParent(parent);
    }
  }

  /**
   * This function sets provided {@link CapabilityWrapper} object.
   *
   * @param capabilityWrapper instance of wrapper object.
   */
  public void setCapabilityWrapper(CapabilityWrapper capabilityWrapper) {
    if (null == capabilityWrapper) {
      throw new NullPointerException();
    }

    this.capabilityWrapper = capabilityWrapper;
  }

  public CapabilityWrapper getCapabilityWrapper() {
    return capabilityWrapper;
  }

  @Override
  public final int hashCode() {
    return getId().ordinal();
  }

  /**
   * A function to get the attributeId corresponding to the integer value of the attributeId.
   *
   * @param id The Integer value of the corresponding enum
   * @return The AttributeId enum.
   */
  private AttributeId getAttributeId(int id) {
    return getId().getAttributeId(id);
  }

  /**
   * Function fetches the corresponding enumId of the provided integer commandId.
   *
   * @param id the raw integer corresponding to enum value
   * @return The enum representing the CommandID(int)
   */
  private CommandId getCommandId(int id) {
    return getId().getCommandId(id);
  }

  /**
   * This function fetches the list of commands supported.
   *
   * @return commands supported
   */
  public List<CommandId> getSupportedCommands() {
    ArrayList<CommandId> commands = new ArrayList<>();

    for (int rawCommand : standardCommandSet) {
      commands.add(getCommandId(rawCommand));
    }

    return commands;
  }

  /**
   * A function to create the command from the Json params.
   * NOTE: This must be only used by classes extending this class.
   *
   * @param commandId     The int form of the commandId
   * @param commandParams The Json params that can form a command
   * @return The command which is formed
   */
  protected abstract Command<? extends Capability.CommandId>
      extendedCreateCommand(int commandId, JsonElement commandParams);

  /**
   * create response args for the command.
   *
   * @param commandId    The command ID which has to be responded
   * @param jsonResponse The response body
   * @return The created args
   * @see CommandResponse.Args
   */
  protected <T extends CommandIdInterface> CommandResponse.Args<T> createCommandResponseArgs(
      T commandId, JsonElement jsonResponse) {
    return CommandResponse.createEmptyResponseArgs();
  }

  /**
   * A function to create the command from the Json params.
   *
   * @param commandId     The int form of the commandId
   * @param commandParams The Json params that can form a command
   * @return The command which is formed
   */
  protected Command<? extends Capability.CommandId> createCommand(int commandId,
                                                                  JsonElement commandParams) {
    if (null == commandParams) {
      commandParams = new JsonObject();
    }

    Command<? extends Capability.CommandId> generatedCommand =
        extendedCreateCommand(commandId, commandParams);

    if (null != generatedCommand) {
      generatedCommand.setCommandId(getCommandId(commandId));
    }

    return generatedCommand;
  }

  /**
   * A function to state if the current capability is updated from
   * {@link NativeCallbacks#resourceCapabilityCallback} or by any other way.
   *
   * @return boolean: states if the current attribute is ready or not.
   */
  public boolean isReady() {
    return parent.isReady() && ready;
  }

  protected void internalSetName(String name) {
    CapabilityWrapper wrapper;

    this.name = name;

    if (null != (wrapper = getCapabilityWrapper())) {
      wrapper.internalSetName(name);
    }
  }

  protected void clearAndSetStandardCommandSet(int[] commands) {
    if (null == commands) {
      return;
    }

    CapabilityWrapper wrapper;

    standardCommandSet.clear();

    for (int command : commands) {
      standardCommandSet.add(command);
    }

    if (null != (wrapper = getCapabilityWrapper())) {
      wrapper.clearAndSetStandardCommandSet(commands);
    }
  }

  protected void internalMarkAsReady() {
    CapabilityWrapper wrapper;

    this.ready = true;

    if (null != (wrapper = getCapabilityWrapper())) {
      wrapper.markAsReady();
    }
  }

  @Override
  public String toString() {
    return "Capability{"
        + "id=" + id
        + ", name='" + name + '\''
        + ", standardCommandSet=" + standardCommandSet
        + ", ready=" + ready
        + '}';
  }

  @Override
  public Iterator<Attribute> iterator() {
    return getAttributeMap().values().iterator();
  }

  /**
   * Listener to assist with status of the
   * {@link #sendResourceCommand(Command, CommandStatusListener)}.
   *
   * @param <T> subtypes of {@link CommandId}
   */
  public interface CommandStatusListener<T extends CommandId> extends Listener {
    void onCommandStatus(CommandResponse<T> response, Throwable tr);
  }

  private static final class CapabilityIdParser implements JsonSerializer<CapabilityId>,
      JsonDeserializer<CapabilityId> {

    @Override
    public CapabilityId deserialize(JsonElement jsonElement, Type type,
                                    JsonDeserializationContext jsonDeserializationContext) throws
        JsonParseException {
      return CapabilityId.getEnum(jsonElement.getAsInt());
    }

    @Override
    public JsonElement serialize(CapabilityId capabilityId, Type type,
                                 JsonSerializationContext jsonSerializationContext) {
      return new JsonPrimitive(capabilityId.getInt());
    }
  }
}
