/*
 * @authors   Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.annotations.SerializedName;

/**
 * This class describes Commands and Attributes for Color control Capability of a Resource.
 */
public class CapabilityColor extends Capability {

  public static final CapabilityId ID = CapabilityId.COLOR_CTRL;

  /**
   * enum denoting various possible attributes of the {@link CapabilityColor}.
   */
  public enum AttributeId implements Capability.AttributeId {
    @Deprecated
    RED,
    @Deprecated
    GREEN,
    @Deprecated
    BLUE,
    @Deprecated
    HUE_DEGREES,
    @Deprecated
    SATURATION_PCT,
    @Deprecated
    LUMINANCE_PCT,

    COLOR_TEMP_KELVIN,
    COMP_ID_ARR,
    COMP_VALUES_ARR,

    @Deprecated
    COLOR_X,
    @Deprecated
    COLOR_Y,
    @Deprecated
    COLOR_MODE,

    HSL,
    RGB;

    public static AttributeId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }

    @Override
    public CapabilityId getCapabilityId() {
      return ID;
    }
  }

  /**
   * enum denoting the possible Commands that can be sent for {@link CapabilityColor}.
   */
  public enum CommandId implements Capability.CommandId {
    SET_COLOR_RGB,
    SET_COLOR_HSL,
    SET_COLOR_TEMP,
    SET_COLOR_COMPONENTS,
    GET_COLOR_RGB,
    GET_COLOR_HSL,
    GET_COLOR_TEMP,
    GET_COLOR_COMPONENTS;

    public static CommandId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }
  }

  /**
   * Constructor of the current class.
   *
   * @param id     The unique id Of the capability
   * @param parent The parent Resource of the capability.
   */
  protected CapabilityColor(int id, Resource parent) {
    super(id, parent);
  }

  /**
   * This function is used to check if a command is supported by the capability or not.
   *
   * @param commandId The Id denoting the command to be sent.
   * @return boolean: If the command is supported then True is returned.
   */
  @Override
  public boolean supports(Capability.CommandId commandId) {
    return (null == commandId || commandId instanceof CommandId) && super.supports(commandId);
  }

  /**
   * A function to create the command from the Json params.
   *
   * @param primitiveCommandId The int form of the commandId
   * @param commandParams      The Json params that can form a command
   * @return The command which is formed
   */
  @Override
  protected Command<? extends Capability.CommandId>
      extendedCreateCommand(int primitiveCommandId, JsonElement commandParams) {
    Command<CommandId> command;
    Gson gson = Command.GSON_BUILDER.create();
    CommandId commandId = CommandId.getEnum(primitiveCommandId);

    switch (commandId) {
      case SET_COLOR_RGB: {
        command = gson.fromJson(commandParams, SetRgb.class);
        break;
      }

      case SET_COLOR_HSL: {
        command = gson.fromJson(commandParams, SetHsl.class);
        break;
      }

      case SET_COLOR_TEMP: {
        command = gson.fromJson(commandParams, SetTemperature.class);
        break;
      }

      case SET_COLOR_COMPONENTS: {
        command = gson.fromJson(commandParams, SetColorComponents.class);
        break;
      }

      case GET_COLOR_RGB: {
        command = gson.fromJson(commandParams, GetRgb.class);
        break;
      }

      case GET_COLOR_HSL: {
        command = gson.fromJson(commandParams, GetHsl.class);
        break;
      }

      case GET_COLOR_TEMP: {
        command = gson.fromJson(commandParams, GetColorTemperature.class);
        break;
      }

      case GET_COLOR_COMPONENTS: {
        command = gson.fromJson(commandParams, GetColorComponents.class);
        break;
      }

      default: {
        command = new Command<>(commandId);
      }
    }

    return command;
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending RGB commands.
   */
  public static class SetRgb extends Command<CommandId> {
    @SerializedName(Constants.RED)
    public int red;

    @SerializedName(Constants.GREEN)
    public int green;

    @SerializedName(Constants.BLUE)
    public int blue;

    @SerializedName(Constants.USE_DEFAULT_TRANSITION)
    public boolean useDefaultTransitionTime;

    @SerializedName(Constants.TRANSITION_TIME)
    public int transitionTime;

    /**
     * Constructor of this class.
     *
     * @param useDefaultTransitionTime flag indicating if the transition has to
     *                                 take place. NOTE: The remaining arguments
     *                                 can be set after the object creation
     */
    public SetRgb(boolean useDefaultTransitionTime) {
      super(CommandId.SET_COLOR_RGB);
      this.useDefaultTransitionTime = useDefaultTransitionTime;
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending HSL commands.
   */
  public static class SetHsl extends Command<CommandId> {
    @SerializedName(Constants.HUE)
    public int hue;

    @SerializedName(Constants.SATURATION)
    public int saturation;

    @SerializedName(Constants.LUMINANCE)
    public int luminance;

    @SerializedName(Constants.USE_DEFAULT_TRANSITION)
    public boolean useDefaultTransitionTime;

    @SerializedName(Constants.TRANSITION_TIME)
    public int transitionTime;

    /**
     * Constructor of this class.
     *
     * @param useDefaultTransitionTime flag indicating if the transition has to
     *                                 take place. NOTE: The remaining arguments
     *                                 can be set after the object creation
     */
    public SetHsl(boolean useDefaultTransitionTime) {
      super(CommandId.SET_COLOR_HSL);
      this.useDefaultTransitionTime = useDefaultTransitionTime;
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand(Command, CommandStatusListener)}
   * while sending Color Temperature command.
   */
  public static class SetTemperature extends Command<CommandId> {
    @SerializedName(Constants.TEMPERATURE_KELVIN)
    public int temperatureKelvin;

    @SerializedName(Constants.USE_DEFAULT_TRANSITION)
    public boolean useDefaultTransitionTime;

    @SerializedName(Constants.TRANSITION_TIME)
    public int transitionTime;

    /**
     * Constructor of this class.
     *
     * @param useDefaultTransitionTime flag indicating if the transition has to
     *                                 take place. NOTE: The remaining arguments
     *                                 can be set after the object creation
     */
    public SetTemperature(boolean useDefaultTransitionTime) {
      super(CommandId.SET_COLOR_TEMP);
      this.useDefaultTransitionTime = useDefaultTransitionTime;
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand}
   * while sending SetComponents commands.
   */
  public static class SetColorComponents extends Command<CommandId> {
    @SerializedName(Constants.COLOR_COMPONENT_IDS)
    public int[] colorComponentIds;

    @SerializedName(Constants.COLOR_COMPONENT_VALUES)
    public int[] colorComponentsValues;

    @SerializedName(Constants.USE_DEFAULT_TRANSITION)
    public boolean useDefaultTransitionTime;

    @SerializedName(Constants.TRANSITION_TIME)
    public int transitionTime;

    /**
     * Constructor for this class.
     *
     * @param useDefaultTransitionTime flag indicating if the transition has to
     *                                 take place. NOTE: The remaining arguments
     *                                 can be set after the object creation
     */
    public SetColorComponents(boolean useDefaultTransitionTime) {
      super(CommandId.SET_COLOR_COMPONENTS);
      this.useDefaultTransitionTime = useDefaultTransitionTime;
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending Get RGB values commands.
   */
  public static class GetRgb extends Command<CommandId> {
    public GetRgb() {
      super(CommandId.GET_COLOR_RGB);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand(Command, CommandStatusListener)} while sending Get HSL values
   * commands.
   */
  public static class GetHsl extends Command<CommandId> {
    public GetHsl() {
      super(CommandId.GET_COLOR_HSL);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand(Command, CommandStatusListener)}
   * while sending getColorTemperature commands.
   */
  public static class GetColorTemperature extends Command<CommandId> {
    public GetColorTemperature() {
      super(CommandId.GET_COLOR_TEMP);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand(Command, CommandStatusListener)}
   * while sending GetColorComponents commands.
   */
  public static class GetColorComponents extends Command<CommandId> {
    public GetColorComponents() {
      super(CommandId.GET_COLOR_COMPONENTS);
    }
  }
}
