/*
 * @authors   Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.Gson;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonParseException;
import com.google.gson.JsonPrimitive;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import com.google.gson.annotations.SerializedName;
import java.lang.reflect.Type;

/**
 * This class describes Commands and Attributes for Door Lock Capability of a Resource.
 */
public class CapabilityDoorLock extends Capability {

  public static final CapabilityId ID = CapabilityId.LOCK_CONTROL;

  /**
   * A class to serialize and deserialize UserType class as c-layer uses ints for enum.
   */
  private static class UserTypeSerializer implements JsonSerializer<UserType>,
      JsonDeserializer<UserType> {
    @Override
    public UserType deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context)
        throws JsonParseException {
      return new UserType(json.getAsInt());
    }

    @Override
    public JsonElement serialize(UserType userType, Type typeOfSrc,
                                 JsonSerializationContext context) {
      return new JsonPrimitive(userType.rawValue());
    }
  }

  /**
   * A class to serialize and deserialize UserStatus class as c-layer uses ints for enum.
   */
  private static class UserStatusSerializer
      implements JsonSerializer<UserStatus>, JsonDeserializer<UserStatus> {
    @Override
    public UserStatus deserialize(JsonElement json, Type typeOfT,
                                  JsonDeserializationContext context) throws JsonParseException {
      return new UserStatus(json.getAsInt());
    }

    @Override
    public JsonElement serialize(UserStatus userStatus, Type typeOfSrc,
                                 JsonSerializationContext context) {
      return new JsonPrimitive(userStatus.rawValue());
    }
  }

  /**
   * enum denoting various possible attributes of the {@link CapabilityDoorLock}.
   */
  public enum AttributeId implements Capability.AttributeId {
    LOCKED_FLAG,
    USER_ID_ARR,
    PIN_CODE_ARR,
    USER_STATUS_ARR,
    USER_TYPE_ARR;

    public static AttributeId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }

    @Override
    public CapabilityId getCapabilityId() {
      return ID;
    }
  }

  /**
   * enum denoting the possible Commands that can be sent for {@link CapabilityDoorLock}.
   */
  public enum CommandId implements Capability.CommandId {
    LOCK,
    UNLOCK,
    TOGGLE_DOOR_LOCK,
    SET_PIN_CODE,
    CLEAR_PIN_CODE,
    SET_USER_STATUS,
    SET_USER_TYPE,
    GET_LOCK_STATUS,
    GET_PIN_CODE,
    GET_USER_STATUS,
    GET_USER_TYPE;

    public static CommandId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }
  }

  /**
   * possible values of user status for DoorLock capability.
   */
  public static final class UserStatus {

    public static final int AVAILABLE = 0;
    public static final int ENABLED = 1;
    public static final int DISABLED = 2;

    private final int userStatus;

    private UserStatus(int userStatus) {
      this.userStatus = userStatus;
    }

    public int rawValue() {
      return userStatus;
    }

    public static UserStatus available() {
      return new UserStatus(AVAILABLE);
    }

    public static UserStatus enabled() {
      return new UserStatus(ENABLED);
    }

    public static UserStatus disabled() {
      return new UserStatus(DISABLED);
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) {
        return true;
      }

      if (o == null || getClass() != o.getClass()) {
        return false;
      }

      UserStatus that = (UserStatus) o;
      return userStatus == that.userStatus;
    }

    @Override
    public int hashCode() {
      return userStatus;
    }

    @Override
    public String toString() {
      return "UserStatus{"
          + "userStatus=" + userStatus
          + '}';
    }
  }

  /**
   * possible values of user type for DoorLock capability.
   */
  public static final class UserType {

    public static final int UNRESTRICTED_USER = 0;
    public static final int YEAR_DAY_SCH_USER = 1;
    public static final int WEEK_DAY_SCH_USER = 2;
    public static final int MASTER_USER = 3;
    public static final int NON_ACCESS_USER = 4;

    private final int userType;

    private UserType(int userType) {
      this.userType = userType;
    }

    public int rawValue() {
      return userType;
    }

    public static UserType unrestrictedUser() {
      return new UserType(UNRESTRICTED_USER);
    }

    public static UserType yearDaySchUser() {
      return new UserType(YEAR_DAY_SCH_USER);
    }

    public static UserType weekDaySchUser() {
      return new UserType(WEEK_DAY_SCH_USER);
    }

    public static UserType masterUser() {
      return new UserType(MASTER_USER);
    }

    public static UserType nonAccessUser() {
      return new UserType(NON_ACCESS_USER);
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) {
        return true;
      }

      if (o == null || getClass() != o.getClass()) {
        return false;
      }

      UserType that = (UserType) o;
      return userType == that.userType;
    }

    @Override
    public int hashCode() {
      return userType;
    }

    @Override
    public String toString() {
      return "UserType{"
          + "userType=" + userType
          + '}';
    }
  }

  /**
   * Constructor of the current class.
   *
   * @param id     The unique id Of the capability
   * @param parent The parent Resource of the capability.
   */
  protected CapabilityDoorLock(int id, Resource parent) {
    super(id, parent);
  }

  /**
   * An initializer for the current class and adds serializers needed for this class.
   */
  static void init() {
    Command.GSON_BUILDER.registerTypeAdapter(UserType.class, new UserTypeSerializer());
    Command.GSON_BUILDER.registerTypeAdapter(UserStatus.class, new UserStatusSerializer());
  }

  /**
   * This function is used to check if a command is supported by the capability or not.
   *
   * @param commandId The Id denoting the command to be sent.
   * @return boolean: If the command is supported then True is returned.
   */
  @Override
  public boolean supports(Capability.CommandId commandId) {
    return (null == commandId || commandId instanceof CommandId) && super.supports(commandId);
  }

  /**
   * A function to create the command from the Json params.
   *
   * @param primitiveCommandId The int form of the commandId
   * @param commandParams      The Json params that can form a command
   * @return The command which is formed
   */
  @Override
  protected Command<? extends Capability.CommandId> extendedCreateCommand(
      int primitiveCommandId, JsonElement commandParams) {
    Command<CommandId> command;
    Gson gson = Command.GSON_BUILDER.create();
    CommandId commandId = CommandId.getEnum(primitiveCommandId);

    switch (commandId) {
      case LOCK: {
        command = gson.fromJson(commandParams, Lock.class);
        break;
      }

      case UNLOCK: {
        command = gson.fromJson(commandParams, Unlock.class);
        break;
      }

      case TOGGLE_DOOR_LOCK: {
        command = gson.fromJson(commandParams, ToggleDoorLock.class);
        break;
      }

      case SET_PIN_CODE: {
        command = gson.fromJson(commandParams, SetPin.class);
        break;
      }

      case CLEAR_PIN_CODE: {
        command = gson.fromJson(commandParams, ClearPin.class);
        break;
      }

      case SET_USER_STATUS: {
        command = gson.fromJson(commandParams, SetUserStatus.class);
        break;
      }

      case SET_USER_TYPE: {
        command = gson.fromJson(commandParams, SetUserType.class);
        break;
      }

      case GET_LOCK_STATUS: {
        command = gson.fromJson(commandParams, GetLockStatus.class);
        break;
      }

      case GET_PIN_CODE: {
        command = gson.fromJson(commandParams, GetPinCode.class);
        break;
      }

      case GET_USER_STATUS: {
        command = gson.fromJson(commandParams, GetUserStatus.class);
        break;
      }

      case GET_USER_TYPE: {
        command = gson.fromJson(commandParams, GetUserType.class);
        break;
      }

      default: {
        command = new Command<>(commandId);
      }
    }

    return command;
  }

  /**
   * This class can be sent as an argument to {@link #sendResourceCommand}
   * while sending Lock command.
   */
  public static class Lock extends Command<CommandId> {
    public Lock() {
      super(CommandId.LOCK);
    }
  }

  /**
   * This class can be sent as an argument to {@link #sendResourceCommand} while
   * sending Unlock command.
   */
  public static class Unlock extends Command<CommandId> {
    @SerializedName(Constants.TIME_OUT)
    public long timeout;

    public Unlock() {
      super(CommandId.UNLOCK);
    }
  }

  /**
   * This class can be sent as an argument to {@link #sendResourceCommand} while
   * sending Toggle door lock command.
   */
  public static class ToggleDoorLock extends Command<CommandId> {
    public ToggleDoorLock() {
      super(CommandId.TOGGLE_DOOR_LOCK);
    }
  }

  /**
   * This class can be sent as an argument to {@link #sendResourceCommand} while
   * setting pin code for a user.
   */
  public static class SetPin extends Command<CommandId> {
    @SerializedName(Constants.USER_ID)
    public long userId;

    @SerializedName(Constants.USER_STATUS)
    public UserStatus userStatus;

    @SerializedName(Constants.USER_TYPE)
    public UserType userType;

    @SerializedName(Constants.PIN_CODE)
    public String pin;

    public SetPin() {
      super(CommandId.SET_PIN_CODE);
    }
  }

  /**
   * This class can be sent as an argument to {@link #sendResourceCommand} while
   * clearing the pin of the user.
   */
  public static class ClearPin extends Command<CommandId> {
    @SerializedName(Constants.USER_ID)
    public long userId;

    public ClearPin() {
      super(CommandId.CLEAR_PIN_CODE);
    }
  }

  /**
   * This class can be sent as an argument to {@link #sendResourceCommand} while
   * setting user status.
   */
  public static class SetUserStatus extends Command<CommandId> {
    @SerializedName(Constants.USER_ID)
    public long userId;

    @SerializedName(Constants.USER_STATUS)
    public UserStatus userStatus;

    public SetUserStatus() {
      super(CommandId.SET_USER_STATUS);
    }
  }

  /**
   * This class can be sent as an argument to {@link #sendResourceCommand} while
   * changing the user type.
   */
  public static class SetUserType extends Command<CommandId> {
    @SerializedName(Constants.USER_ID)
    public long userId;

    @SerializedName(Constants.USER_TYPE)
    public UserType userType;

    public SetUserType() {
      super(CommandId.SET_USER_TYPE);
    }
  }

  /**
   * This class can be sent as an argument to {@link #sendResourceCommand} while
   * fetching lock status command.
   */
  public static class GetLockStatus extends Command<CommandId> {
    public GetLockStatus() {
      super(CommandId.GET_LOCK_STATUS);
    }
  }

  /**
   * This class can be sent as an argument to {@link #sendResourceCommand} while
   * getting pin of a user.
   */
  public static class GetPinCode extends Command<CommandId> {
    @SerializedName(Constants.USER_ID)
    public long userId;

    public GetPinCode() {
      super(CommandId.GET_PIN_CODE);
    }
  }

  /**
   * This class can be sent as an argument to {@link #sendResourceCommand} while
   * getting user status.
   */
  public static class GetUserStatus extends Command<CommandId> {
    @SerializedName(Constants.USER_ID)
    public long userId;

    public GetUserStatus() {
      super(CommandId.GET_USER_STATUS);
    }
  }

  /**
   * This class can be sent as an argument to {@link #sendResourceCommand} while
   * getting the type of user.
   */
  public static class GetUserType extends Command<CommandId> {
    @SerializedName(Constants.USER_ID)
    public long userId;

    public GetUserType() {
      super(CommandId.GET_USER_TYPE);
    }
  }
}
