/*
 * @authors   Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.Gson;
import com.google.gson.JsonElement;

/**
 * This class describes Commands and Attributes for Fluid Sensing Capability of a Resource.
 */
public class CapabilityFluidLevelSensing extends Capability {

  public static final CapabilityId ID = CapabilityId.FLUID_LEVEL_MEASUREMENT;

  /**
   * enum denoting various possible attributes of the {@link CapabilityFluidLevelSensing}.
   */
  public enum AttributeId implements Capability.AttributeId {
    OVERFLOW_FLAG,
    LEAK_FLAG;

    public static AttributeId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }

    @Override
    public CapabilityId getCapabilityId() {
      return ID;
    }
  }

  /**
   * enum denoting the possible Commands that can be sent for {@link CapabilityFluidLevelSensing}.
   */
  public enum CommandId implements Capability.CommandId {
    ENABLE,
    DISABLE;

    public static CommandId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }
  }

  /**
   * Constructor of the current class.
   *
   * @param id     The unique id Of the capability
   * @param parent The parent Resource of the capability.
   */
  protected CapabilityFluidLevelSensing(int id, Resource parent) {
    super(id, parent);
  }

  /**
   * This function is used to check if a command is supported by the capability or not.
   *
   * @param commandId The Id denoting the command to be sent.
   * @return boolean: If the command is supported then True is returned.
   */
  @Override
  public boolean supports(Capability.CommandId commandId) {
    return (null == commandId || commandId instanceof CommandId) && super.supports(commandId);
  }

  /**
   * A function to create the command from the Json params.
   *
   * @param primitiveCommandId The int form of the commandId
   * @param commandParams      The Json params that can form a command
   * @return The command which is formed
   */
  @Override
  protected Command<? extends Capability.CommandId> extendedCreateCommand(
      int primitiveCommandId, JsonElement commandParams) {
    Command<CommandId> command;
    Gson gson = Command.GSON_BUILDER.create();
    CommandId commandId = CommandId.getEnum(primitiveCommandId);

    switch (commandId) {
      case ENABLE: {
        command = gson.fromJson(commandParams, Enable.class);
        break;
      }

      case DISABLE: {
        command = gson.fromJson(commandParams, Disable.class);
        break;
      }

      default:
        command = new Command<>(commandId);
    }
    return command;
  }

  /**
   * This class can be sent as an argument to {@link #sendResourceCommand}
   * while sending enable fluid sensing command.
   */
  public static class Enable extends Command<CommandId> {
    public Enable() {
      super(CommandId.ENABLE);
    }
  }

  /**
   * This class can be sent as an argument to {@link #sendResourceCommand}
   * while sending disable fluid sensing command.
   */
  public static class Disable extends Command<CommandId> {
    public Disable() {
      super(CommandId.DISABLE);
    }
  }
}
