/*
 * @authors   Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.Gson;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonParseException;
import com.google.gson.JsonPrimitive;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import com.google.gson.annotations.SerializedName;
import java.lang.reflect.Type;

/**
 * This class describes Commands and Attributes for HVAC Control Capability of a Resource.
 */
public class CapabilityHvacControl extends Capability {

  public static final CapabilityId ID = CapabilityId.HVAC_CONTROL;

  /**
   * possible values of {@link AttributeId#FAN_SPEED}.
   */
  public static final class FanSpeed {

    public static final int OFF = 0;
    public static final int LOW = 1;
    public static final int MEDIUM = 2;
    public static final int HIGH = 3;
    public static final int AUTO = 4;

    private final int speed;

    private FanSpeed(int speed) {
      this.speed = speed;
    }

    public int rawValue() {
      return speed;
    }

    public static FanSpeed off() {
      return new FanSpeed(OFF);
    }

    public static FanSpeed low() {
      return new FanSpeed(LOW);
    }

    public static FanSpeed med() {
      return new FanSpeed(MEDIUM);
    }

    public static FanSpeed high() {
      return new FanSpeed(HIGH);
    }

    public static FanSpeed auto() {
      return new FanSpeed(AUTO);
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) {
        return true;
      }

      if (!(o instanceof FanSpeed)) {
        return false;
      }

      FanSpeed fanSpeed = (FanSpeed) o;

      return speed == fanSpeed.speed;
    }

    @Override
    public int hashCode() {
      return speed;
    }
  }

  /**
   * possible values for {@link AttributeId#MODE} and {@link AttributeId#SUPPORTED_MODES_ARR}.
   */
  public static final class Mode {

    public static final int COOL = 0;
    public static final int HEAT = 1;
    public static final int AUTO = 2;
    public static final int EMERGENCY_HEAT = 3;
    public static final int PRE_COOLING = 4;
    public static final int FAN_ONLY = 5;
    public static final int DRY = 6;
    public static final int SLEEP = 7;

    private final int mode;

    private Mode(int mode) {
      this.mode = mode;
    }

    public int rawValue() {
      return mode;
    }

    public static Mode cool() {
      return new Mode(COOL);
    }

    public static Mode heat() {
      return new Mode(HEAT);
    }

    public static Mode auto() {
      return new Mode(AUTO);
    }

    public static Mode emergencyHeat() {
      return new Mode(EMERGENCY_HEAT);
    }

    public static Mode preCooling() {
      return new Mode(PRE_COOLING);
    }

    public static Mode fan() {
      return new Mode(FAN_ONLY);
    }

    public static Mode dry() {
      return new Mode(DRY);
    }

    public static Mode sleep() {
      return new Mode(SLEEP);
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) {
        return true;
      }

      if (!(o instanceof Mode)) {
        return false;
      }

      Mode other = (Mode) o;

      return mode == other.mode;
    }

    @Override
    public int hashCode() {
      return mode;
    }
  }

  /**
   * enum denoting various possible attributes of the {@link CapabilityHvacControl}.
   */
  public enum AttributeId implements Capability.AttributeId {
    SUPPORTED_MODES_ARR,
    POWER_ON_FLAG,
    TEMPERATURE_CELSIUS,
    MODE,
    FAN_SPEED,
    SWING_ON_FLAG;

    public static AttributeId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }

    @Override
    public CapabilityId getCapabilityId() {
      return ID;
    }
  }

  /**
   * enum denoting the possible Commands that can be sent for {@link CapabilityHvacControl}.
   */
  public enum CommandId implements Capability.CommandId {
    SET_STATE,
    SYNC_STATE;

    public static CommandId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }
  }

  /**
   * Constructor of the current class.
   *
   * @param id     The unique id Of the capability
   * @param parent The parent Resource of the capability.
   */
  protected CapabilityHvacControl(int id, Resource parent) {
    super(id, parent);
  }

  static void init() {
    Command.GSON_BUILDER.registerTypeAdapter(FanSpeed.class, new FanSpeedParser());
    Command.GSON_BUILDER.registerTypeAdapter(Mode.class, new ModeParser());
  }

  /**
   * This function is used to check if a command is supported by the capability or not.
   *
   * @param commandId The Id denoting the command to be sent.
   * @return boolean: If the command is supported then True is returned.
   */
  @Override
  public boolean supports(Capability.CommandId commandId) {
    return (null == commandId || commandId instanceof CommandId) && super.supports(commandId);
  }

  /**
   * A function to create the command from the Json params.
   *
   * @param primitiveCommandId The int form of the commandId
   * @param commandParams      The Json params that can form a command
   * @return The command which is formed
   */
  @Override
  protected Command<? extends Capability.CommandId> extendedCreateCommand(
      int primitiveCommandId, JsonElement commandParams) {
    Command<? extends Capability.CommandId> command;
    Gson gson = Command.GSON_BUILDER.create();
    CommandId commandId = CommandId.getEnum(primitiveCommandId);

    switch (commandId) {
      case SET_STATE: {
        command = gson.fromJson(commandParams, SetHvacState.class);
        break;
      }

      case SYNC_STATE: {
        command = gson.fromJson(commandParams, SyncState.class);
        break;
      }

      default: {
        command = new Command<>(commandId);
        break;
      }
    }

    return command;
  }

  /**
   * This class can be sent as an argument to {@link #sendResourceCommand}
   * while setting HVAC state command.
   */
  public static class SetHvacState extends Command<CommandId> {
    @SerializedName(Constants.HVAC_POWER_ON_FLAG)
    public boolean powerOnFlag;
    @SerializedName(Constants.HVAC_SWING_ON)
    public boolean swingOnFlag;
    @SerializedName(Constants.HVAC_TEMPERATURE)
    public int temperatureCelsius;
    @SerializedName(Constants.HVAC_MODE)
    public Mode mode;
    @SerializedName(Constants.HVAC_FAN_SPEED)
    public FanSpeed fanSpeed;

    public SetHvacState() {
      super(CommandId.SET_STATE);
    }
  }

  /**
   * This class can be sent as an argument to {@link #sendResourceCommand}
   * while sending sync state command.
   */
  public static class SyncState extends Command<CommandId> {
    public SyncState() {
      super(CommandId.SYNC_STATE);
    }
  }

  private static final class ModeParser implements JsonSerializer<Mode>, JsonDeserializer<Mode> {

    @Override
    public JsonElement serialize(Mode mode, Type type,
                                 JsonSerializationContext jsonSerializationContext) {
      return new JsonPrimitive(mode.mode);
    }

    @Override
    public Mode deserialize(JsonElement json, Type type,
                            JsonDeserializationContext jsonDeserializationContext) throws
        JsonParseException {
      if (!json.isJsonPrimitive() || !json.getAsJsonPrimitive().isNumber()) {
        throw new JsonParseException("invalid type: " + json);
      }

      return new Mode(json.getAsInt());
    }
  }

  private static final class FanSpeedParser
      implements JsonSerializer<FanSpeed>, JsonDeserializer<FanSpeed> {

    @Override
    public JsonElement serialize(FanSpeed fanSpeed, Type type,
                                 JsonSerializationContext jsonSerializationContext) {
      return new JsonPrimitive(fanSpeed.speed);
    }

    @Override
    public FanSpeed deserialize(JsonElement json, Type type,
                                JsonDeserializationContext jsonDeserializationContext)
        throws JsonParseException {
      if (!json.isJsonPrimitive() || !json.getAsJsonPrimitive().isNumber()) {
        throw new JsonParseException("invalid type: " + json);
      }

      return new FanSpeed(json.getAsInt());
    }
  }
}
