/*
 * @authors   Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.annotations.SerializedName;

/**
 * This class describes Commands and Attributes for Image control Capability of a Resource.
 */
public class CapabilityImageControl extends Capability {

  public static final CapabilityId ID = CapabilityId.IMAGE_CONTROL;

  /**
   * enum denoting various possible attributes of the {@link CapabilityImageControl}.
   */
  public enum AttributeId implements Capability.AttributeId {
    AF_MODE_FLAG,
    AE_MODE_FLAG,
    AWB_MODE_FLAG,
    ZOOM_TYPE,
    IR_CUT_MODE_TYPE,
    IMG_PRESET_TYPE,
    IMG_HUE_PCT,
    IMG_SATURATION_PCT,
    IMG_CONTRAST_PCT,
    IMG_BRIGHTNESS_PCT,
    IMG_FLIP_FLAG,           // Image flip is enabled if true
    IMG_MIRROR_FLAG,         // Image mirror is enabled if true
    IMG_TEXT,
    IMG_TEXT_X_POSITION,
    IMG_TEXT_Y_POSITION;

    public static AttributeId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }

    @Override
    public CapabilityId getCapabilityId() {
      return ID;
    }
  }

  /**
   * enum denoting the possible Commands that can be sent for {@link CapabilityImageControl}.
   */
  public enum CommandId implements Capability.CommandId {
    /**
     * not yet supported.
     */
    SET_AF_MODE,
    SET_AE_MODE,
    SET_AWB_MODE,
    SET_ZOOM,
    SET_IR_CUT_MODE,
    SET_PRESET,
    SET_HUE,
    SET_SATURATION,
    SET_CONTRAST,
    SET_BRIGHTNESS,
    /**
     * supported.
     */
    SET_FLIP,
    SET_MIRROR,
    SET_TEXT;

    public static CommandId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }
  }

  /**
   * Constructor of the current class.
   *
   * @param id     The unique id Of the capability
   * @param parent The parent Resource of the capability.
   */
  protected CapabilityImageControl(int id, Resource parent) {
    super(id, parent);
  }

  /**
   * This function is used to check if a command is supported by the capability or not.
   *
   * @param commandId The Id denoting the command to be sent.
   * @return boolean: If the command is supported then True is returned.
   */
  @Override
  public boolean supports(Capability.CommandId commandId) {
    return (null == commandId || commandId instanceof CommandId) && super.supports(commandId);
  }

  /**
   * A function to create the command from the Json params.
   *
   * @param primitiveCommandId The int form of the commandId
   * @param commandParams      The Json params that can form a command
   * @return The command which is formed
   */
  @Override
  protected Command<? extends Capability.CommandId>
      extendedCreateCommand(int primitiveCommandId, JsonElement commandParams) {

    Command<CommandId> command;
    Gson gson = Command.GSON_BUILDER.create();
    CommandId commandId = CommandId.getEnum(primitiveCommandId);

    switch (commandId) {
      case SET_AF_MODE: {
        command = gson.fromJson(commandParams, SetAfMode.class);
        break;
      }

      case SET_AE_MODE: {
        command = gson.fromJson(commandParams, SetAeMode.class);
        break;
      }

      case SET_AWB_MODE: {
        command = gson.fromJson(commandParams, SetAwbMode.class);
        break;
      }

      case SET_ZOOM: {
        command = gson.fromJson(commandParams, SetZoom.class);
        break;
      }

      case SET_IR_CUT_MODE: {
        command = gson.fromJson(commandParams, SetIrCutMode.class);
        break;
      }

      case SET_PRESET: {
        command = gson.fromJson(commandParams, SetPreset.class);
        break;
      }

      case SET_HUE: {
        command = gson.fromJson(commandParams, SetHue.class);
        break;
      }

      case SET_SATURATION: {
        command = gson.fromJson(commandParams, SetSaturation.class);
        break;
      }

      case SET_CONTRAST: {
        command = gson.fromJson(commandParams, SetContrast.class);
        break;
      }

      case SET_BRIGHTNESS: {
        command = gson.fromJson(commandParams, SetBrightness.class);
        break;
      }

      case SET_TEXT: {
        command = gson.fromJson(commandParams, SetText.class);
        break;
      }

      case SET_FLIP: {
        command = gson.fromJson(commandParams, SetFlip.class);
        break;
      }

      case SET_MIRROR: {
        command = gson.fromJson(commandParams, SetMirror.class);
        break;
      }

      default:
        command = new Command<>(commandId);
    }

    return command;
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending Auto Focus mode commands.
   */
  public static class SetAfMode extends Command<CommandId> {
    public SetAfMode() {
      super(CommandId.SET_AF_MODE);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending Auto Exposure mode commands.
   */
  public static class SetAeMode extends Command<CommandId> {
    public SetAeMode() {
      super(CommandId.SET_AE_MODE);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending Auto White Balance mode commands.
   */
  public static class SetAwbMode extends Command<CommandId> {
    public SetAwbMode() {
      super(CommandId.SET_AWB_MODE);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending set zoom commands.
   */
  public static class SetZoom extends Command<CommandId> {
    public SetZoom() {
      super(CommandId.SET_ZOOM);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending Infrared cut mode commands.
   */
  public static class SetIrCutMode extends Command<CommandId> {
    public SetIrCutMode() {
      super(CommandId.SET_IR_CUT_MODE);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending set preset commands.
   */
  public static class SetPreset extends Command<CommandId> {
    public SetPreset() {
      super(CommandId.SET_PRESET);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending set hue commands.
   */
  public static class SetHue extends Command<CommandId> {
    public SetHue() {
      super(CommandId.SET_HUE);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending set saturation commands.
   */
  public static class SetSaturation extends Command<CommandId> {
    public SetSaturation() {
      super(CommandId.SET_SATURATION);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending set contrast commands.
   */
  public static class SetContrast extends Command<CommandId> {
    public SetContrast() {
      super(CommandId.SET_CONTRAST);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending set brightness commands.
   */
  public static class SetBrightness extends Command<CommandId> {
    public SetBrightness() {
      super(CommandId.SET_BRIGHTNESS);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending set text commands.
   */
  public static class SetText extends Command<CommandId> {
    @SerializedName(Constants.TEXT)
    public String text;
    @SerializedName(Constants.X_POSITION)
    public int xpos;
    @SerializedName(Constants.Y_POSITION)
    public int ypos;

    public SetText() {
      super(CommandId.SET_TEXT);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending set mirror commands.
   */
  public static class SetMirror extends Command<CommandId> {
    @SerializedName(Constants.MIRROR_FLAG)
    public boolean enable;

    public SetMirror() {
      super(CommandId.SET_MIRROR);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending set flip commands.
   */
  public static class SetFlip extends Command<CommandId> {
    @SerializedName(Constants.FLIP_FLAG)
    public boolean enable;

    public SetFlip() {
      super(CommandId.SET_FLIP);
    }
  }
}
