/*
 * @authors   Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonParseException;
import com.google.gson.JsonPrimitive;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import java.lang.reflect.Type;

/**
 * This class describes Commands and Attributes for Key press sensing Capability of a Resource.
 */
public class CapabilityKeyPressSensing extends Capability {

  public static final CapabilityId ID = CapabilityId.KEY_PRESS_SENSING;

  /**
   * enum denoting various possible attributes of the {@link CapabilityKeyPressSensing}.
   */
  public enum AttributeId implements Capability.AttributeId {
    SCENE_ID,
    SCENE_KEY,
    DISARM_FLAG,
    ARM_DAY_FLAG,
    ARM_NIGHT_FLAG,
    ARM_ALL_FLAG,
    PANIC_FLAG,
    EMERGENCY_FLAG,
    FALL_TRIGGER_FLAG,
    ACTIVATED_COMBO_ID,
    COMBO_NAME_ARR;

    public static AttributeId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }

    @Override
    public CapabilityId getCapabilityId() {
      return ID;
    }
  }

  /**
   * enum denoting the possible Commands that can be sent for {@link CapabilityKeyPressSensing}.
   */
  public enum CommandId implements Capability.CommandId {
    // Note : Update extendedCreateCommand method whenever commandId is added
    ;

    public static CommandId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }
  }

  /**
   * possible values of key press scene for the key press sensing capability.
   */
  public static final class KeyPressScene {
    public static final int SINGLE = 0;
    public static final int DOUBLE = 1;
    public static final int TRIPLE = 2;
    public static final int QUAD = 3;
    public static final int QUINT = 4;
    public static final int HOLD = 5;
    public static final int RELEASE = 6;

    private final int keyPressScene;

    private KeyPressScene(int keyPressScene) {
      this.keyPressScene = keyPressScene;
    }

    public int rawValue() {
      return keyPressScene;
    }

    public static KeyPressScene sceneSingle() {
      return new KeyPressScene(SINGLE);
    }

    public static KeyPressScene sceneDouble() {
      return new KeyPressScene(DOUBLE);
    }

    public static KeyPressScene sceneTriple() {
      return new KeyPressScene(TRIPLE);
    }

    public static KeyPressScene sceneQuad() {
      return new KeyPressScene(QUAD);
    }

    public static KeyPressScene sceneQuint() {
      return new KeyPressScene(QUINT);
    }

    public static KeyPressScene sceneHold() {
      return new KeyPressScene(HOLD);
    }

    public static KeyPressScene sceneRelease() {
      return new KeyPressScene(RELEASE);
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) {
        return true;
      }
      if (o == null || getClass() != o.getClass()) {
        return false;
      }

      KeyPressScene that = (KeyPressScene) o;
      return keyPressScene == that.keyPressScene;
    }

    @Override
    public int hashCode() {
      return keyPressScene;
    }

    @Override
    public String toString() {
      return "KeyPressScene{"
          + "keyPressScene=" + keyPressScene
          + '}';
    }
  }

  private static class KeyPressSceneSerializer implements JsonSerializer<KeyPressScene>,
      JsonDeserializer<KeyPressScene> {

    @Override
    public KeyPressScene deserialize(
        JsonElement json, Type typeOfT, JsonDeserializationContext context)
        throws JsonParseException {
      return new KeyPressScene(json.getAsInt());
    }

    @Override
    public JsonElement serialize(KeyPressScene keyPressScene, Type typeOfSrc,
                                 JsonSerializationContext context) {
      return new JsonPrimitive(keyPressScene.rawValue());
    }
  }

  static void init() {
    Command.GSON_BUILDER.registerTypeAdapter(KeyPressScene.class, new KeyPressSceneSerializer());
  }

  /**
   * Constructor of the current class.
   *
   * @param id     The unique id Of the capability
   * @param parent The parent Resource of the capability.
   */
  protected CapabilityKeyPressSensing(int id, Resource parent) {
    super(id, parent);
  }

  /**
   * This function is used to check if a command is supported by the capability or not.
   *
   * @param commandId The Id denoting the command to be sent.
   * @return boolean: If the command is supported then True is returned.
   */
  @Override
  public boolean supports(Capability.CommandId commandId) {
    return (null == commandId || commandId instanceof CommandId) && super.supports(commandId);
  }

  /**
   * A function to create the command from the Json params.
   *
   * @param primitiveCommandId The int form of the commandId
   * @param commandParams      The Json params that can form a command
   * @return The command which is formed
   */
  @Override
  protected Command<? extends Capability.CommandId> extendedCreateCommand(
      int primitiveCommandId, JsonElement commandParams) {
    CommandId commandId = CommandId.getEnum(primitiveCommandId);

    return new Command<>(commandId);
  }
}
