/*
 * @authors   Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.annotations.SerializedName;

/**
 * This class describes Commands and Attributes for Level Control Capability of a Resource.
 */
public class CapabilityLevel extends Capability {

  public static final CapabilityId ID = CapabilityId.LEVEL_CTRL;

  /**
   * enum denoting various possible attributes of the {@link CapabilityLevel}.
   */
  public enum AttributeId implements Capability.AttributeId {
    LEVEL_PCT;

    public static AttributeId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }

    @Override
    public CapabilityId getCapabilityId() {
      return ID;
    }
  }

  /**
   * enum denoting the possible Commands that can be sent for {@link CapabilityLevel}.
   */
  public enum CommandId implements Capability.CommandId {
    SET_LEVEL,
    SET_LEVEL_WITH_ON_OFF,
    FETCH_LEVEL;

    public static CommandId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }
  }

  /**
   * Constructor of the current class.
   *
   * @param id     The unique id Of the capability
   * @param parent The parent Resource of the capability.
   */
  protected CapabilityLevel(int id, Resource parent) {
    super(id, parent);
  }

  /**
   * This function is used to check if a command is supported by the capability or not.
   *
   * @param commandId The Id denoting the command to be sent.
   * @return boolean: If the command is supported then True is returned.
   */
  @Override
  public boolean supports(Capability.CommandId commandId) {
    return (null == commandId || commandId instanceof CommandId) && super.supports(commandId);
  }

  /**
   * A function to create the command from the Json params.
   *
   * @param primitiveCommandId The int form of the commandId
   * @param commandParams      The Json params that can form a command
   * @return The command which is formed
   */
  @Override
  protected Command<? extends Capability.CommandId> extendedCreateCommand(
      int primitiveCommandId, JsonElement commandParams) {
    Command<CommandId> command;
    Gson gson = Command.GSON_BUILDER.create();
    CommandId commandId = CommandId.getEnum(primitiveCommandId);

    switch (commandId) {
      case SET_LEVEL: {
        command = gson.fromJson(commandParams, SetLevel.class);
        break;
      }

      case SET_LEVEL_WITH_ON_OFF: {
        command = gson.fromJson(commandParams, SetLevelOnOff.class);
        break;
      }

      case FETCH_LEVEL: {
        command = gson.fromJson(commandParams, FetchLevel.class);
        break;
      }

      default: {
        command = new Command<>(commandId);
      }
    }

    return command;
  }

  /**
   * This is a super class for SetLevel and SetLevelOnOff commands.
   */
  public static class LevelCommand extends Command<CommandId> {

    @SerializedName(Constants.LEVEL)
    public int levelPct;

    @SerializedName(Constants.USE_DEFAULT_TRANSITION)
    public boolean useDefaultTransitionTime;

    @SerializedName(Constants.TRANSITION_TIME)
    public int transitionTime;

    LevelCommand(CommandId commandId, long commandTimeout) {
      super(commandId, commandTimeout);
    }

    LevelCommand(CommandId commandId) {
      super(commandId);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand}
   * while sending level control commands.
   */
  public static class SetLevel extends LevelCommand {

    /**
     * A no args constructor for GSON.
     */
    SetLevel() {
      super(CommandId.SET_LEVEL);
    }

    /**
     * Constructor with a param.
     *
     * @param useDefaultTransitionTime flag indicating if the transition has to take
     *                                 place. NOTE:The remaining arguments can be set
     *                                 after the object creation.
     */
    public SetLevel(boolean useDefaultTransitionTime) {
      super(CommandId.SET_LEVEL);
      this.useDefaultTransitionTime = useDefaultTransitionTime;
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand}
   * while sending Level control commands with OnOff toggling.
   */
  public static class SetLevelOnOff extends LevelCommand {

    @SerializedName(Constants.RESTORE_LEVEL)
    public boolean restoreLevel;

    /**
     * A no args constructor for GSON.
     */
    SetLevelOnOff() {
      super(CommandId.SET_LEVEL_WITH_ON_OFF);
    }

    /**
     * Constructor with a param.
     *
     * @param useDefaultTransitionTime flag indicating if the transition has to
     *                                 take place. NOTE: The remaining arguments
     *                                 can be set after the object creation.
     */
    public SetLevelOnOff(boolean useDefaultTransitionTime) {
      super(CommandId.SET_LEVEL_WITH_ON_OFF);
      this.useDefaultTransitionTime = useDefaultTransitionTime;
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand}
   * while fetching Level commands.
   */
  public static class FetchLevel extends Command<CommandId> {
    public FetchLevel() {
      super(CommandId.FETCH_LEVEL);
    }
  }
}
