/*
 * @authors   Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.Gson;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonParseException;
import com.google.gson.JsonPrimitive;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import com.google.gson.annotations.SerializedName;
import java.lang.reflect.Type;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Objects;

/**
 * This class describes Commands and Attributes for Media Streaming Capability of a Resource.
 */
public class CapabilityMediaStreaming extends Capability {
  private static final String TAG = "CapabilityMediaStreaming";

  public static final CapabilityId ID = CapabilityId.MEDIA_STREAM;

  /**
   * possible values of frame types for the media streaming capability.
   */
  public static final class FrameTypes {
    public static final int NONE = 0;   // not a key frame
    public static final int KEY = 1;    // key frame
  }

  /**
   * enum denoting various possible attributes of the {@link CapabilityMediaStreaming}.
   */
  public enum AttributeId implements Capability.AttributeId {
    DESCRIPTOR,                         // media stream description
    ID_ARR,                             // array of streamingIds eg: {0, 1, 2}
    ACTIVE_CLIENTS_ARR,                 // array of active clients count per stream eg: {2, 2, 2}
    MAX_ACTIVE_CLIENTS_ARR,             // array of  max clients count per stream eg:{2, 2, 2}
    REC_STATUS_ARR,                     /* array of streamIds recording status
                                           eg: {true, false, false}
                                           true indicates recording in-progress */
    REC_SUPPORTED_ARR,                  /* boolean array mapped to streamIds
                                           indicates the whether a stream supports recording
                                           eg: {true, false, false}
                                           true indicates streamId supports recording */
    RECORDING_IN_PROGRESS,              /* array of timers one-to-one mapped to streamIds
                                           in COCO_STD_ATTR_MEDIA_STREAM_ID_ARR */
    CONTENT_SEND_BUFFER_SIZE_ARR,       /* array of send buffer sizes mapped to streamIds
                                           in COCO_STD_ATTR_MEDIA_STREAM_ID_ARR */
    CONTENT_RECEIVE_BUFFER_SIZE_ARR;    /* array of receive buffer sizes mapped to streamIds
                                           in COCO_STD_ATTR_MEDIA_STREAM_ID_ARR */

    public static AttributeId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }

    @Override
    public CapabilityId getCapabilityId() {
      return ID;
    }
  }

  /**
   * enum denoting the possible Commands that can be sent for {@link CapabilityMediaStreaming}.
   */
  public enum CommandId implements Capability.CommandId {
    STREAM_START,
    STREAM_STOP,
    RECORD_START,
    RECORD_STOP,
    RECORD_CONFIG;

    public static CommandId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }
  }

  /**
   * possible values of media transport type for the media streaming capability.
   */
  public enum TransportType {
    TCP,
    UDP,
    RTP_UDP;

    static TransportType getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    int getInt() {
      return ordinal();
    }
  }

  /**
   * possible values of media stream status for the media streaming capability.
   */
  public enum Status {
    UNKNOWN,
    OPENED,
    FAILED,
    CLOSED,
    TIMEOUT,
    MAX_CLIENT_REACHED;

    static Status getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    int getInt() {
      return ordinal();
    }
  }

  /**
   * possible values of media session type for the media streaming capability.
   */
  public enum MediaSessionType {
    BUFFERED,
    LIVE;

    static MediaSessionType getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    int getInt() {
      return ordinal();
    }
  }

  /**
   * possible values of record mode for the media streaming capability.
   */
  public enum RecordMode {
    MANUAL_STOP,
    TIME,
    SIZE;

    static RecordMode getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    int getInt() {
      return ordinal();
    }
  }

  /**
   * Listener that receives updates from callbacks.
   */
  public interface MediaStreamListener extends Listener {
    // streamHandle is a native pointer and mustn't be messed
    default void onStatusChanged(long streamHandle, int channelPort, Status status) {
    }
    /* NOTE: data will be freed as soon this call ends.
    So, if being used async then a copy is mandatory */

    /**
     * NOTE: Frame types are present in {@link FrameTypes}.
     */
    default void onDataReceived(long streamHandle, int channelPort, long frameIndex, int frameType,
                                long frameDuration, long framePts, ByteBuffer data) {
    }
  }

  private static final class RecordModeParser implements JsonSerializer<RecordMode>,
      JsonDeserializer<RecordMode> {

    @Override
    public RecordMode deserialize(JsonElement json, Type typeOfT,
                                  JsonDeserializationContext context) throws
        JsonParseException {
      return RecordMode.getEnum(json.getAsInt());
    }

    @Override
    public JsonElement serialize(RecordMode src, Type typeOfSrc, JsonSerializationContext context) {
      return new JsonPrimitive(src.getInt());
    }
  }

  /**
   * Constructor of the current class.
   *
   * @param id     The unique id Of the capability
   * @param parent The parent Resource of the capability.
   */
  protected CapabilityMediaStreaming(int id, Resource parent) {
    super(id, parent);
  }

  static void init() {
    Command.GSON_BUILDER.registerTypeAdapter(RecordMode.class, new RecordModeParser());
  }

  @Override
  public boolean supports(Capability.CommandId commandId) {
    return (null == commandId || commandId instanceof CommandId) && super.supports(commandId);
  }

  /**
   * This function fetches the capability that is using provided stream handle.
   *
   * @param streamHandle       a non zero long value representing a native pointer
   * @return Capability retrieved from capability map.
   * @see Resource#getCapability
   */
  public static Capability getHandlingCapability(long streamHandle) {
    Capability capability = null;
    Resource resource =
        CocoClient.getInstance().getNativeHandler().getHandlingResource(streamHandle);

    if (null != resource) {
      capability = resource.getCapability(CapabilityId.MEDIA_STREAM);
    }

    return capability;
  }

  public static long getHandlingStreamId(long streamHandle) {
    return CocoClient.getInstance().getNativeHandler().getHandlingStreamId(streamHandle);
  }

  public static int getHandlingStreamSessionId(long streamHandle) {
    return CocoClient.getInstance().getNativeHandler().getHandlingStreamSessionId(streamHandle);
  }

  public static int[] getHandlingChannelPorts(long streamHandle) {
    return CocoClient.getInstance().getNativeHandler().getHandlingChannelPorts(streamHandle);
  }

  @Override
  protected Command<? extends Capability.CommandId> extendedCreateCommand(
      int primitiveCommandId, JsonElement commandParams) {
    Command<CommandId> command;
    Gson gson = Command.GSON_BUILDER.create();
    CommandId commandId = CommandId.getEnum(primitiveCommandId);

    switch (commandId) {
      case STREAM_START:
      case STREAM_STOP: {
        command = null;
        break;
      }

      case RECORD_START: {
        command = gson.fromJson(commandParams, StartRecording.class);
        break;
      }

      case RECORD_STOP: {
        command = gson.fromJson(commandParams, StopRecording.class);
        break;
      }

      case RECORD_CONFIG: {
        command = gson.fromJson(commandParams, ConfigRecording.class);
        break;
      }

      default:
        command = new Command<>(commandId);
    }

    return command;
  }

  public int[] getAvailableChannelPorts(int portCount) {
    return getParent().getParent().getAvailableChannelPorts(portCount);
  }

  /**
   * A function to start streaming using this capability.
   *
   * @param streamId             The ID over which this communication should take place
   * @param streamSessionId      The sessionId which you would like to provide
   * @param streamDescription    The description (probably in SDP or other
   *                             formats supported by the capability)
   * @param transportTypes       The transportTypes corresponding to the channels
   * @param sessionType          An enum denoting session is live/buffered
   * @param timeout              The timeout for this command in millis
   * @param listener             The interface over which the callbacks should be received
   * @return  An array of free ports available for communication.
   */
  public int[] startStream(long streamId, int streamSessionId, String streamDescription,
                           TransportType[] transportTypes, MediaSessionType sessionType,
                           long timeout, MediaStreamListener listener) {

    int[] freePorts = getParent().getParent().getAvailableChannelPorts(transportTypes.length);

    if (null == freePorts) {
      throw new RuntimeException("cannot find free ports");
    }

    Log.d(TAG, "freePorts: " + Arrays.toString(freePorts));

    startStream(streamId, streamSessionId, streamDescription, freePorts, transportTypes,
        sessionType, timeout, listener);

    return freePorts;
  }

  /**
   * A function to start streaming using this capability.
   *
   * @param streamId          The ID over which this communication should take place
   * @param streamSessionId   The sessionId which you would like to provide
   * @param streamDescription The description (probably in SDP or other
   *                          formats supported by the capability)
   * @param channelPorts      The Ports over which this communication should take place
   * @param transportTypes    The transportTypes corresponding to the channels
   * @param listener          The interface over which the callbacks should be received
   * @param timeout           The timeout for this command in millis
   *     NOTE: For multiplexing ability over individualChannels see:
   *       {@link MediaStreamListenerMultiplexer} and
   *       {@link DefaultMediaStreamListenerMultiplexer}
   */
  public void startStream(long streamId, int streamSessionId, String streamDescription,
                          int[] channelPorts, TransportType[] transportTypes,
                          MediaSessionType sessionType, long timeout,
                          MediaStreamListener listener) {

    Objects.requireNonNull(listener);

    CocoClient.getInstance().getNativeHandler()
        .startMediaStream(this, streamId, streamSessionId, streamDescription, channelPorts,
            transportTypes, sessionType, timeout, listener);
  }

  /**
   * This function sends data for communication.
   *
   * @param streamHandle          a non zero long value representing a native pointer
   * @param channelPort           The Port over which this communication should take place
   * @param frameIndex            index of
   * @param frameType             frame type as present in {@link FrameTypes}
   * @param frameDuration         duration for each frame
   * @param framePts              presentation time stamp field
   * @param data                  byte array of data to be sent
   * @param offset                offset to included for communication delay
   * @param size                  size of the frame data
   * @return queued number of bytes
   */
  public int sendData(long streamHandle, int channelPort, long frameIndex, int frameType,
                      long frameDuration, long framePts, byte[] data, int offset, int size) {

    return sendData(streamHandle, channelPort, frameIndex, frameType, frameDuration, framePts,
        ByteBuffer.wrap(data, offset, size));
  }

  /**
   * This function sends data for communication.
   * NOTE: reads between 0 and limit on the buffer synchronization is needed coz,
   *       a single buffer is maintained in c-layer for all the send commands
   *
   * @param streamHandle          A non zero long value representing a native pointer
   * @param channelPort           The Port over which this communication should take place
   * @param frameIndex            index which increments on every call
   * @param frameType             frame type as present in {@link FrameTypes}
   * @param frameDuration         duration for each frame
   * @param framePts              presentation time stamp of when this frame is displayed.
   * @param data                  byte buffer comprising data and offset
   * @return queued number of bytes
   *
   */
  // reads between 0 and limit on the buffer
  // synchronization is needed coz, a single buffer is maintained in c-layer
  // for all the send commands
  public synchronized int sendData(long streamHandle, int channelPort, long frameIndex,
                                   int frameType,
                                   long frameDuration, long framePts, ByteBuffer data) {

    return CocoClient.getInstance().getNativeHandler()
        .sendMediaStreamData(streamHandle, channelPort, frameIndex, frameType, frameDuration,
            framePts, data);
  }

  public void stopStream(long streamHandle) {
    CocoClient.getInstance().getNativeHandler().stopMediaStream(streamHandle);
  }

  @Override
  protected <T extends Capability.CommandId> void interceptCommand(Command<T> command) {
    if (CommandId.STREAM_START == command.getCommandId()
        || CommandId.STREAM_STOP == command.getCommandId()) {

      throw new IllegalArgumentException("use individual functions instead of sendResourceCommand");
    }

    super.interceptCommand(command);
  }

  /**
   * This interface has the relevant signatures to multiplex the calls over individual channels.
   */
  public interface MediaStreamListenerMultiplexer extends MediaStreamListener {
    boolean hasCallbackHandlerAssigned(int channelPort);

    MediaStreamListener getCallbackHandler(int channelPort);

    void setChannelHandleCallback(int channelPort, MediaStreamListener callback);

    void removeChannelHandleCallback(int channelPort);

    @Override
    default void onStatusChanged(long streamHandle, int channelPort, Status status) {
      MediaStreamListener callback = getCallbackHandler(channelPort);

      if (null != callback) {
        callback.onStatusChanged(streamHandle, channelPort, status);
      }
    }

    @Override
    default void onDataReceived(long streamHandle, int channelPort, long frameIndex, int frameType,
                                long frameDuration, long framePts, ByteBuffer data) {
      MediaStreamListener callback = getCallbackHandler(channelPort);

      if (null != callback) {
        callback.onDataReceived(streamHandle, channelPort, frameIndex, frameType, frameDuration,
            framePts, data);
      }
    }
  }

  /**
   * This class has the ability to multiplex the calls over individual channels.
   */
  public abstract static class DefaultMediaStreamListenerMultiplexer
      implements MediaStreamListenerMultiplexer {
    private final HashMap<Integer, MediaStreamListener> channelPortCallbackMap;

    public DefaultMediaStreamListenerMultiplexer() {
      channelPortCallbackMap = new HashMap<>(10);
    }

    @Override
    public boolean hasCallbackHandlerAssigned(int channelPort) {
      return channelPortCallbackMap.containsKey(channelPort);
    }

    @Override
    public void removeChannelHandleCallback(int channelPort) {
      channelPortCallbackMap.remove(channelPort);
    }

    @Override
    public MediaStreamListener getCallbackHandler(int channelPort) {
      return channelPortCallbackMap.get(channelPort);
    }

    @Override
    public synchronized void setChannelHandleCallback(int channelPort,
                                                      MediaStreamListener callback) {
      channelPortCallbackMap.put(channelPort, callback);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending start recording commands.
   */
  public static class StartRecording extends Command<CommandId> {
    @SerializedName(Constants.STREAM_IDS)
    public int[] streamIds;
    @SerializedName(Constants.STREAM_DESCRIPTIONS)
    public String[] streamDescriptions;

    public StartRecording() {
      super(CommandId.RECORD_START);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending stop recording commands.
   */
  public static class StopRecording extends Command<CommandId> {
    @SerializedName(Constants.STREAM_IDS)
    public int[] streamIds;

    public StopRecording() {
      super(CommandId.RECORD_STOP);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending configure recording commands.
   */
  public static class ConfigRecording extends Command<CommandId> {
    @SerializedName(Constants.RECORD_MODE)
    public RecordMode recordMode;
    @SerializedName(Constants.SIZE)
    public int size; // in MB
    @SerializedName(Constants.RECORD_TIME)
    public long duration; // in millis
    @SerializedName(Constants.RECORD_COOL_OFF_PERIOD)
    public long coolOffTime; // in millis

    public ConfigRecording() {
      super(CommandId.RECORD_CONFIG);
    }
  }
}
