/*
 * @authors   Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.annotations.SerializedName;

/**
 * This class describes Commands and Attributes for Motion Sensing Capability of a Resource.
 */
public class CapabilityMotionSensing extends Capability {

  public static final CapabilityId ID = CapabilityId.MOTION_SENSING;

  /**
   * enum denoting various possible attributes of the {@link CapabilityMotionSensing}.
   */
  public enum AttributeId implements Capability.AttributeId {
    DETECTED_FLAG,
    SENSING_ENABLED,
    IMAGE_WIDTH,
    IMAGE_HEIGHT,
    ROI_WIDTH,
    ROI_HEIGHT,
    ROI_INDICES_COUNT,
    ROI_INDICES_ARR,
    SENSITIVITY_PCT;

    public static AttributeId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }

    @Override
    public CapabilityId getCapabilityId() {
      return ID;
    }
  }

  /**
   * enum denoting the possible Commands that can be sent for {@link CapabilityMotionSensing}.
   */
  public enum CommandId implements Capability.CommandId {
    ENABLE,
    DISABLE,
    SET_ROI,
    REMOVE_ROI,
    REMOVE_ALL_ROI,
    SET_SENSITIVITY;

    public static CommandId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }
  }

  /**
   * Constructor of the current class.
   *
   * @param id     The unique id Of the capability
   * @param parent The parent Resource of the capability.
   */
  protected CapabilityMotionSensing(int id, Resource parent) {
    super(id, parent);
  }

  /**
   * This function is used to check if a command is supported by the capability or not.
   *
   * @param commandId The Id denoting the command to be sent.
   * @return boolean: If the command is supported then True is returned.
   */
  @Override
  public boolean supports(Capability.CommandId commandId) {
    return (null == commandId || commandId instanceof CommandId) && super.supports(commandId);
  }

  /**
   * A function to create the command from the Json params.
   *
   * @param primitiveCommandId The int form of the commandId
   * @param commandParams      The Json params that can form a command
   * @return The command which is formed
   */
  @Override
  protected Command<? extends Capability.CommandId> extendedCreateCommand(
      int primitiveCommandId, JsonElement commandParams) {
    Command<CommandId> command;
    Gson gson = Command.GSON_BUILDER.create();
    CommandId commandId = CommandId.getEnum(primitiveCommandId);

    switch (commandId) {
      case ENABLE: {
        command = gson.fromJson(commandParams, Enable.class);
        break;
      }

      case DISABLE: {
        command = gson.fromJson(commandParams, Disable.class);
        break;
      }

      case SET_ROI: {
        command = gson.fromJson(commandParams, SetRoi.class);
        break;
      }

      case SET_SENSITIVITY: {
        command = gson.fromJson(commandParams, SetSensitivity.class);
        break;
      }

      case REMOVE_ROI: {
        command = gson.fromJson(commandParams, RemoveRoi.class);
        break;
      }

      case REMOVE_ALL_ROI: {
        command = gson.fromJson(commandParams, RemoveAllRoi.class);
        break;
      }

      default:
        command = new Command<>(commandId);
    }

    return command;
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending set Region of Interest commands.
   */
  public static class SetRoi extends Command<CommandId> {
    public SetRoi() {
      super(CommandId.SET_ROI);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending remove Region of Interest commands.
   */
  public static class RemoveRoi extends Command<CommandId> {
    public RemoveRoi() {
      super(CommandId.REMOVE_ROI);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending remove all Region of Interest commands.
   */
  public static class RemoveAllRoi extends Command<CommandId> {
    public RemoveAllRoi() {
      super(CommandId.REMOVE_ALL_ROI);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending set sensitivity commands.
   */
  public static class SetSensitivity extends Command<CommandId> {

    @SerializedName(Constants.SENSITIVITY)
    public int sensitivity;

    public SetSensitivity() {
      super(CommandId.SET_SENSITIVITY);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending enable motion sensing commands.
   */
  public static class Enable extends Command<CommandId> {

    public Enable() {
      super(CommandId.ENABLE);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending disable motion sensing commands.
   */
  public static class Disable extends Command<CommandId> {

    public Disable() {
      super(CommandId.DISABLE);
    }
  }
}
