/*
 * @authors   Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.Gson;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonParseException;
import com.google.gson.JsonPrimitive;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import com.google.gson.annotations.SerializedName;
import java.lang.reflect.Type;

/**
 * This class describes Commands and Attributes for Motor Control Capability of a Resource.
 */
public class CapabilityMotorControl extends Capability {

  public static final CapabilityId ID = CapabilityId.MOTOR_CONTROL;

  private static final class PanDirectionParser implements JsonDeserializer<PanDirection>,
      JsonSerializer<PanDirection> {

    @Override
    public PanDirection deserialize(JsonElement json, Type typeOfT,
                                    JsonDeserializationContext context) throws
        JsonParseException {
      return PanDirection.getEnum(json.getAsInt());
    }

    @Override
    public JsonElement serialize(PanDirection src, Type typeOfSrc,
                                 JsonSerializationContext context) {
      return new JsonPrimitive(src.getInt());
    }
  }

  private static final class TiltDirectionParser
      implements JsonDeserializer<TiltDirection>, JsonSerializer<TiltDirection> {

    @Override
    public TiltDirection deserialize(JsonElement json, Type typeOfT,
                                     JsonDeserializationContext context) throws JsonParseException {
      return TiltDirection.getEnum(json.getAsInt());
    }

    @Override
    public JsonElement serialize(TiltDirection src, Type typeOfSrc,
                                 JsonSerializationContext context) {
      return new JsonPrimitive(src.getInt());
    }
  }

  /**
   * enum denoting the possible Commands that can be sent for {@link CapabilityMotorControl}.
   */
  public enum CommandId implements Capability.CommandId {
    PAN,
    TILT,
    SET_PRESET,
    CLEAR_PRESET,
    GO_TO_PRESET,
    START_TRACK,
    STOP_TRACK;

    public static CommandId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }
  }

  /**
   * enum denoting various possible attributes of the {@link CapabilityMotorControl}.
   */
  public enum AttributeId implements Capability.AttributeId {
    PAN_DOF,                   // Max degrees from initial
    TILT_DOF,
    SPEED_RPM,                 // motor speed in RPM
    PRESET_POINT_ARR,          // array of preset points
    TRACK_PRESET_POINT_ARR,    // array of track preset points
    TRACK_ACTIVE_FLAG,         // track active flag, true if the track is active
    TRACK_REPEAT_COUNT,        // track repeat count
    TRACK_DELAY_MS;            // delay in milliseconds between two preset points in track

    public static AttributeId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }

    @Override
    public CapabilityId getCapabilityId() {
      return ID;
    }
  }

  /**
   * Super interface for pan and tilt directions.
   */
  public interface Direction {
    Direction LEFT = PanDirection.LEFT;
    Direction RIGHT = PanDirection.RIGHT;
    Direction UP = TiltDirection.UP;
    Direction DOWN = TiltDirection.DOWN;
  }

  /**
   * enum denoting various possible values of pan direction for motor control.
   */
  public enum PanDirection implements Direction {
    LEFT,
    RIGHT;

    public static PanDirection getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }
  }

  /**
   * enum denoting various possible values of tilt direction for motor control.
   */
  public enum TiltDirection implements Direction {
    UP,
    DOWN;

    public static TiltDirection getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }
  }

  static void init() {
    Command.GSON_BUILDER.registerTypeAdapter(PanDirection.class, new PanDirectionParser());
    Command.GSON_BUILDER.registerTypeAdapter(TiltDirection.class, new TiltDirectionParser());
  }

  /**
   * Constructor of the current class.
   *
   * @param id     The unique id Of the capability
   * @param parent The parent Resource of the capability.
   */
  protected CapabilityMotorControl(int id, Resource parent) {
    super(id, parent);
  }

  @Override
  public boolean supports(Capability.CommandId commandId) {
    return (null == commandId || commandId instanceof CommandId) && super.supports(commandId);
  }

  @Override
  protected Command<? extends Capability.CommandId> extendedCreateCommand(
      int primitiveCommandId, JsonElement commandParams) {
    Command<CommandId> command;
    Gson gson = Command.GSON_BUILDER.create();
    CommandId commandId = CommandId.getEnum(primitiveCommandId);

    switch (commandId) {
      case PAN: {
        command = gson.fromJson(commandParams, Pan.class);
        break;
      }

      case TILT: {
        command = gson.fromJson(commandParams, Tilt.class);
        break;
      }

      case SET_PRESET: {
        command = gson.fromJson(commandParams, SetPreset.class);
        break;
      }

      case CLEAR_PRESET: {
        command = gson.fromJson(commandParams, ClearPreset.class);
        break;
      }

      case GO_TO_PRESET: {
        command = gson.fromJson(commandParams, GoToPreset.class);
        break;
      }

      case START_TRACK: {
        command = gson.fromJson(commandParams, StartTrack.class);
        break;
      }

      case STOP_TRACK: {
        command = gson.fromJson(commandParams, StopTrack.class);
        break;
      }

      default: {
        command = new Command<>(commandId);
      }
    }

    return command;
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending pan commands.
   */
  public static class Pan extends Command<CommandId> {

    @SerializedName(Constants.DIRECTION)
    public PanDirection direction;
    @SerializedName(Constants.DEGREE)
    public int degree;                               // 0 to 360 degrees
    @SerializedName(Constants.SPEED)
    public int speed;                                // 0 to 1000 and -1 for default speed

    public Pan() {
      super(CommandId.PAN);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending tilt commands.
   */
  public static class Tilt extends Command<CommandId> {

    @SerializedName(Constants.DIRECTION)
    public TiltDirection direction;
    @SerializedName(Constants.DEGREE)
    public int degree;                               // 0 to 360 degrees
    @SerializedName(Constants.SPEED)
    public int speed;                                // 0 to 1000 and -1 for default speed

    public Tilt() {
      super(CommandId.TILT);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending set preset commands.
   */
  public static class SetPreset extends Command<CommandId> {
    public SetPreset() {
      super(CommandId.SET_PRESET);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending clear preset commands.
   */
  public static class ClearPreset extends Command<CommandId> {
    public ClearPreset() {
      super(CommandId.CLEAR_PRESET);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending go to preset commands.
   */
  public static class GoToPreset extends Command<CommandId> {
    public GoToPreset() {
      super(CommandId.GO_TO_PRESET);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending start track commands.
   */
  public static class StartTrack extends Command<CommandId> {
    public StartTrack() {
      super(CommandId.START_TRACK);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending stop track commands.
   */
  public static class StopTrack extends Command<CommandId> {
    public StopTrack() {
      super(CommandId.STOP_TRACK);
    }
  }
}
