/*
 * @authors   Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.Gson;
import com.google.gson.JsonElement;

/**
 * This class describes Commands and Attributes for OnOff Capability of a Resource.
 */
public class CapabilityOnOff extends Capability {

  public static final CapabilityId ID = CapabilityId.ON_OFF_CONTROL;

  /**
   * enum denoting various possible attributes of the {@link CapabilityOnOff}.
   */
  public enum AttributeId implements Capability.AttributeId {
    ON_FLAG;

    public static AttributeId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }

    @Override
    public CapabilityId getCapabilityId() {
      return ID;
    }
  }

  /**
   * enum denoting the possible Commands that can be sent for {@link CapabilityOnOff}.
   */
  public enum CommandId implements Capability.CommandId {
    ON,
    OFF,
    TOGGLE,
    FETCH_ON_OFF_STATUS;

    public static CommandId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }
  }

  /**
   * Constructor of the current class.
   *
   * @param id     The unique id Of the capability
   *               cocostandard inherently.
   * @param parent The parent Resource of the capability.
   */
  protected CapabilityOnOff(int id, Resource parent) {
    super(id, parent);
  }

  /**
   * This function is used to check if a command is supported by the capability or not.
   *
   * @param commandId The Id denoting the command to be sent.
   * @return boolean: If the command is supported then True is returned.
   */
  @Override
  public boolean supports(Capability.CommandId commandId) {
    return (null == commandId || commandId instanceof CommandId) && super.supports(commandId);
  }

  /**
   * A function to create the command from the Json params.
   *
   * @param primitiveCommandId The int form of the commandId
   * @param commandParams      The Json params that can form a command
   * @return The command which is formed
   */
  @Override
  protected Command<? extends Capability.CommandId> extendedCreateCommand(
      int primitiveCommandId, JsonElement commandParams) {
    Command<CommandId> command;
    Gson gson = Command.GSON_BUILDER.create();
    CommandId commandId = CommandId.getEnum(primitiveCommandId);

    switch (commandId) {
      case ON: {
        command = gson.fromJson(commandParams, On.class);
        break;
      }

      case OFF: {
        command = gson.fromJson(commandParams, Off.class);
        break;
      }

      case TOGGLE: {
        command = gson.fromJson(commandParams, Toggle.class);
        break;
      }

      case FETCH_ON_OFF_STATUS: {
        command = gson.fromJson(commandParams, FetchOnOffStatus.class);
        break;
      }

      default:
        command = new Command<>(commandId);
    }

    return command;
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending On commands.
   */
  public static class On extends Command<CommandId> {
    public On() {
      super(CommandId.ON);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending Off commands.
   */
  public static class Off extends Command<CommandId> {
    public Off() {
      super(CommandId.OFF);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending toggle mode commands.
   */
  public static class Toggle extends Command<CommandId> {
    public Toggle() {
      super(CommandId.TOGGLE);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending fetch on off commands.
   */
  public static class FetchOnOffStatus extends Command<CommandId> {
    public FetchOnOffStatus() {
      super(CommandId.FETCH_ON_OFF_STATUS);
    }
  }
}
