/*
 * @authors   Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.JsonElement;
import com.google.gson.annotations.SerializedName;

/**
 * This class describes Commands and Attributes for Realtime position Capability of a Resource.
 */
public class CapabilityRealTimePosition extends Capability {

  public static final CapabilityId ID = CapabilityId.REAL_TIME_POS;

  /**
   * enum denoting various possible attributes of the {@link CapabilityRealTimePosition}.
   */
  public enum AttributeId implements Capability.AttributeId {
    BEACON_LOC;

    public static Capability.AttributeId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    @Override
    public CapabilityId getCapabilityId() {
      return ID;
    }

    @Override
    public int getInt() {
      return ordinal();
    }
  }

  /**
   * enum denoting the possible Commands that can be sent for {@link CapabilityRealTimePosition}.
   */
  public enum CommandId implements Capability.CommandId {
    // Note : Update extendedCreateCommand method whenever commandId is added
    ;

    public static CapabilityRealTimePosition.CommandId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    @Override
    public int getInt() {
      return ordinal();
    }
  }

  /**
   * Constructor of the current class.
   *
   * @param id     The unique id Of the capability
   * @param parent The parent Resource of the capability.
   */
  protected CapabilityRealTimePosition(int id, Resource parent) {
    super(id, parent);
  }

  @Override
  public boolean supports(Capability.CommandId commandId) {
    return (null == commandId || commandId instanceof CommandId) && super.supports(commandId);
  }

  @Override
  protected Command<? extends Capability.CommandId>
      extendedCreateCommand(int cmdId, JsonElement commandParams) {
    CommandId commandId = CommandId.getEnum(cmdId);
    return new Command<>(commandId);
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending beacon location commands.
   */
  public static class BeaconLocation extends CapabilityStationaryPosition.Position {
    @SerializedName(Constants.BEACON_ID)
    public String beaconIdStr;
    @SerializedName(Constants.DEVICE_RES_NAME)
    public String deviceResourceNameStr;
    @SerializedName(Constants.RADIUS_OF_BEACON)
    public double radiusOfBeacon;
    @SerializedName(Constants.ERROR_PRECISION)
    public double errorPrecision;
    @SerializedName(Constants.RSSI)
    public int rssi;
    @SerializedName(Constants.MEASURED_POWER)
    public int measuredPower;
  }
}
