/*
 * @authors   Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.Gson;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonPrimitive;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import com.google.gson.annotations.SerializedName;
import com.google.gson.reflect.TypeToken;
import java.lang.reflect.Type;
import java.util.Collections;

/**
 * This class describes Commands and Attributes for Remote Control Capability of a Resource.
 */
public class CapabilityRemoteControl extends Capability {

  private static final String TAG = "CapabilityRemoteControl";
  public static final CapabilityId ID = CapabilityId.REMOTE_CONTROL;

  /**
   * possible values for {@link AttributeId#CATEGORY_ID_ARR}.
   */
  public static final class Category {

    public static final int MIN = -1;
    public static final int AC = 0;
    public static final int TV = 1;

    private final int rawValue;

    private Category(int rawValue) {
      this.rawValue = rawValue;
    }

    public static Category ac() {
      return new Category(AC);
    }

    public static Category tv() {
      return new Category(TV);
    }

    public static Category none() {
      return new Category(MIN);
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) {
        return true;
      }

      if (o == null || getClass() != o.getClass()) {
        return false;
      }

      Category category = (Category) o;
      return rawValue == category.rawValue;
    }

    @Override
    public int hashCode() {
      return rawValue;
    }

    @Override
    public String toString() {
      return "Category{"
          + "rawValue=" + rawValue
          + '}';
    }
  }

  /**
   * possible values for {@link AttributeId#SUPPORTED_PROGRAM_MODES_ARR}.
   */
  public static final class ProgramMode {

    public static final int PRESET_MODE = 0;
    public static final int LEARN_MODE = 1;
    public static final int SEARCH_MODE = 2;

    private final int mode;

    private ProgramMode(int mode) {
      this.mode = mode;
    }

    public static ProgramMode preset() {
      return new ProgramMode(PRESET_MODE);
    }

    public static ProgramMode learn() {
      return new ProgramMode(LEARN_MODE);
    }

    public static ProgramMode search() {
      return new ProgramMode(SEARCH_MODE);
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) {
        return true;
      }

      if (o == null || getClass() != o.getClass()) {
        return false;
      }

      ProgramMode that = (ProgramMode) o;
      return mode == that.mode;
    }

    @Override
    public int hashCode() {
      return mode;
    }
  }

  /**
   * enum denoting various possible attributes of the {@link CapabilityRemoteControl}.
   */
  public enum AttributeId implements Capability.AttributeId {
    SUPPORTED_PROGRAM_MODES_ARR,
    MAX_APPLIANCE_COUNT,
    CATEGORY_ID_ARR,
    APPLIANCE_LIST,
    MAX_BUTTON_COUNT;

    public static AttributeId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }

    @Override
    public CapabilityId getCapabilityId() {
      return ID;
    }
  }

  /**
   * enum denoting the possible Commands that can be sent for {@link CapabilityRemoteControl}.
   */
  public enum CommandId implements Capability.CommandId {
    TEST_COMMAND,
    ADD_APPLIANCE,
    SET_APPLIANCE_PAIRING_CODE,
    SET_APPLIANCE_NAME,
    REMOVE_APPLIANCE,
    EXECUTE_COMMAND,
    SET_SEARCH_MODE,
    ENTER_LEARN_MODE,
    CANCEL_LEARN_MODE,
    RENAME_BUTTON,
    TEST_LEARNED_COMMAND,
    SAVE_LEARNED_COMMAND,
    EXECUTE_LEARNED_COMMAND,
    REMOVE_BUTTON;

    public static CommandId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }
  }

  /**
   * Constructor of the current class.
   *
   * @param id     The unique id Of the capability
   * @param parent The parent Resource of the capability.
   */
  protected CapabilityRemoteControl(int id, Resource parent) {
    super(id, parent);
  }

  static void init() {
    Command.GSON_BUILDER.registerTypeAdapter(Category.class, new CategoryParser());
    Command.GSON_BUILDER.registerTypeAdapter(ProgramMode.class, new ProgramModeParser());
    Command.GSON_BUILDER.registerTypeAdapter(ExecuteCommand.class, new ExecuteCommandParser());
  }

  /**
   * This function is used to check if a command is supported by the capability or not.
   *
   * @param commandId The Id denoting the command to be sent.
   * @return boolean: If the command is supported then True is returned.
   */
  @Override
  public boolean supports(Capability.CommandId commandId) {
    return (null == commandId || commandId instanceof CommandId) && super.supports(commandId);
  }

  /**
   * A function to create the command from the Json params.
   *
   * @param primitiveCommandId The int form of the commandId
   * @param commandParams      The Json params that can form a command
   * @return The command which is formed
   */
  @Override
  protected Command<? extends Capability.CommandId> extendedCreateCommand(
      int primitiveCommandId, JsonElement commandParams) {
    Command<? extends Capability.CommandId> command;
    Gson gson = Command.GSON_BUILDER.create();
    CommandId commandId = CommandId.getEnum(primitiveCommandId);

    switch (commandId) {
      case TEST_COMMAND: {
        TypeToken<TestCommand<Command<Capability.CommandId>>> typeToken =
            new TypeToken<TestCommand<Command<Capability.CommandId>>>() {
            };
        command = gson.fromJson(commandParams, typeToken.getType());
        Log.d(TAG, "extendedCreateCommand: creating test command with type: " + typeToken);
        break;
      }

      case EXECUTE_COMMAND: {
        TypeToken<ExecuteCommand<Command<Capability.CommandId>>> typeToken =
            new TypeToken<ExecuteCommand<Command<Capability.CommandId>>>() {
            };
        command = gson.fromJson(commandParams, typeToken.getType());
        Log.d(TAG, "extendedCreateCommand: creating execute command with type: " + typeToken);
        break;
      }

      case ADD_APPLIANCE: {
        command = gson.fromJson(commandParams, AddAppliance.class);
        break;
      }

      case REMOVE_APPLIANCE: {
        command = gson.fromJson(commandParams, RemoveAppliance.class);
        break;
      }

      case SET_APPLIANCE_NAME: {
        command = gson.fromJson(commandParams, SetApplianceName.class);
        break;
      }

      case SET_APPLIANCE_PAIRING_CODE: {
        command = gson.fromJson(commandParams, SetAppliancePairingCode.class);
        break;
      }

      case SET_SEARCH_MODE: {
        command = gson.fromJson(commandParams, SetSearchMode.class);
        break;
      }

      case ENTER_LEARN_MODE: {
        command = gson.fromJson(commandParams, EnterLearnMode.class);
        break;
      }

      case CANCEL_LEARN_MODE: {
        command = gson.fromJson(commandParams, CancelLearnMode.class);
        break;
      }

      case RENAME_BUTTON: {
        command = gson.fromJson(commandParams, RenameButton.class);
        break;
      }

      case TEST_LEARNED_COMMAND: {
        command = gson.fromJson(commandParams, TestLearnedCommand.class);
        break;
      }

      case SAVE_LEARNED_COMMAND: {
        command = gson.fromJson(commandParams, SaveLearnedCommand.class);
        break;
      }

      case EXECUTE_LEARNED_COMMAND: {
        command = gson.fromJson(commandParams, ExecuteLearnedCommand.class);
        break;
      }

      case REMOVE_BUTTON: {
        command = gson.fromJson(commandParams, RemoveButton.class);
        break;
      }

      default: {
        command = new Command<>(commandId);
      }
    }

    return command;
  }

  /**
   * create response for the command.
   *
   * @param commandId    The command ID which has to be responded
   * @param jsonResponse The response body
   * @return The created response
   */
  @Override
  protected <T extends CommandIdInterface> CommandResponse.Args<T> createCommandResponseArgs(
      T commandId, JsonElement jsonResponse) {

    if (CommandId.SET_SEARCH_MODE == commandId) {

      Gson gson = Command.GSON_BUILDER.create();

      SetSearchModeResponse response = gson.fromJson(jsonResponse, SetSearchModeResponse.class);

      return CommandResponse.castArgs(response);
    }

    if (CommandId.ADD_APPLIANCE == commandId) {
      Gson gson = Command.GSON_BUILDER.create();

      AddApplianceResponse response = gson.fromJson(jsonResponse, AddApplianceResponse.class);

      return CommandResponse.castArgs(response);
    }

    return super.createCommandResponseArgs(commandId, jsonResponse);
  }

  /**
   * A function to make an info request for search program mode.
   *
   * @param searchCount      Search count for info request
   * @param offsetIndex      offset provided for search request
   * @param st               Search type (Brand)
   * @param so               Sort order (Ascending/Descending)
   * @param soT              Sort type (Ex: sort by time of creation)
   * @param cat              Category of the command (AC/TV)
   * @param brandSearchQuery a string param for brand search criteria
   * @param cbs              Listener that receives status of info request made
   */
  public void searchBrands(int searchCount, int offsetIndex, Parameter.SearchType st,
                           InfoRequest.SortOrder so, InfoRequest.SortType soT,
                           Category cat, String brandSearchQuery,
                           Device.InfoRequestStatusListener cbs) {

    Resource r = getParent();
    Device d = r.getParent();
    Network n = d.getParent();

    InfoRequest req = new InfoRequest(null,
        Collections.singletonList(
            new Parameter<>(
                Parameter.Key.REMOTE_SEARCH,
                new InfoRequest.Search(searchCount, offsetIndex, st,
                    new Identifier.SourceIdentifier(n.getId(), d.getId(), r.getId()), soT, so,
                    new InfoRequest.BrandSearchCriteria(cat, brandSearchQuery))
            )
        )
    );

    d.sendInfoRequest(req, cbs);
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending test command to IR Blaster.
   */
  public static class TestCommand<T extends Command<? extends Capability.CommandId>>
      extends Command<CommandId> {
    @SerializedName(Constants.CATEGORY_ID)
    public final Category categoryId;
    @SerializedName(Constants.BRAND_ID)
    public final int brandId;
    @SerializedName(Constants.PAIRING_CODE)
    public final int pairingCode;

    @SerializedName(Constants.CAPABILITY_ID)
    public final CapabilityId capabilityId;
    @SerializedName(Constants.REMOTE_COMMAND_ID)
    private final int commandId;
    @SerializedName(Constants.REMOTE_COMMAND_PARAMS)
    public final T command;

    /**
     * The command which will be blasted by IR Blaster.
     *
     * @param category     Category of the command (AC/TV)
     * @param brandId      Brand id of the recipient device
     * @param pairingCode  Pairing code generated during pairing
     * @param capabilityId CapabilityID of the recipient device
     * @param command      Command which has to be blasted
     */
    public TestCommand(Category category, int brandId, int pairingCode, CapabilityId capabilityId,
                       T command) {
      super(CommandId.TEST_COMMAND);

      this.categoryId = category;
      this.brandId = brandId;
      this.pairingCode = pairingCode;
      this.capabilityId = capabilityId;
      this.commandId = command.getCommandId().getInt();
      this.command = command;
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending execute commands.
   */
  public static class ExecuteCommand<T extends Command<? extends Capability.CommandId>>
      extends Command<CommandId> {

    @SerializedName(Constants.APPLIANCE_ID)
    public final String applianceId;
    @SerializedName(Constants.CAPABILITY_ID)
    public final CapabilityId capabilityId;
    @SerializedName(Constants.REMOTE_COMMAND_ID)
    private final int commandId;
    @SerializedName(Constants.REMOTE_COMMAND_PARAMS)
    public final T command;

    /**
     * Constructor for Execute command.
     *
     * @param applianceId  String id of the appliance which executes command
     * @param capabilityId CapabilityID of the recipient device
     * @param command      Command which has to be executed
     */
    public ExecuteCommand(String applianceId, CapabilityId capabilityId, T command) {
      super(CommandId.EXECUTE_COMMAND);

      this.applianceId = applianceId;
      this.capabilityId = capabilityId;
      this.command = command;
      this.commandId = command.getCommandId().getInt();
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending add appliance commands.
   */
  public static class AddAppliance extends Command<CommandId> {
    public static final int LEARN_MODE_PAIRING_CODE = 0XFFFF;
    public static final int LEARN_MODE_BRAND_ID = 0xFF;

    @SerializedName(Constants.BRAND_ID)
    public final int brandId;
    @SerializedName(Constants.PAIRING_CODE)
    public final int pairingCode;
    @SerializedName(Constants.APPLIANCE_NAME)
    public final String applianceName;
    @SerializedName(Constants.CATEGORY_ID)
    public final Category category;
    @SerializedName(Constants.PROGRAM_MODE)
    public final ProgramMode mode;

    /**
     * Constructor for AddAppliance command.
     *
     * @param brandId       integer id of brand
     * @param pairingCode   Pairing code generated during pairing
     * @param applianceName name of appliance to be added
     * @param category      Category of the command (AC/TV) {@link CapabilityRemoteControl.Category}
     * @param mode          programming mode applicable {@link CapabilityRemoteControl.Category}
     */
    public AddAppliance(int brandId, int pairingCode, String applianceName, Category category,
                        ProgramMode mode) {
      super(CommandId.ADD_APPLIANCE);
      this.brandId = brandId;
      this.pairingCode = pairingCode;
      this.applianceName = applianceName;
      this.category = category;
      this.mode = mode;
    }

    /**
     * Convenience function for getting {@link AddAppliance} for preset mode.
     *
     * @param applianceName Name of the appliance
     * @param category      {@link Category} of the appliance
     * @param brandId       The brand id from {@link InfoResponse.BrandSearchResults}
     * @param pairingCode   The pairing code from {@link InfoResponse.RemotePairingCode}
     * @return AddAppliance command which can be used
     */
    public static AddAppliance forPresetMode(String applianceName, Category category,
                                             int brandId, int pairingCode) {
      return new AddAppliance(brandId, pairingCode, applianceName, category, ProgramMode.preset());
    }

    /**
     * Convenience function for getting {@link AddAppliance} for search mode.
     *
     * @param applianceName Name of the appliance
     * @param category      {@link Category} of the appliance
     * @param brandId       The brand id from {@link InfoResponse.BrandSearchResults}
     * @param pairingCode   The pairing code from {@link SetSearchModeResponse}
     * @return AddAppliance command which can be used
     */
    public static AddAppliance forSearchMode(String applianceName, Category category,
                                             int brandId, int pairingCode) {
      return new AddAppliance(brandId, pairingCode, applianceName, category, ProgramMode.search());
    }

    /**
     * Convenience function for getting {@link AddAppliance} for learn mode.
     *
     * @param applianceName Name of the appliance
     * @param category      {@link Category} of the appliance
     * @return AddAppliance command which can be used
     */
    public static AddAppliance forLearnMode(String applianceName, Category category) {
      return new AddAppliance(LEARN_MODE_BRAND_ID, LEARN_MODE_PAIRING_CODE, applianceName, category,
          ProgramMode.learn());
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while setting pairing code for appliances.
   */
  public static class SetAppliancePairingCode extends Command<CommandId> {

    @SerializedName(Constants.APPLIANCE_ID)
    public final String applianceId;
    @SerializedName(Constants.BRAND_ID)
    public final int brandId;
    @SerializedName(Constants.PAIRING_CODE)
    public final int pairingCode;

    /**
     * Constructor for SetAppliancePairingCode command.
     *
     * @param applianceId String id of the appliance
     * @param brandId Brand id of the device
     * @param pairingCode Pairing code generated during pairing
     */
    public SetAppliancePairingCode(String applianceId, int brandId, int pairingCode) {
      super(CommandId.SET_APPLIANCE_PAIRING_CODE);

      this.applianceId = applianceId;
      this.brandId = brandId;
      this.pairingCode = pairingCode;
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending set name for appliances command.
   */
  public static class SetApplianceName extends Command<CommandId> {

    @SerializedName(Constants.APPLIANCE_ID)
    public final String applianceId;
    @SerializedName(Constants.APPLIANCE_NAME)
    public final String applianceName;

    /**
     * Constructor for SetApplianceName command.
     *
     * @param applianceId String id of appliance
     * @param applianceName name that is to be set for the applianceId
     */
    public SetApplianceName(String applianceId, String applianceName) {
      super(CommandId.SET_APPLIANCE_NAME);

      this.applianceId = applianceId;
      this.applianceName = applianceName;
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending remove appliances command.
   */
  public static class RemoveAppliance extends Command<CommandId> {

    @SerializedName(Constants.APPLIANCE_ID)
    public final String applianceId;

    /**
     * Constructor for RemoveAppliance command.
     *
     * @param applianceId String id of appliance to be removed
     */
    public RemoveAppliance(String applianceId) {
      super(CommandId.REMOVE_APPLIANCE);

      this.applianceId = applianceId;
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending set search mode command.
   */
  public static class SetSearchMode extends Command<CommandId> {
    public SetSearchMode() {
      super(CommandId.SET_SEARCH_MODE);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending enter learn mode command.
   */
  public static class EnterLearnMode extends Command<CommandId> {
    @SerializedName(Constants.APPLIANCE_ID)
    public final String applianceId;
    @SerializedName(Constants.BUTTON_NAME)
    public final String buttonName;

    /**
     * Constructor for enter learn mode command.
     *
     * @param applianceId String id of appliance
     * @param buttonName button name for learn mode
     */
    public EnterLearnMode(String applianceId, String buttonName) {
      super(CommandId.ENTER_LEARN_MODE);

      this.applianceId = applianceId;
      this.buttonName = buttonName;
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending cancel learn mode command.
   */
  public static class CancelLearnMode extends Command<CommandId> {
    public CancelLearnMode() {
      super(CommandId.CANCEL_LEARN_MODE);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending execute learnt command.
   */
  public static class ExecuteLearnedCommand extends Command<CommandId> {
    @SerializedName(Constants.APPLIANCE_ID)
    public final String applianceId;
    @SerializedName(Constants.BUTTON_ID)
    public final String buttonId;

    /**
     * Constructor for ExecuteLearnedCommand.
     *
     * @param applianceId String id of appliance
     * @param buttonId button id corresponding to learn mode
     */
    public ExecuteLearnedCommand(String applianceId, String buttonId) {
      super(CommandId.EXECUTE_LEARNED_COMMAND);
      this.applianceId = applianceId;
      this.buttonId = buttonId;
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending rename button command.
   */
  public static class RenameButton extends Command<CommandId> {
    @SerializedName(Constants.APPLIANCE_ID)
    public final String applianceId;
    @SerializedName(Constants.BUTTON_ID)
    public final String buttonId;
    @SerializedName(Constants.BUTTON_NAME)
    public final String buttonName;

    /**
     * Constructor for RenameButton command.
     *
     * @param applianceId String id of appliance
     * @param buttonId button id corresponding to learn mode
     * @param buttonName button name for learn mode
     */
    public RenameButton(String applianceId, String buttonId, String buttonName) {
      super(CommandId.RENAME_BUTTON);
      this.applianceId = applianceId;
      this.buttonId = buttonId;
      this.buttonName = buttonName;
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending remove button command.
   */
  public static class RemoveButton extends Command<CommandId> {
    @SerializedName(Constants.APPLIANCE_ID)
    public final String applianceId;
    @SerializedName(Constants.BUTTON_ID)
    public final String buttonId;

    /**
     * Constructor for RemoveButton command.
     *
     * @param applianceId String id of appliance
     * @param buttonId button id corresponding to learn mode that is to be removed
     */
    public RemoveButton(String applianceId, String buttonId) {
      super(CommandId.REMOVE_BUTTON);
      this.applianceId = applianceId;
      this.buttonId = buttonId;
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending test learned command.
   */
  public static class TestLearnedCommand extends Command<CommandId> {
    public TestLearnedCommand() {
      super(CommandId.TEST_LEARNED_COMMAND);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending save learned command.
   */
  public static class SaveLearnedCommand extends Command<CommandId> {
    public SaveLearnedCommand() {
      super(CommandId.SAVE_LEARNED_COMMAND);
    }
  }

  /**
   * Class useful in binding json response to object.
   */
  public static class AddApplianceResponse extends CommandResponse.Args<CommandId> {
    @SerializedName(Constants.APPLIANCE_ID)
    public String applianceId;
  }

  /**
   * Class useful in binding json response to object.
   *
   * @see InfoResponse.RemotePairingCode
   */
  public static class SetSearchModeResponse extends CommandResponse.Args<CommandId> {
    @SerializedName(Constants.PAIRING_CODES)
    public int[] pairingCodes;
  }

  /**
   * Json structure for elements of attribute value of {@link AttributeId#APPLIANCE_LIST}.
   */
  public static final class RemoteControlButtonInfo {
    @SerializedName(Constants.BUTTON_ID)
    private final String buttonId;
    @SerializedName(Constants.BUTTON_NAME)
    private final String settings;

    public RemoteControlButtonInfo(String buttonId, String settings) {
      this.buttonId = buttonId;
      this.settings = settings;
    }

    public String getButtonId() {
      return buttonId;
    }

    public String getSettings() {
      return settings;
    }
  }

  // NOTE: not implementing DeSerializer s
  private static final class CategoryParser implements JsonSerializer<Category>,
      JsonDeserializer<Category> {

    @Override
    public JsonElement serialize(Category category, Type type,
                                 JsonSerializationContext jsonSerializationContext) {
      return new JsonPrimitive(category.rawValue);
    }

    @Override
    public Category deserialize(JsonElement json, Type type,
                                JsonDeserializationContext jsonDeserializationContext) throws
        JsonParseException {
      if (!json.isJsonPrimitive() || !json.getAsJsonPrimitive().isNumber()) {
        throw new JsonParseException("invalid type: " + json);
      }

      return new Category(json.getAsInt());
    }
  }

  private static final class ProgramModeParser
      implements JsonSerializer<ProgramMode>, JsonDeserializer<ProgramMode> {

    @Override
    public JsonElement serialize(ProgramMode programMode, Type type,
                                 JsonSerializationContext jsonSerializationContext) {
      return new JsonPrimitive(programMode.mode);
    }

    @Override
    public ProgramMode deserialize(JsonElement json, Type type,
                                   JsonDeserializationContext jsonDeserializationContext)
        throws JsonParseException {
      if (!json.isJsonPrimitive() || !json.getAsJsonPrimitive().isNumber()) {
        throw new JsonParseException("invalid type: " + json);
      }

      return new ProgramMode(json.getAsInt());
    }
  }

  private static final class ExecuteCommandParser implements JsonDeserializer<ExecuteCommand<?>> {

    @Override
    public ExecuteCommand<?> deserialize(JsonElement jsonElement, Type type,
                                         JsonDeserializationContext dsc) throws JsonParseException {
      JsonObject jo = jsonElement.getAsJsonObject();

      String applianceId = jo.get(Constants.APPLIANCE_ID).getAsString();
      int cmdId = jo.get(Constants.REMOTE_COMMAND_ID).getAsInt();
      CapabilityId capId = dsc.deserialize(jo.get(Constants.CAPABILITY_ID), CapabilityId.class);

      // TODO: create command without creating capability
      Command<? extends Capability.CommandId> command =
          Factory.createCapability(capId.getInt(), null)
              .createCommand(cmdId, jo.get(Constants.REMOTE_COMMAND_PARAMS));

      return new ExecuteCommand<>(applianceId, capId, command);
    }
  }
}
