/*
 * @authors   Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import java.util.Objects;

/**
 * This class describes Commands and Attributes for Snapshot Capability of a Resource.
 */
public class CapabilitySnapshot extends Capability {

  public static final CapabilityId ID = CapabilityId.SNAPSHOT;

  /**
   * enum denoting various possible attributes of the {@link CapabilitySnapshot}.
   */
  public enum AttributeId implements Capability.AttributeId {
    IMG_SUPPORT_RESOLUTION_HEIGHT_ARR,
    IMG_SUPPORT_RESOLUTION_WIDTH_ARR,
    IMG_DEFAULT_RESOLUTION_HEIGHT,
    IMG_DEFAULT_RESOLUTION_WIDTH;

    public static AttributeId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }

    @Override
    public CapabilityId getCapabilityId() {
      return ID;
    }
  }

  /**
   * enum denoting the possible Commands that can be sent for {@link CapabilitySnapshot}.
   */
  public enum CommandId implements Capability.CommandId {
    GRAB_IMAGE;

    public static CommandId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }
  }

  /**
   * possible values for status of snapshot.
   */
  public static final class SnapshotStatus {
    public static final int SUCCESS = 1;
    public static final int FAILURE = 2;
    public static final int TIMEOUT = 3;

    private SnapshotStatus() {
      throw new RuntimeException();
    }
  }

  /**
   * Listener provides callback to listen for snapshot events.
   */
  public interface SnapshotListener extends Listener {
    // status corresponds to value from SnapshotStatus
    void onSnapshotCaptured(String filePath, int status);
  }

  /**
   * Constructor of the current class.
   *
   * @param id     The unique id Of the capability
   * @param parent The parent Resource of the capability.
   */
  protected CapabilitySnapshot(int id, Resource parent) {
    super(id, parent);
  }

  @Override
  public boolean supports(Capability.CommandId commandId) {
    return (null == commandId || commandId instanceof CommandId) && super.supports(commandId);
  }

  @Override
  protected Command<? extends Capability.CommandId> extendedCreateCommand(
      int primitiveCommandId, JsonElement commandParams) {
    Command<CommandId> command;
    Gson gson = Command.GSON_BUILDER.create();
    CommandId commandId = CommandId.getEnum(primitiveCommandId);

    switch (commandId) {
      case GRAB_IMAGE: {
        command = gson.fromJson(commandParams, GrabImage.class);
        break;
      }

      default:
        command = new Command<>(commandId);
    }

    return command;
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending Grab Image commands.
   */
  public static class GrabImage extends Command<CommandId> {
    public GrabImage() {
      super(CommandId.GRAB_IMAGE);
    }
  }

  /**
   * A function to capture snapshot from the resource.
   *
   * @param downloadPath The path to the file which has to downloaded.
   *                     NOTE: This should be the absolute path
   * @param heightPx     The height of the image which has to be downloaded
   * @param widthPx      The width of the image which has to be downloaded
   * @param timeout      The timeout for this command
   * @param listener     The listener which will be triggered after the callback
   */
  public void captureSnapshot(String downloadPath, int heightPx, int widthPx, long timeout,
                              SnapshotListener listener) {
    Objects.requireNonNull(listener);
    Objects.requireNonNull(downloadPath);

    CocoClient.getInstance().getNativeHandler()
        .captureSnapshot(this, downloadPath, heightPx, widthPx, timeout, listener);
  }
}
