/*
 * @authors   Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.JsonElement;
import com.google.gson.annotations.SerializedName;

/**
 * This class describes Commands and Attributes for Stationary Position Capability of a Resource.
 */
public class CapabilityStationaryPosition extends Capability {

  public static final CapabilityId ID = CapabilityId.STATIONARY_POS;

  /**
   * enum denoting various possible attributes of the {@link CapabilityStationaryPosition}.
   */
  public enum AttributeId implements Capability.AttributeId {
    BEACON_LOC,
    GATEWAY_LOC;

    public static CapabilityStationaryPosition.AttributeId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    @Override
    public int getInt() {
      return this.ordinal();
    }

    @Override
    public Capability.CapabilityId getCapabilityId() {
      return ID;
    }
  }

  /**
   * enum denoting the possible Commands that can be sent for {@link CapabilityStationaryPosition}.
   */
  public enum CommandId implements Capability.CommandId {
    SET_DEVICE_LOC;

    public static CapabilityStationaryPosition.CommandId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    @Override
    public int getInt() {
      return ordinal();
    }
  }

  /**
   * Constructor of the current class.
   *
   * @param id     The unique id Of the capability
   * @param parent The parent Resource of the capability.
   */
  protected CapabilityStationaryPosition(int id, Resource parent) {
    super(id, parent);
  }

  @Override
  public boolean supports(Capability.CommandId commandId) {
    return (null == commandId || commandId instanceof CommandId) && super.supports(commandId);
  }

  @Override
  protected Command<? extends Capability.CommandId> extendedCreateCommand(
      int primitiveCommandId, JsonElement commandParams) {
    Command<CommandId> command;
    CommandId commandId = CommandId.getEnum(primitiveCommandId);

    switch (commandId) {
      case SET_DEVICE_LOC: {
        command = Command.GSON_BUILDER.create().fromJson(commandParams, SetDeviceLocation.class);
        break;
      }

      default: {
        command = new Command<>(commandId);
      }
    }

    return command;
  }

  /**
   * Class provides important fields for its subclasses {@link DeviceLocation}
   * and {@link BeaconLocation}.
   */
  public static class Position {
    @SerializedName(Constants.LOCATION_ID)
    public String locationId;
    @SerializedName(Constants.LOT_ID)
    public String lotId;
    @SerializedName(Constants.X_COORDINATE)
    public double xcoordinate;
    @SerializedName(Constants.Y_COORDINATE)
    public double ycoordinate;
  }

  /**
   * Class specifies fields necessary for DeviceLocation object.
   */
  public static class DeviceLocation extends Position {
    @SerializedName(Constants.ENVIRONMENTAL_FACTOR)
    public double environmentalFactor;
  }

  /**
   * Class specifies fields necessary for BeaconLocation object.
   */
  public static class BeaconLocation extends Position {
    @SerializedName(Constants.BEACON_ID)
    public String beaconIdStr;
    @SerializedName(Constants.DEVICE_RES_NAME)
    public String deviceResourceNameStr;
    @SerializedName(Constants.AVG_RADIUS_BEACON)
    public double avgRadiusOfBeacon;
    @SerializedName(Constants.ERROR_PRECISION)
    public double errorPrecision;
    @SerializedName(Constants.AVG_TIME_DUR_BEACON)
    public int avgTimeDurOfBeacon;
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending set device location commands.
   */
  public static class SetDeviceLocation extends Command<CommandId> {
    @SerializedName(Constants.LOCATION_ID)
    public final String locationId;
    @SerializedName(Constants.LOT_ID)
    public final String lotId;
    @SerializedName(Constants.X_COORDINATE)
    public final double xcoordinate;
    @SerializedName(Constants.Y_COORDINATE)
    public final double ycoordinate;
    @SerializedName(Constants.ENVIRONMENTAL_FACTOR)
    public final double environmentalFactor;

    /**
     * Constructor for SetDeviceLocation command.
     *
     * @param locationId string id of location to be set
     * @param lotId lot identification string.
     * @param x abscissa of the location to be set
     * @param y ordinate of the location to be set
     * @param environmentalFactor environmental factors
     */
    public SetDeviceLocation(String locationId, String lotId, double x, double y,
                             double environmentalFactor) {
      super(CommandId.SET_DEVICE_LOC);
      this.locationId = locationId;
      this.lotId = lotId;
      this.xcoordinate = x;
      this.ycoordinate = y;
      this.environmentalFactor = environmentalFactor;
    }
  }
}
