/*
 * @authors   Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.Gson;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonParseException;
import com.google.gson.JsonPrimitive;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import java.lang.reflect.Type;

/**
 * This class describes Commands and Attributes for Storage Control Capability of a Resource.
 */
public class CapabilityStorageControl extends Capability {

  public static final CapabilityId ID = CapabilityId.STORAGE_CONTROL;

  /**
   * enum denoting various possible attributes of the {@link CapabilityStorageControl}.
   */
  public enum AttributeId implements Capability.AttributeId {
    CONTENT_COUNT,
    CAPACITY_MB,                     // Storage Capacity of storage node in Megabytes
    AVAILABLE_MB,                    // Storage currently available, in Megabytes
    SUPPORTED_TYPES_ARR,             /* Supported content types,
                                        array of coco_std_storage_content_type_t */
    RECORDING_TIME_CONFIG,           // configuration for max time allowed for recording
    RECORDING_SIZE_CONFIG,           // configuration for max size allowed for recording
    LAST_UPDATED_CONTENT_METADATA,   // last updated content metadata
    COOL_OFF_PERIOD,                 // Cool off period between 2 consecutive recordings
    RECORDING_STARTED,               /* This attribute is to be updated
                                        whenever a recording is started */
    SUPPORTED_STREAM_PROTOCOLS;      /* Supported stream protocol types,
                                        array of coco_std_stream_protocol_type_t */

    public static AttributeId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }

    @Override
    public CapabilityId getCapabilityId() {
      return ID;
    }
  }

  /**
   * enum denoting the possible Commands that can be sent for {@link CapabilityStorageControl}.
   */
  public enum CommandId implements Capability.CommandId {
    UPLOAD,
    DOWNLOAD;

    public static CommandId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }
  }

  /**
   * possible values for stream protocols.
   */
  public enum StreamProtocol {
    DASH,
    HLS,
    HDS,
    SMOOTH;

    public static StreamProtocol getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }
  }

  /**
   * possible values for playback errors.
   */
  public enum PlaybackError {
    NONE,
    TIMEOUT,
    NOT_FOUND,
    PROTOCOL_NOT_SUPPORTED,
    INTERNAL;

    public static PlaybackError getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }
  }

  static void init() {
    Command.GSON_BUILDER.registerTypeAdapter(StreamProtocol.class, new StreamProtocolParser());
    Command.GSON_BUILDER.registerTypeAdapter(PlaybackError.class, new PlaybackErrorParser());
  }

  /**
   * Constructor of the current class.
   *
   * @param id     The unique id Of the capability
   * @param parent The parent Resource of the capability.
   */
  protected CapabilityStorageControl(int id, Resource parent) {
    super(id, parent);
  }

  /**
   * This function is used to check if a command is supported by the capability or not.
   *
   * @param commandId The Id denoting the command to be sent.
   * @return boolean: If the command is supported then True is returned.
   */
  @Override
  public boolean supports(Capability.CommandId commandId) {
    return (null == commandId || commandId instanceof CommandId) && super.supports(commandId);
  }

  /**
   * A function to create the command from the Json params.
   *
   * @param primitiveCommandId The int form of the commandId
   * @param commandParams      The Json params that can form a command
   * @return The command which is formed
   */
  @Override
  protected Command<? extends Capability.CommandId> extendedCreateCommand(
      int primitiveCommandId, JsonElement commandParams) {
    Command<CommandId> command;
    Gson gson = Command.GSON_BUILDER.create();
    CommandId commandId = CommandId.getEnum(primitiveCommandId);

    switch (commandId) {
      case UPLOAD: {
        command = gson.fromJson(commandParams, Upload.class);
        break;
      }

      case DOWNLOAD: {
        command = gson.fromJson(commandParams, Download.class);
        break;
      }

      default:
        command = new Command<>(commandId);
    }

    return command;
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending download commands.
   */
  public static class Download extends Command<CommandId> {
    public Download() {
      super(CommandId.DOWNLOAD);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending upload commands.
   */
  public static class Upload extends Command<CommandId> {
    public Upload() {
      super(CommandId.UPLOAD);
    }
  }

  private static final class StreamProtocolParser implements JsonSerializer<StreamProtocol>,
      JsonDeserializer<StreamProtocol> {

    @Override
    public StreamProtocol deserialize(JsonElement json, Type typeOfT,
                                      JsonDeserializationContext context) throws
        JsonParseException {
      return StreamProtocol.getEnum(json.getAsInt());
    }

    @Override
    public JsonElement serialize(StreamProtocol src, Type typeOfSrc,
                                 JsonSerializationContext context) {
      return new JsonPrimitive(src.getInt());
    }
  }

  private static final class PlaybackErrorParser
      implements JsonSerializer<PlaybackError>, JsonDeserializer<PlaybackError> {

    @Override
    public PlaybackError deserialize(JsonElement json, Type typeOfT,
                                     JsonDeserializationContext context) throws JsonParseException {
      return PlaybackError.getEnum(json.getAsInt());
    }

    @Override
    public JsonElement serialize(PlaybackError src, Type typeOfSrc,
                                 JsonSerializationContext context) {
      return new JsonPrimitive(src.getInt());
    }
  }
}
