/*
 * @authors   Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonParseException;
import com.google.gson.JsonPrimitive;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.Map;

/**
 * This class describes Commands and Attributes for Tunneling Capability of a Resource.
 */
public class CapabilityTunnel extends Capability {

  public static final CapabilityId ID = CapabilityId.TUNNEL_CONTROL;

  static class StatePort {
    int port;
    State state;

    StatePort() {
    }
  }

  private final HashMap<Long, StatePort> tunnelHandleMap;

  /**
   * enum denoting various possible attributes of the {@link CapabilityTunnel}.
   */
  public enum AttributeId implements Capability.AttributeId {
    @Deprecated
    SUPPORTED_TRANSPORT_TYPE_COUNT,   // supported transport type count
    SUPPORTED_TRANSPORT_TYPE_ARR,     // array of transport types supported by tunnel
    @Deprecated
    SUPPORTED_PROTOCOL_COUNT,         // supported protocol count
    SUPPORTED_PROTOCOL_ARR,           // array of protocols supported by tunnel
    SUPPORTED_PROTOCOL_METADATA_ARR,  // array of protocol metadata supported by tunnel
    @Deprecated
    CURRENT_ACTIVE_CLIENTS_COUNT,     // no of active clients that are accessing tunnel
    CURRENT_ACTIVE_CLIENTS_ARR,       // array of active client ids that are accessing tunnel.
    MAX_ACTIVE_CLIENTS_COUNT;         // max active clients allowed to access tunnel

    static AttributeId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }

    @Override
    public CapabilityId getCapabilityId() {
      return ID;
    }
  }

  /**
   * enum denoting possible values for type of transport.
   */
  public enum TransportType {
    TCP,
    UDP;

    static TransportType getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    int getInt() {
      return ordinal();
    }
  }

  /**
   * enum denoting possible values for tunnel protocol.
   */
  public enum Protocol {
    RTSP;

    static Protocol getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    int getInt() {
      return ordinal();
    }
  }

  /**
   * enum denoting possible values for Ip version.
   */
  public enum IpVersion {
    IPv4,
    IPv6;

    static IpVersion getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    int getInt() {
      return ordinal();
    }
  }

  /**
   * enum denoting possible values for tunnel state.
   */
  public enum State {
    OPENING,
    OPENED,
    REOPENED,
    OPEN_FAILED,
    CLOSED,
    CLOSE_FAILED,
    TIMEOUT,
    MAX_CLIENT_REACHED;

    static State getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    int getInt() {
      return ordinal();
    }

    /**
     * A function to check if tunnel state is open.
     *
     * @param state The enum denoting the state of the tunnel.
     * @return True if open
     */
    public static boolean isOpen(State state) {
      switch (state) {
        case OPENED:
        case REOPENED:
          return true;
        default:
          return false;
      }
    }

    /**
     * A function to check if tunnel state is opening or opened.
     *
     * @param state The enum denoting the state of the tunnel.
     * @return True if opening/open/reopened
     */
    public static boolean isOpeningOrOpened(State state) {
      switch (state) {
        case OPENING:
        case OPENED:
        case REOPENED:
          return true;
        default:
          return false;
      }
    }

    /**
     * A function to check if tunnel state is open.
     *
     * @param state The enum denoting the state of the tunnel.
     * @return True if closed/open_failed/timeout
     */
    public static boolean isClosed(State state) {
      switch (state) {
        case CLOSED:
        case OPEN_FAILED:
        case TIMEOUT:
          return true;
        default:
          return false;
      }
    }
  }

  /**
   * enum denoting possible values for type of tunnel.
   */
  public enum TunnelType {
    LISTEN,
    CONNECT;

    static TunnelType getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    int getInt() {
      return ordinal();
    }
  }

  /**
   * enum denoting the possible Commands that can be sent for {@link CapabilityTunnel}.
   */
  // this is purely an internal commandId, not a part of coco-standard
  public enum CommandId implements Capability.CommandId {
    OPEN,
    CLOSE;

    static CommandId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    @Override
    public int getInt() {
      return ordinal();
    }
  }

  private static class TransportTypeParser implements JsonSerializer<TransportType>,
      JsonDeserializer<TransportType> {

    @Override
    public JsonElement serialize(TransportType type, Type typeOfSrc,
                                 JsonSerializationContext context) {
      return new JsonPrimitive(type.getInt());
    }

    @Override
    public TransportType deserialize(JsonElement json, Type typeOfT,
                                     JsonDeserializationContext context) throws JsonParseException {
      int enumValue = json.getAsInt();
      return TransportType.getEnum(enumValue);
    }
  }

  private static class ProtocolParser
      implements JsonSerializer<Protocol>, JsonDeserializer<Protocol> {

    @Override
    public JsonElement serialize(Protocol protocol, Type typeOfSrc,
                                 JsonSerializationContext context) {
      return new JsonPrimitive(protocol.getInt());
    }

    @Override
    public Protocol deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context)
        throws
        JsonParseException {
      int enumValue = json.getAsInt();
      return Protocol.getEnum(enumValue);
    }
  }

  private static class IpVersionParser
      implements JsonSerializer<IpVersion>, JsonDeserializer<IpVersion> {

    @Override
    public JsonElement serialize(IpVersion ipVer, Type typeOfSrc,
                                 JsonSerializationContext context) {
      return new JsonPrimitive(ipVer.getInt());
    }

    @Override
    public IpVersion deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context)
        throws JsonParseException {
      int enumValue = json.getAsInt();
      return IpVersion.getEnum(enumValue);
    }
  }

  private static class TunnelTypeParser
      implements JsonSerializer<TunnelType>, JsonDeserializer<TunnelType> {

    @Override
    public JsonElement serialize(TunnelType type, Type typeOfSrc,
                                 JsonSerializationContext context) {
      return new JsonPrimitive(type.getInt());
    }

    @Override
    public TunnelType deserialize(JsonElement json, Type typeOfT,
                                  JsonDeserializationContext context) throws JsonParseException {
      int enumValue = json.getAsInt();
      return TunnelType.getEnum(enumValue);
    }
  }

  static void init() {
    Command.GSON_BUILDER.registerTypeAdapter(TransportType.class, new TransportTypeParser());
    Command.GSON_BUILDER.registerTypeAdapter(Protocol.class, new ProtocolParser());
    Command.GSON_BUILDER.registerTypeAdapter(IpVersion.class, new IpVersionParser());
    Command.GSON_BUILDER.registerTypeAdapter(TunnelType.class, new TunnelTypeParser());
  }

  /**
   * Constructor of the current class.
   *
   * @param id     The unique id Of the capability
   * @param parent The parent Resource of the capability.
   */
  protected CapabilityTunnel(int id, Resource parent) {
    super(id, parent);
    this.tunnelHandleMap = new HashMap<>();
  }

  @Override
  public boolean supports(Capability.CommandId commandId) {
    return (null == commandId || commandId instanceof CommandId) && super.supports(commandId);
  }

  /**
   * A function to open UDP tunnel (simple wrap of {@link #open}).
   *
   * @param forwardHostName The hostname which will receive the data
   * @param forwardHostPort The port over which data will be received
   * @param tunnelType      The type of the tunnel (LISTEN/CONNECT)
   * @param ipVersion       The ipVersion used for identification (ipV4/ipV6)
   * @param timeout         The timeout for this command in millis
   * @param listener        The listener which will be triggered after the callback This will be
   *                        useful in tracking in tunnelStatusCbs
   */
  public void openUdpTunnel(String forwardHostName, int forwardHostPort, TunnelType tunnelType,
                            IpVersion ipVersion, long timeout, TunnelStatusListener listener) {
    open(forwardHostName, forwardHostPort, null, 0,
        TransportType.UDP, tunnelType, Protocol.RTSP, ipVersion, timeout, listener);
  }

  /**
   * A function to open TCP tunnel (simple wrap of {@link #open}).
   *
   * @param listeningHostName The hostname of the transmitting device
   * @param listeningHostPort The port over which the device will transmit data
   * @param tunnelType        The type of the tunnel (LISTEN/CONNECT)
   * @param ipVersion         The ipVersion used for identification (ipV4/ipV6)
   * @param timeout           The timeout for this command in millis
   * @param listener          The listener which will be triggered after the callback. This will be
   *                          useful in tracking in tunnelStatusCbs
   */
  public void openTcpTunnel(String listeningHostName, int listeningHostPort, TunnelType tunnelType,
                            IpVersion ipVersion, long timeout, TunnelStatusListener listener) {
    open(null, 0, listeningHostName, listeningHostPort,
        TransportType.TCP, tunnelType, Protocol.RTSP, ipVersion, timeout, listener);
  }

  /**
   * This function is used to open Tunnel.
   *
   * @param forwardHostName   The hostname which will receive the data
   * @param forwardHostPort   The port over which data will be received
   * @param listeningHostName The hostname of the transmitting device
   * @param listeningHostPort The port over which the device will transmit data
   * @param transportType     The TransportType (TCP/UDP) on which the transmission shall take place
   * @param tunnelType        The type of the tunnel (LISTEN/CONNECT)
   * @param protocol          The protocol of the transmission (RTSP)
   * @param ipVersion         The ipVersion used for identification (ipV4/ipV6)
   * @param timeout           The timeout for this command in millis
   * @param listener          The listener which will be triggered after the callback. This will be
   *                          useful in tracking in tunnelStatusCbs
   */
  public void open(String forwardHostName, int forwardHostPort, String listeningHostName,
                   int listeningHostPort, TransportType transportType,
                   TunnelType tunnelType, Protocol protocol, IpVersion ipVersion, long timeout,
                   TunnelStatusListener listener) {

    CocoClient.getInstance().getNativeHandler()
        .openTunnel(this, forwardHostName, forwardHostPort, listeningHostName, listeningHostPort,
            ipVersion,
            protocol, tunnelType, transportType, timeout, listener);
  }

  /**
   * A function used to close an existing tunnel.
   *
   * @param timeout timeout defined for the operation
   */
  public void closeAllTunnels(long timeout) {
    Exception lastException = null;

    synchronized (tunnelHandleMap) {
      for (long tunnelHandle : tunnelHandleMap.keySet()) {
        try {
          close(tunnelHandle, timeout);
        } catch (Exception e) {
          lastException = e;
        }
      }
    }

    if (null != lastException) {
      throw new IllegalArgumentException(lastException);
    }
  }

  /**
   * This function is used to close the existing tunnel present of this capability.
   *
   * @throws RuntimeException on failure to close the tunnel.
   */
  public void close(long tunnelHandle, long timeout) {
    if (!tunnelHandleMap.containsKey(tunnelHandle)) {
      throw new IllegalArgumentException("unrecognized tunnel handle");
    }

    CocoClient.getInstance().getNativeHandler().closeTunnel(this, tunnelHandle, timeout);
  }

  /**
   * This function is used to get the port over which the tunnel is open.
   *
   * @return int: The port number which is available for transmission.
   */
  public int getPort(long tunnelHandle) {
    StatePort statePort = tunnelHandleMap.get(tunnelHandle);

    return (null == statePort) ? 0 : statePort.port;
  }

  /**
   * This function is used to get The state of the tunnel which is
   * represented by the enum State.
   *
   * @return State: The enum denoting the state of the tunnel.
   */
  public State getState(long tunnelHandle) {
    StatePort statePort = tunnelHandleMap.get(tunnelHandle);

    return (null == statePort) ? null : statePort.state;
  }

  /**
   * A function to create the command from the Json params.
   *
   * @param primitiveCommandId The int form of the commandId
   * @param commandParams      The Json params that can form a command
   * @return The command which is formed
   */
  @Override
  protected Command<? extends Capability.CommandId> extendedCreateCommand(
      int primitiveCommandId, JsonElement commandParams) {
    return null;
  }

  protected Map<Long, StatePort> getTunnelHandleMap() {
    return tunnelHandleMap;
  }

  /**
   * Listener which will be triggered after the callback.
   */
  public interface TunnelStatusListener {
    void onStatusChanged(long tunnelHandle, int port, CapabilityTunnel.State state);
  }
}
