/*
 * @authors   Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.Gson;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonParseException;
import com.google.gson.JsonPrimitive;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import com.google.gson.annotations.SerializedName;
import java.lang.reflect.Type;

/**
 * This class describes Commands and Attributes for Warning Capability of a Resource.
 */
public class CapabilityWarning extends Capability {

  public static final CapabilityId ID = CapabilityId.WARNING_DEV_CONTROL;

  /**
   * A class to serialize and deserialize WarningType class as c-layer uses ints for enum.
   */
  private static class WarningTypeSerializer implements JsonSerializer<WarningType>,
      JsonDeserializer<WarningType> {
    @Override
    public WarningType deserialize(JsonElement json, Type typeOfT,
                                   JsonDeserializationContext context) throws
        JsonParseException {
      return new WarningType(json.getAsInt());
    }

    @Override
    public JsonElement serialize(WarningType warningType, Type typeOfSrc,
                                 JsonSerializationContext context) {
      return new JsonPrimitive(warningType.rawValue());
    }
  }

  /**
   * enum denoting various possible attributes of the {@link CapabilityWarning}.
   */
  public enum AttributeId implements Capability.AttributeId {
    STATE;

    public static AttributeId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }

    @Override
    public CapabilityId getCapabilityId() {
      return ID;
    }
  }

  /**
   * enum denoting the possible Commands that can be sent for {@link CapabilityWarning}.
   */
  public enum CommandId implements Capability.CommandId {
    START,
    STOP;

    public static CommandId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    public int getInt() {
      return ordinal();
    }
  }

  /**
   * This class denotes possible values for type of warning.
   */
  public static final class WarningType {
    public static final int BURGLAR = 0;
    public static final int FIRE = 1;
    public static final int EMERGENCY = 2;
    public static final int POLICE_PANIC = 3;
    public static final int FIRE_PANIC = 4;
    public static final int EMERGENCY_PANIC = 5;

    private final int warningType;

    public WarningType(int warningType) {
      this.warningType = warningType;
    }

    public int rawValue() {
      return warningType;
    }

    public static WarningType burglar() {
      return new WarningType(BURGLAR);
    }

    public static WarningType fire() {
      return new WarningType(FIRE);
    }

    public static WarningType emergency() {
      return new WarningType(EMERGENCY);
    }

    public static WarningType policePanic() {
      return new WarningType(POLICE_PANIC);
    }

    public static WarningType firePanic() {
      return new WarningType(FIRE_PANIC);
    }

    public static WarningType emergencyPanic() {
      return new WarningType(EMERGENCY_PANIC);
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) {
        return true;
      }

      if (o == null || getClass() != o.getClass()) {
        return false;
      }

      WarningType that = (WarningType) o;
      return warningType == that.warningType;
    }

    @Override
    public int hashCode() {
      return warningType;
    }
  }

  /**
   * Constructor of the current class.
   *
   * @param id     The unique id Of the capability
   * @param parent The parent Resource of the capability.
   */
  protected CapabilityWarning(int id, Resource parent) {
    super(id, parent);
  }

  /**
   * An initializer for the current class and adds serializers needed for this class.
   */
  static void init() {
    Command.GSON_BUILDER.registerTypeAdapter(WarningType.class, new WarningTypeSerializer());
  }

  /**
   * This function is used to check if a command is supported by the capability or not.
   *
   * @param commandId The Id denoting the command to be sent.
   * @return boolean: If the command is supported then True is returned.
   */
  @Override
  public boolean supports(Capability.CommandId commandId) {
    return (null == commandId || commandId instanceof CommandId) && super.supports(commandId);
  }

  /**
   * A function to create the command from the Json params.
   *
   * @param primitiveCommandId The int form of the commandId
   * @param commandParams      The Json params that can form a command
   * @return The command which is formed
   */
  @Override
  protected Command<? extends Capability.CommandId> extendedCreateCommand(
      int primitiveCommandId, JsonElement commandParams) {
    Command<CommandId> command;
    Gson gson = Command.GSON_BUILDER.create();
    CommandId commandId = CommandId.getEnum(primitiveCommandId);

    switch (commandId) {
      case START: {
        command = gson.fromJson(commandParams, StartWarning.class);
        break;
      }

      case STOP: {
        command = gson.fromJson(commandParams, StopWarning.class);
        break;
      }

      default: {
        command = new Command<>(commandId);
      }
    }

    return command;
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending warning command.
   */
  public static class StartWarning extends Command<CommandId> {
    @SerializedName(Constants.WARNING_TYPE)
    public WarningType warningType;

    public StartWarning() {
      super(CommandId.START);
    }

    /**
     * The overloaded constructor taking the warningType.
     *
     * @param warningType The warningType of the alarm.
     */
    public StartWarning(WarningType warningType) {
      this();
      this.warningType = warningType;
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendResourceCommand} while sending warning command.
   */
  public static class StopWarning extends Command<CommandId> {
    public StopWarning() {
      super(CommandId.STOP);
    }
  }
}
