/*
 * @authors   Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

/**
 * The base SDK Class the developer will call.
 * CocoClient class follows the singleton design pattern which upon instantiation initializes
 * GSON builders, Native handlers, callback handlers and facilitates subscription to callbacks.
 */
public class CocoClient {

  /**
   * This class defines default SDK timeouts and allows developer to set his/her own SDK timers.
   */
  public static final class ConnectivityTimers {

    // all of these timers in seconds
    public static final long DEFAULT_FAST_RETRY_DURATION = 20L;
    public static final long DEFAULT_BACKGROUND_MAX_RETRY_PERIOD = 600L;
    public static final long DEFAULT_FOREGROUND_MAX_RETRY_PERIOD = 30L;
    public static final long DEFAULT_KEEP_ALIVE_TIMEOUT = 5L;
    public static final long DEFAULT_KEEP_ALIVE_INTERVAL = 10L;

    // SDK will try to connect to other nodes at an interval
    // of 1 second for this duration in seconds. It must be a non-negative
    // integer. Set it to COCO_CLIENT_DEFAULT_FAST_RETRY_DURATION for using the
    // default value.
    final long fastRetryDuration;

    // After fastRetryDuration, SDK will try to connect to
    // other nodes at intervals which will keep on increasing with each
    // retry. backgroundMaxRetryPeriod in seconds is the maximum value of
    // this interval when the app is in background connectivity mode.
    // It must be a positive integer. Set it to
    // COCO_CLIENT_DEFAULT_BACKGROUND_MAX_RETRY_PERIOD for using the default value.
    final long backgroundMaxRetryPeriod;

    // After fastRetryDuration, SDK will try to connect to
    // other nodes at intervals which will keep on increasing with each
    // retry. foregroundMaxRetryPeriod in seconds is the maximum value of
    // this interval when the app is in foreground connectivity mode.
    // It must be a positive integer. Set it to
    // COCO_CLIENT_DEFAULT_FOREGROUND_MAX_RETRY_PERIOD for using the default value.
    final long foregroundMaxRetryPeriod;

    // SDK nodes will send a keep alive packet to other
    // connected nodes at an internal of keepAliveInterval seconds.
    // It must be a positive integer. Set it to
    // COCO_CLIENT_DEFAULT_KEEP_ALIVE_INTERVAL for using the default value.
    final long keepAliveInterval;

    // This the timeout in seconds for which SDK will wait
    // for the response of the keep alive packet. It must be a positive
    // integer. Set it to COCO_CLIENT_DEFAULT_KEEP_ALIVE_TIMEOUT for using the
    // default value.
    final long keepAliveTimeout;

    /**
     * Empty constructor that initializes default timers.
     */
    public ConnectivityTimers() {
      this(DEFAULT_FAST_RETRY_DURATION, DEFAULT_BACKGROUND_MAX_RETRY_PERIOD,
          DEFAULT_FOREGROUND_MAX_RETRY_PERIOD, DEFAULT_KEEP_ALIVE_INTERVAL,
          DEFAULT_KEEP_ALIVE_TIMEOUT);
    }

    /**
     * Overloaded constructor for a developer to set customized timers.
     *
     * @param fastRetryDuration non-negative integer after which SDK will connect to other nodes
     * @param backgroundMaxRetryPeriod interval when the app is in background connectivity mode
     * @param foregroundMaxRetryPeriod interval when the app is in foreground connectivity mode
     * @param keepAliveInterval SDK nodes will send a keep alive packet at keepAliveInterval seconds
     * @param keepAliveTimeout the timeout in seconds for which SDK will wait
     */
    public ConnectivityTimers(long fastRetryDuration, long backgroundMaxRetryPeriod,
                              long foregroundMaxRetryPeriod, long keepAliveInterval,
                              long keepAliveTimeout) {
      this.fastRetryDuration = fastRetryDuration;
      this.backgroundMaxRetryPeriod = backgroundMaxRetryPeriod;
      this.foregroundMaxRetryPeriod = foregroundMaxRetryPeriod;
      this.keepAliveInterval = keepAliveInterval;
      this.keepAliveTimeout = keepAliveTimeout;
    }

    public ConnectivityTimers toFastRetryDuration(long fastRetryDuration) {
      return new ConnectivityTimers(fastRetryDuration, backgroundMaxRetryPeriod,
          foregroundMaxRetryPeriod, keepAliveInterval, keepAliveTimeout);
    }

    public ConnectivityTimers toBackgroundMaxRetryPeriod(long backgroundMaxRetryPeriod) {
      return new ConnectivityTimers(fastRetryDuration, backgroundMaxRetryPeriod,
          foregroundMaxRetryPeriod, keepAliveInterval, keepAliveTimeout);
    }

    public ConnectivityTimers toForegroundMaxRetryPeriod(long foregroundMaxRetryPeriod) {
      return new ConnectivityTimers(fastRetryDuration, backgroundMaxRetryPeriod,
          foregroundMaxRetryPeriod, keepAliveInterval, keepAliveTimeout);
    }

    public ConnectivityTimers toKeepAliveInterval(long keepAliveInterval) {
      return new ConnectivityTimers(fastRetryDuration, backgroundMaxRetryPeriod,
          foregroundMaxRetryPeriod, keepAliveInterval, keepAliveTimeout);
    }

    public ConnectivityTimers toKeepAliveTimeout(long keepAliveTimeout) {
      return new ConnectivityTimers(fastRetryDuration, backgroundMaxRetryPeriod,
          foregroundMaxRetryPeriod, keepAliveInterval, keepAliveTimeout);
    }
  }

  /**
   * An enum denoting possible values of mode of connectivity.
   */
  public enum ConnectivityMode {
    FOREGROUND, // when app is in foreground mode
    BACKGROUND; // when app is in background mode

    private static final int offset = 0;

    public int getInt() {
      return ordinal() + offset;
    }
  }

  private static CocoClient instance = null;

  private final Map<String, Network> networkMap;

  private NativeInterface nativeHandler;
  private NativeCallbacksInterface callbacksHandler;

  /**
   * Constructor for the current class.
   * This is a private constructor because there should be only one
   * instance of the CocoClient to be existent at any point of time.
   *
   * @param platformInterface This is an interface implemented class from which
   *                          necessary details like cwdPath, clientId,
   *                          appAccessList, deviceId are extracted.
   * @throws IllegalArgumentException This constructor may throw exceptions because
   *                                  init is called inside the constructor itself,
   *                                  taking burden away from developer.
   */
  private CocoClient(PlatformInterface platformInterface,
                     NativeInterface nativeHandler, NativeCallbacksInterface callbackHandler,
                     ConnectivityTimers timers) throws IllegalArgumentException {

    String cwdPath = platformInterface.getCwdPath();
    String clientId = platformInterface.getClientId();
    String appAccessList = platformInterface.getAppAccessList();
    String downloadPath = platformInterface.getDownloadPath();

    this.nativeHandler = nativeHandler;
    this.callbacksHandler = callbackHandler;
    this.networkMap = new ConcurrentHashMap<>();

    Command.init();

    Capability.init();

    Resource.init();
    ResourceAction.init();
    Attribute.init();
    ScheduleCondition.init();

    CapabilityDoorLock.init();
    CapabilityWarning.init();

    ResourceCondition.init();
    Scene.init();
    Rule.init();

    Parameter.init();
    InfoRequest.init();
    InfoResponse.init();

    CapabilityTunnel.init();
    StorageContentMetadata.init();
    CapabilityMotorControl.init();
    RadioProtocol.init();
    CapabilityStorageControl.init();
    CapabilityMediaStreaming.init();
    CapabilityHvacControl.init();
    CapabilityRemoteControl.init();
    CapabilityKeyPressSensing.init();

    try {
      getNativeHandler().init(cwdPath, appAccessList, clientId, downloadPath, timers);
    } catch (Exception e) {
      throw new IllegalArgumentException(e);
    }
  }

  void setNativeHandler(NativeInterface nativeHandler) {
    this.nativeHandler = nativeHandler;
  }

  void setCallbacksHandler(NativeCallbacksInterface callbacksHandler) {
    this.callbacksHandler = callbacksHandler;
  }

  NativeInterface getNativeHandler() {
    return nativeHandler;
  }

  NativeCallbacksInterface getCallbacksHandler() {
    return callbacksHandler;
  }

  public String getVersionInfo() {
    return getNativeHandler().getVersion();
  }

  /**
   * A function to clear all the subscriptions to the NativeCallbacks.
   */
  public void clearSubscriptions() {
    CallbackMultiplexer.getInstance().removeAllExternalSubscribers();
  }

  /**
   * A function to add extra subscribers for callbacks.
   *
   * @param nativeCallbacks The object which has to be notified
   */
  public void addSubscription(CallbacksInterface nativeCallbacks) {
    CallbackMultiplexer.getInstance().addSubscription(nativeCallbacks);
  }

  /**
   * A function to remove the subscription of an object.
   *
   * @param nativeCallbacks The object which has to be removed
   */
  public void removeSubscription(CallbacksInterface nativeCallbacks) {
    CallbackMultiplexer.getInstance().removeSubscription(nativeCallbacks);
  }

  /**
   * A function to set the listener for Platform callbacks.
   *
   * @param listener The listener which will get the callbacks
   */
  public void setListener(PlatformInterface listener) {
    PlatformCallbacksHandler.getInstance().setListener(listener);
  }

  /**
   * A function to invalidate the current accessToken.
   *
   * @throws RuntimeException on Failure
   */
  public void invalidateAccessToken() {
    getNativeHandler().invalidateAccessToken();
  }

  /**
   * This function returns the CocoClient instance that was created earlier.
   *
   * @return This functions returns the CocoClient instance that has been created
   */
  public static CocoClient getInstance() {
    return instance;
  }

  /**
   * This function return an array of networks which are previous connected by user.
   * This function is an abstraction for {@link Native#nativeGetSavedNetworks()}.
   *
   * @return Network[] an Array of Network Objects which the developer can
   *         leverage to connect.NOTE: It is recommended to use static function
   *         present in the Network class to connect to an array of networks
   *         instead of using connect function in individual objects
   */
  public Network[] getSavedNetworks() {
    return getNativeHandler().getSavedNetworks();
  }

  /**
   * This function is used to request the available networks from the
   * server.
   * This function is an abstraction for native getAllNetworks function.
   */
  public void getAllNetworks(NetworkListListener listener) {
    getNativeHandler().getAllNetworks(listener);
  }

  /**
   * A function to restore the state of the network during second run.
   *
   * @see Network#saveState()
   * @param jsonNetworkState The json formatted string which is returned by
   *                         {@link Network#saveState()}
   * @return Network The network object which is restored.
   * @throws com.google.gson.JsonParseException on malformed JSONs
   * @throws IllegalStateException              on initiating restore on already connected
   *                                            or connecting networks
   */
  public Network restoreNetworkState(String jsonNetworkState) {
    if (null == jsonNetworkState) {
      return null;
    }

    Network network = DeepDeserializer.deserialize(jsonNetworkState);

    if (containsNetwork(network.getId())) {
      throw new IllegalArgumentException("already deserialized or connection initiated");
    }

    internalAddNetwork(network);
    return network;
  }

  /**
   * This function must be called when OAuthCallback is received.
   * The tokens must be passed as a stringified JSON response
   * NOTE: This function must be called after the {@link NativeCallbacks#authCallback}
   * is received to set the tokens.
   *
   * @param response The stringified JSON response.
   * @throws RuntimeException Throws an exception on failure to setTokens in database.
   */
  public void setTokens(String response) {
    getNativeHandler().setTokens(response);
  }

  /**
   * This function can be used to get OAuth Tokens from the c-sdk.
   * The tokens can be received in the getTokensCallback().
   *
   * @param listener The listener that is triggered after callbacks.
   */
  public void getAccessTokens(AccessTokensListener listener) {
    getNativeHandler().getAccessTokens(listener);
  }

  /**
   * This function is used to return the HashMap that the sdk uses to map the
   * networkIds to the network Objects.
   * This function can be used to data-bind the sdk to the application.
   *
   * @return The key being the networkId and the value
   *         being the networkObject. NOTE: This HashMap is for read-only any
   *         misuse may cause the program to misbehave and even crash.
   */
  public Map<String, Network> getNetworkMap() {
    return this.networkMap;
  }

  public boolean containsNetwork(String networkId) {
    return getNetworkMap().containsKey(networkId);
  }

  protected void internalAddNetwork(Network network) {
    getNetworkMap().put(network.getId(), network);
  }

  protected void internalRemoveNetwork(String networkId) {
    getNetworkMap().remove(networkId);
  }

  protected void clearNetworks() {
    getNetworkMap().clear();
  }

  public <T extends Network> T getNetwork(String networkId) {
    return Utils.castUp(getNetworkMap().get(networkId));
  }

  /**
   * sets the connectivity mode using Native method defined using JNI.
   *
   * @param mode {@link ConnectivityMode}
   */
  public void setConnectivityMode(ConnectivityMode mode) {
    if (null == mode) {
      throw new NullPointerException("mode cannot be null");
    }

    getNativeHandler().setConnectivityMode(mode);
  }

  public void informNetworkChange() {
    getNativeHandler().informNetworkChange();
  }

  /**
   * NOTE: This is a singleton builder.
   */
  public static class Configurator {

    private PlatformInterface platform = null;

    private final Set<CallbacksInterface> additionalCallbacks = new HashSet<>();

    private NativeInterface nativeHandler;
    private NativeCallbacksInterface callbackHandler;

    private Creator creator = new DefaultCreator() {
    };
    private Log.Logger logger = new Log.Logger() {
    };
    private ConnectivityTimers timers = new ConnectivityTimers();

    private long defaultCommandTimeout = Command.defaultTimeOut;
    private int clearOnNetworkStates = Network.clearNetworkForState;

    public Configurator() {
    }

    /* for testing only */
    Configurator setNativeHandler(NativeInterface nativeHandler) {
      this.nativeHandler = nativeHandler;
      return this;
    }

    Configurator setCallbackHandler(NativeCallbacksInterface callbackHandler) {
      this.callbackHandler = callbackHandler;
      return this;
    }

    public Configurator clearNetworkForStates(Network.State... states) {
      this.clearOnNetworkStates = Network.createClearOnDisconnectionFlag(states);
      return this;
    }

    public Configurator withCreator(Creator creator) {
      this.creator = Objects.requireNonNull(creator);
      return this;
    }

    public Configurator withLogger(Log.Logger logger) {
      this.logger = Objects.requireNonNull(logger);
      return this;
    }

    public Configurator withPlatform(PlatformInterface platform) {
      this.platform = platform;
      return this;
    }

    public Configurator withTimers(ConnectivityTimers timers) {
      this.timers = timers;
      return this;
    }

    public Configurator withDefaultCommandTimeout(long timeout) {
      this.defaultCommandTimeout = timeout;
      return this;
    }

    public Configurator addCallbackListener(CallbacksInterface listener) {
      additionalCallbacks.add(listener);
      return this;
    }

    public Configurator removeCallbackListener(CallbacksInterface listener) {
      additionalCallbacks.remove(listener);
      return this;
    }

    /**
     * This function is responsible to build a singleton instance of CocoClient.
     *
     * @return instance of CocoClient
     */
    public CocoClient configure() {
      if (null != instance) {
        throw new IllegalStateException(
            "CocoClient already initialized, CocoClient is a singleton "
                + "and it is being instantiated multiple times");
      }

      Objects.requireNonNull(platform);

      Factory.creator = creator;
      Log.logger = logger;
      Network.clearNetworkForState = clearOnNetworkStates;
      Command.defaultTimeOut = defaultCommandTimeout;

      nativeHandler = (null == nativeHandler) ? new DefaultNativeHandler() : nativeHandler;
      callbackHandler = (null == callbackHandler) ? new DefaultCallbacksHandler() : callbackHandler;
      timers = (null == timers) ? new ConnectivityTimers() : timers;

      instance = new CocoClient(platform, nativeHandler, callbackHandler, timers);

      instance.setListener(platform);

      instance.clearSubscriptions();

      for (CallbacksInterface listener : additionalCallbacks) {
        instance.addSubscription(listener);
      }

      return instance;
    }
  }

  /**
   * The listener which will be triggered after the {@link NativeCallbacks#networkListCallback}.
   */
  public interface NetworkListListener extends Listener {
    void onResponse(List<Network> networks, Throwable tr);
  }

  /**
   * The listener which will be triggered after the {@link NativeCallbacks#accessTokenCallback}.
   */
  public interface AccessTokensListener extends Listener {
    void onResponse(String accessToken, Throwable tr);
  }
}
