/*
 * @author   Krishna, krishnasarma@elear.solutions
 *           Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.GsonBuilder;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonParser;
import com.google.gson.JsonPrimitive;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import java.lang.reflect.Type;
import java.time.Instant;

/**
 * Command class encapsulates the commands that can be sent to
 * {@link Network#sendNetworkManagementCommand}
 * {@link Device#sendDeviceManagementCommand}
 * {@link Capability#sendResourceCommand}.
 *
 * @param <T> The CommandIdInterface which can uniquely identify the type of the command.
 */
public class Command<T extends CommandIdInterface> {

  private static final String TAG = "Command";

  static long defaultTimeOut = 0;

  static final GsonBuilder GSON_BUILDER = new GsonBuilder();

  /**
   * An enum denoting possible values of state of Command sent.
   */
  public enum State {
    SUCCESS,
    INVALID,
    FAILURE,
    PARTIAL_SUCCESS,
    TIMEOUT,
    REJECTED,
    DEVICE_BUSY,
    IN_PROGRESS,
    AUTH_FAILED,
    RESOURCE_NOT_SUPPORTED,
    SUCCESS_INSECURE,
    PARTIAL_SUCCESS_INSECURE,
    CONNECTIVITY_ERROR,
    COMMAND_NOT_SUPPORTED,
    TOKEN_NOT_SET,
    TOKEN_REFRESH_FAILED,
    DISCOVERY_NOT_APPLICABLE,
    NETWORK_DISCONNECTED,
    MAX_COUNT_REACHED,
    DEVICE_UNREACHABLE,
    NETWORK_BLOCKED;

    static State getEnum(int index) {
      return Utils.findEnum(index, values());
    }

  }

  /**
   * Marker interface for Commands.
   * Will capture the Capability / Device / Network
   * Whichever sent the command.
   *
   * @see NativeInterface#sendDeviceManagementCommand
   * @see NativeInterface#sendResourceCommand
   * @see DefaultNativeHandler#sendNetworkManagementCommand
   */
  interface Tag {
    <T extends CommandIdInterface> CommandResponse.Args<T> generateArgs(T commandId,
                                                                        String response);
  }

  private transient Tag tag;

  private transient T commandId;

  private transient long commandTimeout;

  /**
   * An initializer for the current class and adds serializers needed for this class.
   */
  static void init() {
    InstantParser instantParser = new InstantParser();
    BooleanSerializer booleanSerializer = new BooleanSerializer();

    GSON_BUILDER.registerTypeAdapter(Boolean.class, booleanSerializer);
    GSON_BUILDER.registerTypeAdapter(boolean.class, booleanSerializer);

    GSON_BUILDER.registerTypeAdapter(Instant.class, instantParser);
  }

  /**
   * This is used to as the argument to sendResourceCommand().
   * The commandId types can be found in individual Capability classes.
   *
   * @param commandId      The Object denoting which command that is being sent.
   * @param commandTimeout The timeout for this command
   */
  protected Command(T commandId, long commandTimeout) {
    if (null == commandId) {
      throw new IllegalArgumentException("commandId cannot be null");
    }

    this.commandId = commandId;
    this.commandTimeout = commandTimeout;
  }

  /**
   * This is used to as the argument to sendResourceCommand().
   * The commandId types can be found in individual Capability classes.
   *
   * @param commandId The Object denoting which command that is being sent.
   */
  protected Command(T commandId) {
    this(commandId, defaultTimeOut);
  }

  /**
   * A function to get the id of the command.
   *
   * @return T: The CommandIdInterface which can uniquely identify the type of the command.
   */
  public T getCommandId() {
    return commandId;
  }

  /**
   * A function to set the id of the command.
   * NOTE: The safety of the function purely depends on it's usage
   */
  protected void setCommandId(CommandIdInterface commandId) {
    this.commandId = Utils.castUp(commandId);
  }

  /**
   * A function to set the timeout of this command.
   *
   * @param commandTimeout The timeout for this command
   * @return Command returns the same object
   */
  public Command<T> setTimeout(long commandTimeout) {
    this.commandTimeout = commandTimeout;
    return this;
  }

  /**
   * A function to set the timeout of this command.
   */
  public long getTimeout() {
    return commandTimeout;
  }

  /**
   * A function to create the Json format of the command.
   *
   * @return JsonObject: The json format of the command
   */
  protected JsonObject toJsonObject() {
    return GSON_BUILDER.create().toJsonTree(this).getAsJsonObject();
  }

  /**
   * Set the tag of this command.
   *
   * @param tag any of {@link DeviceTag} {@link NetworkTag} {@link CapabilityTag}
   */
  final void setTag(Tag tag) {
    Tag oldTag = getTag();

    if (null != oldTag) {
      Log.w(TAG, "Command already sent. oldTag: " + oldTag);
    }

    this.tag = tag;
  }

  /**
   * A function to get the tag. Which has been set by the
   * {@link DefaultNativeHandler#sendResourceCommand}.
   *
   * @return Tag which has been set
   */
  final Tag getTag() {
    return tag;
  }

  @Override
  public String toString() {
    return "Command{"
        + "commandId=" + commandId
        + ", commandTimeout=" + commandTimeout
        + ", class=" + getClass()
        + '}';
  }

  static final class DeviceTag implements Tag {
    private final Device device;

    DeviceTag(Device device) {
      this.device = device;
    }

    @Override
    public <T extends CommandIdInterface> CommandResponse.Args<T> generateArgs(T commandId,
                                                                               String response) {
      Log.d(TAG, "ignoring genArgs on dev: " + device + ", for: " + commandId);
      return CommandResponse.createEmptyResponseArgs();
    }
  }

  static final class NetworkTag implements Tag {
    private final Network network;

    NetworkTag(Network network) {
      this.network = network;
    }

    @Override
    public <T extends CommandIdInterface> CommandResponse.Args<T> generateArgs(T commandId,
                                                                               String response) {
      Log.d(TAG, "ignoring genArgs on net: " + network + ", for: " + commandId);
      return CommandResponse.createEmptyResponseArgs();
    }
  }

  static final class CapabilityTag implements Tag {
    private final Capability capability;

    CapabilityTag(Capability capability) {
      this.capability = capability;
    }

    @Override
    public <T extends CommandIdInterface> CommandResponse.Args<T> generateArgs(T commandId,
                                                                               String response) {

      try {
        JsonElement responseParamsJson =
            JsonParser.parseString(response).getAsJsonObject().get(Constants.COMMAND_RESPONSE);
        return capability.createCommandResponseArgs(commandId, responseParamsJson);
      } catch (JsonParseException e) {
        Log.w(TAG, "error while parsing: " + response, e);
      }

      return CommandResponse.createEmptyResponseArgs();
    }
  }

  /**
   * A class to serialize boolean values. As coco-C-sdk uses int to represent a boolean Value.
   */
  private static final class BooleanSerializer implements JsonSerializer<Boolean>,
      JsonDeserializer<Boolean> {
    @Override
    public Boolean deserialize(JsonElement jsonEle, Type typeOfT,
                               JsonDeserializationContext context) throws JsonParseException {
      JsonPrimitive json = jsonEle.getAsJsonPrimitive();

      if (json.isBoolean()) {
        return json.getAsBoolean();
      }

      if (json.isNumber()) {
        return 0 != json.getAsInt();
      }

      if (json.isString()) {
        return "true".equalsIgnoreCase(json.getAsString())
            || "1".equalsIgnoreCase(json.getAsString());
      }

      throw new JsonParseException("unknown jsonType: " + json);
    }

    @Override
    public JsonElement serialize(Boolean src, Type typeOfSrc, JsonSerializationContext context) {
      return new JsonPrimitive(src ? 1 : 0);
    }
  }

  private static final class InstantParser
      implements JsonSerializer<Instant>, JsonDeserializer<Instant> {

    private static final Instant YEAR_3000 = Instant.ofEpochMilli(32503680000000L);

    @Override
    public Instant deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context)
        throws JsonParseException {
      Instant instant;
      long epochTime = 0;

      if (json.isJsonPrimitive() && json.getAsJsonPrimitive().isNumber()) {
        epochTime = json.getAsLong();
      }

      if (json.isJsonPrimitive() && json.getAsJsonPrimitive().isString()) {
        return Instant.from(Formatters.getDateTimeFormatter().parse(json.getAsString()));
      }

      instant = Instant.ofEpochSecond(epochTime);

      // note the hack
      if (instant.isBefore(YEAR_3000)) {
        return instant;
      }

      return Instant.ofEpochMilli(epochTime);
    }

    @Override
    public JsonElement serialize(Instant src, Type typeOfSrc, JsonSerializationContext context) {
      return new JsonPrimitive(src.getEpochSecond());
    }
  }
}
