/*
 * @author   Krishna, krishnasarma@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

/**
 * This class is a blueprint for Command Response received as JSON and
 * encapsulates {@link Command} with it.
 *
 * @param <T> The CommandIdInterface which can uniquely identify the type of the command.
 */
public final class CommandResponse<T extends CommandIdInterface> {

  private static final String TAG = "CommandResponse";

  /**
   * This class covers possible integer error codes as final constant fields,
   * used to create {@link Error} objects.
   */
  public static final class ErrorCode {

    private ErrorCode() {
      throw new IllegalStateException();
    }

    public static final int UNKNOWN             = 0;
    public static final int BAD_REQUEST         = 40000;
    public static final int MISSING_PARAMS      = 40001;
    public static final int INVALID_PARAMS      = 40002;
    public static final int INVALID_RANGE       = 40003;
    public static final int DUP_DATA            = 40004;
    public static final int USR_NOT_EXISTS      = 40006;
    public static final int GW_NOT_IN_NW        = 40007;
    public static final int NAME_RECURRING      = 40009;
    public static final int USR_EXISTS          = 40010;
    public static final int DUP_PRODUCT         = 40011;
    public static final int AUTH_REQUIRED       = 40101;
    public static final int SESSION_EXPIRED     = 40102;
    public static final int INVALID_CREDENTIALS = 40103;
    public static final int ACCESS_RESTRICTED   = 40301;
    public static final int GATEWAY_BUSY        = 40302;
    public static final int JOIN_PROGRESS       = 40303;
    public static final int GW_NOT_PROVISION    = 40304;
    public static final int GW_ALREADY_OWNED    = 40305;
    public static final int ALREADY_PROGRESS    = 40306;
    public static final int DISABLE_APPLICATION = 40307;
    public static final int NETWORK_BLOCKED     = 40314;
    public static final int NOT_FOUND           = 40401;
    public static final int FILE_NOT_FOUND      = 40402;
    public static final int RES_NOT_FOUND       = 40403;
    public static final int INPUT_RES_NOT_FOUND = 40404;
    public static final int NW_UNALLOCATED      = 40406;
    public static final int INTERNAL_SERVER     = 50001;
    public static final int PAYMENT_FAILED      = 50002;
    public static final int TX_FAILED           = 50050;
    public static final int DATABASE_ERROR      = 50301;
    public static final int SERVER_ERROR        = 50302;
    public static final int HTTP_TIMEOUT        = 50450;
    public static final int TIMEOUT             = 50451;
  }

  /**
   * Error class contains information about the {@link ErrorCode},
   * error message and field in response.
   */
  public static final class Error {
    public final int errorCode;
    public final String message;
    public final String fieldName;

    Error(int errorCode, String message, String fieldName) {
      this.errorCode = errorCode;
      this.message = message;
      this.fieldName = fieldName;
    }
  }

  private final Args<T> none = new Args<>();

  private Args<T> args = none;
  private final String responseJson;

  private transient Command<T> command;
  private transient Command.State state;
  private transient Error error;

  private CommandResponse(String responseJson) {
    this.responseJson = responseJson;
  }

  @SuppressWarnings("unchecked")
  static <W extends CommandIdInterface, U extends CommandResponse<W>, V> V cast(U response) {
    return (V) response;
  }

  @SuppressWarnings("unchecked")
  static <W extends CommandIdInterface, U extends CommandResponse.Args<W>, V> V castArgs(
      U response) {
    return (V) response;
  }

  /**
   * To be used by {@link Device} {@link Network} {@link Capability} to create responses.
   *
   * @return The Command response
   */
  static <T extends CommandIdInterface> Args<T> createEmptyResponseArgs() {
    return new CommandResponse.Args<>();
  }

  /**
   * To be used by {@link NativeCallbacks} only.
   *
   * @param command command to which the response has to be created
   * @param responseJson The json response body
   * @return The Command response
   */
  static <T extends CommandIdInterface> CommandResponse<T> createResponse(
      Command<T> command, String responseJson) {
    // currently, only capability tags have special responses
    CommandResponse<T> response = new CommandResponse<>(responseJson);

    response.setCommand(command);

    return response;
  }

  CommandResponse<T> setCommand(Command<T> command) {
    this.command = command;
    return this;
  }

  CommandResponse<T> setState(Command.State state) {
    this.state = state;
    return this;
  }

  CommandResponse<T> setError(Error error) {
    this.error = error;
    return this;
  }

  /**
   * This function is used to get the state of the command.
   *
   * @return State: The enum denoting the state of the command.
   */
  public Command.State getState() {
    return this.state;
  }

  /**
   * A function to get the error occurred while executing this command.
   */
  public CommandResponse.Error getError() {
    return error;
  }

  /**
   * A function to get the command corresponding to the response.
   *
   * @return get the executed command
   */
  public Command<T> getCommand() {
    return command;
  }

  /**
   * A function used to generate args from JSON response.
   *
   * @return args corresponding to {@link buzz.getcoco.iot.Command.DeviceTag},
   *         {@link buzz.getcoco.iot.Command.NetworkTag} and
   *         {@link buzz.getcoco.iot.Command.CapabilityTag}
   */
  public Args<T> getArgs() {
    if (none != args) {
      return args;
    }

    Command.Tag tag = (null == command) ? null : command.getTag();

    args = (null == tag) ? createEmptyResponseArgs() :
        tag.generateArgs(command.getCommandId(), responseJson);

    return args;
  }

  /**
   * Args found in JSON response.
   *
   * @param <T> The CommandIdInterface which can uniquely identify the type of the command.
   */
  public static class Args<T> {}

  @Override
  public String toString() {
    return "CommandResponse{"
        + "command=" + command
        + ", state=" + state
        + ", error=" + error
        + '}';
  }
}
