/*
 * @author   Krishna, krishnasarma@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentHashMap;

/**
 * ConnectivityStateManager maintains states of networks and notifies
 * Listener in case it changes.
 * It is reliant on {@link FiniteStateMachine} to handle events.
 */
public final class ConnectivityStateManager implements DefaultCallbacksInterface {

  private static final String TAG = "ConnectivityManager";

  private static ConnectivityStateManager instance = null;

  private long connectingDelay = 20 * 1000; // 20s
  private long localDelay = 7 * 1000; // 7s
  private long localDelay2 = 15 * 1000; // 15s
  private long localDelay3 = 15 * 1000; // 15s

  private final ThreadLocal<Exception> exceptionLocal = new ThreadLocal<>();
  private final Map<String, NetworkDatum> networkData;

  private CacheState cachedState;
  private WeakReference<ConnectivityStateChangeListener> listener;

  private ConnectivityStateManager() {

    CallbackMultiplexer.getInstance().internalAddSubscription(this);

    this.networkData = new ConcurrentHashMap<>(10);
  }

  /**
   * This function calls overloaded getInstance to retrieve the instance of
   * {@link ConnectivityStateManager} with a null listener.
   *
   * @return Instance of ConnectivityStateManager.
   * @see ConnectivityStateManager#getInstance(ConnectivityStateChangeListener)
   */
  public static ConnectivityStateManager getInstance() {
    if (null == instance) {
      return getInstance(null);
    }

    return instance;
  }

  /**
   * This overloaded function fetches instance of this class with assigned listener param.
   * NOTE: a weak-reference will be used to store the listener.
   *       So, the reference must be maintained by the caller.
   *
   * @param listener  listens for network state change {@link ConnectivityStateChangeListener}
   * @return Instance of ConnectivityStateManager.
   */
  //
  public static synchronized ConnectivityStateManager getInstance(
      ConnectivityStateChangeListener listener) {
    if (null == instance) {
      Log.d(TAG, "no instance available creating one");
      instance = new ConnectivityStateManager();
    }

    if (null == listener) {
      return instance;
    }

    instance.listener = new WeakReference<>(listener);

    return instance;
  }

  private boolean disconnectInternal(Network network) {
    try {
      network.disconnect();
      exceptionLocal.set(null);
    } catch (Exception e) {
      Log.w(TAG, "error while disconnection", e);
      exceptionLocal.set(e);

      return false;
    }

    return true;
  }

  private void removeNetworkData(String networkId) {
    networkData.remove(networkId);
  }

  private NetworkDatum getNetworkData(Network network) {

    NetworkDatum datum = networkData.get(network.getId());

    if (null == datum) {
      datum = new NetworkDatum(network);
      networkData.put(network.getId(), datum);

      Log.d(TAG, "getNetworkData: created: " + datum + ", for: " + network.getId());
    }

    return datum;
  }

  /**
   * This function is responsible for caching of networks that are in connected state.
   *
   * @return CacheState object with list of connected networks.
   */
  public CacheState cacheStates() {
    CacheState cachedState = new CacheState(CocoClient.getInstance().getNetworkMap().size());

    for (Network network : CocoClient.getInstance().getNetworkMap().values()) {
      Network.State state = network.getState();

      if (null != state && state.isConnected()) {
        cachedState.connectedNetworks.add(network);
        Log.d(TAG, "adding " + network.getName() + " to cache");
      }
    }

    this.cachedState = cachedState;
    return cachedState;
  }

  public void setConnectingTimeout(long connectingDelay) {
    this.connectingDelay = connectingDelay;
  }

  public void setLocalTimeout(long localDelay) {
    this.localDelay = localDelay;
  }

  public void setLocal2Timeout(long localDelay2) {
    this.localDelay2 = localDelay2;
  }

  public void setLocal3Timeout(long localDelay3) {
    this.localDelay3 = localDelay3;
  }

  public Exception getRecentException() {
    return exceptionLocal.get();
  }

  public void restoreToCachedState() {
    restoreToCachedState(cachedState);
  }

  /**
   * This function is responsible to restore the connection with the cached networks
   * by reconnecting if needed.
   *
   * @param cacheState used for caching networks in {@link ConnectivityStateManager#cacheStates()}
   */
  public void restoreToCachedState(CacheState cacheState) {
    if (null == cacheState) {
      return;
    }

    for (Network network : cacheState.connectedNetworks) {
      Network.State state = network.getState();

      Log.d(TAG, "restoring " + network.getName() + " from cache");

      if (null == state || state.isEquivalentToDisconnected()) {

        Log.d(TAG, "reconnecting on: " + network.getName());
        connect(network);
      }
    }
  }

  public static boolean disconnect(Network network) {
    return getInstance().disconnectInternal(network);
  }

  private boolean connectInternal(Network network) {
    // NOTE: without waiting for connect statusCB.
    //  setting the fsm value to Connecting.
    //  once, cb is received. The correct state will be set anyways
    try {
      NetworkDatum datum = getNetworkData(network);
      network.connect();

      datum.handleEvent(Event.CONNECTING);

      exceptionLocal.set(null);
    } catch (Exception e) {
      Log.w(TAG, "error while connection", e);

      exceptionLocal.set(e);
      return false;
    }
    return true;
  }

  public static boolean connect(Network network) {
    return getInstance().connectInternal(network);
  }

  @Override
  public void connectStatusCallback(Network network) {
    Log.d(TAG, "connectStatusCB network: " + network.getId() + ", status: " + network.getState());

    Event event;

    if (network.getState().isConnected()) {
      event = Event.CONNECTED;
    } else if (network.getState().isConnecting()) {
      event = Event.CONNECTING;
    } else if (network.getState().isLeftOrDeleted()) {
      event = Event.RESET;
    } else if (network.getState().isBlocked()) {
      event = Event.BLOCKED;
    } else if (network.getState().connectionNotPermitted()) {
      event = Event.NOT_PERMITTED;
    } else {
      event = Event.DISCONNECTED;
    }

    NetworkDatum datum = getNetworkData(network);

    if (Event.DISCONNECTED == event) {
      removeNetworkData(network.getId());
    }

    // if connected then stay in the state, else set network data received as false
    datum.handleEvent(event);
  }

  @Override
  public void nodeConnectionStatusCallback(Network network, long nodeId, NodeType nodeType,
                                           boolean isOnline) {
    Log.d(TAG, "nodeStatus: " + isOnline + ", type: " + nodeType + ", id: "
        + nodeId + ", networkId: " + network.getId());

    if (NodeType.NETWORK != nodeType) {
      return;
    }

    NetworkDatum datum = getNetworkData(network);
    Event event = (isOnline) ? Event.NODE_STATUS_ONLINE : Event.NODE_STATUS_OFFLINE;

    datum.handleEvent(event);
  }

  @Override
  public void networkDataCallback(Network network) {
    Log.d(TAG, "network data received for: " + network.getName());

    NetworkDatum datum = getNetworkData(network);
    datum.handleEvent(Event.NETWORK_DATA_RECEIVED);
  }

  /**
   * This method fetches state of Network param sent.
   *
   * @param network Network object in interest.
   * @return state from the {@link State} enum.
   */
  public static State getState(Network network) {
    if (null == network) {
      Log.d(TAG, "getState: null network received");
      return State.OFFLINE;
    }

    return getState(getInstance().getNetworkData(network));
  }

  private static State getState(NetworkDatum datum) {
    if (null == datum) {
      Log.d(TAG, "getState: null datum received");
      return State.OFFLINE;
    }

    return datum.getExternalState();
  }

  /**
   * Listener which is triggered by callbacks.
   */
  public interface ConnectivityStateChangeListener {
    void onStateChanged(Network network, State fromState, State toState);
  }

  private class NetworkDatum implements FiniteStateMachine.TimerCallback<NetworkDatum> {
    private static final int TASK_UNKNOWN = 0;
    private static final int TASK_DISCONNECT = 1;
    private static final int TASK_SET_LOCAL = 2;

    private final Timer timer;
    private final String networkId;
    private final FiniteStateMachine<InternalState, Event, NetworkDatum> fsm;

    private TimerTask nextTask = null;
    private int nextTaskType = TASK_UNKNOWN;

    private List<FiniteStateMachine.EventTable<InternalState, Event, NetworkDatum>>
        getEventTable() {

      List<FiniteStateMachine.EventTable<InternalState, Event, NetworkDatum>> eventTables =
          Arrays.asList(
          //                Event
          //        Condition State           nextState              actionCallback
          new FiniteStateMachine.EventTable<>(Event.RESET,
              InternalState.ANY,        InternalState.DELETED,    null),
          new FiniteStateMachine.EventTable<>(Event.BLOCKED,
              InternalState.ANY,        InternalState.BLOCKED,    null),
          new FiniteStateMachine.EventTable<>(Event.NOT_PERMITTED,
              InternalState.ANY,        InternalState.NOT_PERMITTED, null),

          new FiniteStateMachine.EventTable<>(Event.CONNECTING,
              InternalState.ANY,        InternalState.CONNECTING, null),
          new FiniteStateMachine.EventTable<>(Event.DISCONNECTED,
              InternalState.ANY,        InternalState.OFFLINE,    null),

          new FiniteStateMachine.EventTable<>(Event.CONNECTED,
              InternalState.CONNECTING, InternalState.LOCAL_1,    null),
          new FiniteStateMachine.EventTable<>(Event.NODE_STATUS_ONLINE,
              InternalState.LOCAL_1,    InternalState.LOCAL_2,    null),
          new FiniteStateMachine.EventTable<>(Event.NETWORK_DATA_RECEIVED,
              InternalState.LOCAL_1,    InternalState.LOCAL_3,    d -> logImpossibleState(
                  Event.NETWORK_DATA_RECEIVED, InternalState.LOCAL_1)),

          new FiniteStateMachine.EventTable<>(Event.NETWORK_DATA_RECEIVED,
              InternalState.ONLINE,     InternalState.ONLINE,     null),
          new FiniteStateMachine.EventTable<>(Event.NODE_STATUS_ONLINE,
              InternalState.ONLINE,     InternalState.ONLINE,     null),

          new FiniteStateMachine.EventTable<>(Event.NODE_STATUS_ONLINE,
              InternalState.LOCAL_3,    InternalState.ONLINE,     null),
          new FiniteStateMachine.EventTable<>(Event.NETWORK_DATA_RECEIVED,
              InternalState.LOCAL_2,    InternalState.ONLINE,     null),

          new FiniteStateMachine.EventTable<>(Event.NETWORK_DATA_LOST,
              InternalState.ONLINE,    InternalState.LOCAL_2,     d -> logImpossibleState(
                  Event.NETWORK_DATA_LOST, InternalState.ONLINE)),
          new FiniteStateMachine.EventTable<>(Event.NODE_STATUS_OFFLINE,
              InternalState.ONLINE,    InternalState.LOCAL_1,     null),

          new FiniteStateMachine.EventTable<>(Event.NODE_STATUS_OFFLINE,
              InternalState.LOCAL_2,   InternalState.LOCAL_1,     null),
          new FiniteStateMachine.EventTable<>(Event.NETWORK_DATA_LOST,
              InternalState.LOCAL_3,   InternalState.LOCAL_1,     d -> logImpossibleState(
                  Event.NETWORK_DATA_LOST, InternalState.LOCAL_3)),

          new FiniteStateMachine.EventTable<>(Event.TIMEOUT,
              InternalState.LOCAL_1,    InternalState.LOCAL_1,    d -> broadCastIfLocal()),
          new FiniteStateMachine.EventTable<>(Event.TIMEOUT,
              InternalState.LOCAL_2,    InternalState.LOCAL_2,    d -> broadCastIfLocal()),
          new FiniteStateMachine.EventTable<>(Event.TIMEOUT,
              InternalState.LOCAL_3,    InternalState.LOCAL_3,    d -> broadCastIfLocal()),
          new FiniteStateMachine.EventTable<>(Event.TIMEOUT,
              InternalState.CONNECTING, InternalState.CONNECTING, d -> disconnectIfConnecting())
      );

      for (int i = 0; i < eventTables.size(); i++) {
        Log.d(TAG, "eventTable[" + i + "]:" + eventTables.get(i));
      }

      return eventTables;
    }

    private List<FiniteStateMachine.StateTable<InternalState, Event, NetworkDatum>>
        getStateTable() {

      List<FiniteStateMachine.StateTable<InternalState, Event, NetworkDatum>> stateTables =
          Arrays.asList(
          //                           state
          //        timeout            entryCB              exitCB
          new FiniteStateMachine.StateTable<>(InternalState.LOCAL_1,
              () -> localDelay,      null,    null),
          new FiniteStateMachine.StateTable<>(InternalState.LOCAL_2,
              () -> localDelay2,     null,    null),
          new FiniteStateMachine.StateTable<>(InternalState.LOCAL_3,
              () -> localDelay3,     null,    null),
          new FiniteStateMachine.StateTable<>(InternalState.DELETED,
              0,                     this::broadcastStateChange,    null),
          new FiniteStateMachine.StateTable<>(InternalState.OFFLINE,
              0,                     this::broadcastStateChange,    null),
          new FiniteStateMachine.StateTable<>(InternalState.ONLINE,
              0,                     this::broadcastStateChange,    null),
          new FiniteStateMachine.StateTable<>(InternalState.BLOCKED,
              0,                     this::broadcastStateChange,    null),
          new FiniteStateMachine.StateTable<>(InternalState.NOT_PERMITTED,
              0,                     this::broadcastStateChange,    null),
          new FiniteStateMachine.StateTable<>(InternalState.CONNECTING,
              () -> connectingDelay, this::broadcastStateChange,    null)
      );

      for (int i = 0; i < stateTables.size(); i++) {
        Log.d(TAG, "stateTable[" + i + "]:" + stateTables.get(i));
      }

      return stateTables;
    }

    private NetworkDatum(Network network) {
      InternalState expectedState = getExpectedState(network);

      network.internalSetConnectivityState(expectedState.getExposedState());

      this.networkId = network.getId();

      this.timer = new Timer();
      this.fsm = new FiniteStateMachine<>(expectedState, this, getEventTable(), getStateTable());
    }

    private InternalState getExpectedState(Network network) {
      State exposedState;
      Network.State networkState;

      if (null == network) {
        return InternalState.OFFLINE;
      }

      networkState = network.getState();

      if (null != (exposedState = network.getConnectivityManagerState())) {
        return exposedState.getInternalState();
      }

      if (null == networkState) {
        return InternalState.OFFLINE;
      }

      if (networkState.isConnecting()) {
        return InternalState.CONNECTING;
      }

      if (networkState.isConnected()) {
        return InternalState.ONLINE;
      }

      return InternalState.OFFLINE;
    }

    private Network getNetwork() {
      return CocoClient.getInstance().getNetwork(networkId);
    }

    private State getExternalState() {
      Network network = getNetwork();

      if (null == network) {
        return State.OFFLINE;
      }

      return network.getConnectivityManagerState();
    }

    private void logImpossibleState(Event event, InternalState state) {
      Log.w(TAG, "impossible event: " + event + "state: " + state);
    }

    private void disconnectIfConnecting() {
      if (InternalState.CONNECTING != fsm.getCurrentState()) {
        return;
      }

      Network network = getNetwork();

      if (null == network) {
        return;
      }

      ConnectivityStateManager.disconnect(network);
    }

    private void broadCastIfLocal() {
      InternalState state = fsm.getCurrentState();
      WeakReference<ConnectivityStateChangeListener> weakListener =
          ConnectivityStateManager.this.listener;
      ConnectivityStateChangeListener listener = (null == weakListener) ? null : weakListener.get();

      if (InternalState.LOCAL_1 != state
          &&
          InternalState.LOCAL_2 != state
          &&
          InternalState.LOCAL_3 != state) {
        return;
      }

      State externalState = State.LOCAL;

      Network network = getNetwork();

      if (null != network) {
        network.internalSetConnectivityState(externalState);
      }

      if (null == listener) {
        Log.d(TAG, "no listener !!");
        return;
      }

      Log.d(TAG, "posting to listener");
      listener.onStateChanged(network, null, externalState);
      Log.d(TAG, "posted to listener");
    }

    private Event broadcastStateChange(NetworkDatum arg) {
      broadcastStateChange();
      return null;
    }

    private void broadcastStateChange() {
      Network network = getNetwork();
      InternalState internalState = fsm.getCurrentState();
      WeakReference<ConnectivityStateChangeListener> weakListener =
          ConnectivityStateManager.this.listener;
      ConnectivityStateChangeListener listener = (null == weakListener) ? null : weakListener.get();

      Log.d(TAG,
          "broadcasting for: " + ((null == network) ? "-" : network.getName()) + " for state: "
              + internalState);

      State externalState = internalState.getExposedState();

      if (null != network) {
        network.internalSetConnectivityState(externalState);
      }

      if (null == listener) {
        Log.d(TAG, "no listener !!");
        return;
      }

      Log.d(TAG, "posting to listener");
      listener.onStateChanged(network, null, externalState);
      Log.d(TAG, "posted to listener");
    }

    @Override
    public void triggerTimer(long timeout, NetworkDatum datum) {
      InternalState state = datum.fsm.getCurrentState();
      boolean stateConnecting = (InternalState.CONNECTING == state);
      boolean stateLocal = (InternalState.LOCAL_1 == state
          || InternalState.LOCAL_2 == state
          || InternalState.LOCAL_3 == state);

      if (stateConnecting && TASK_DISCONNECT == nextTaskType) {
        return;
      }

      if (stateLocal && TASK_SET_LOCAL == nextTaskType) {
        return;
      }

      if (stateConnecting) {
        nextTaskType = TASK_DISCONNECT;
      } else if (stateLocal) {
        nextTaskType = TASK_SET_LOCAL;
      } else {
        nextTaskType = TASK_UNKNOWN;
      }

      if (null != nextTask) {
        nextTask.cancel();
        nextTask = null;
      }

      timer.purge();

      if (nextTaskType == TASK_UNKNOWN) {
        return;
      }

      timer.schedule(nextTask = new TimerTask() {
        @Override
        public void run() {
          handleEvent(Event.TIMEOUT);
        }
      }, timeout);
    }

    private void handleEvent(Event ev) {
      try {
        fsm.handleEvent(ev, this);
      } catch (Exception e) {
        // ignoring
        Log.d(TAG, "cannot handle event", e);
      }
    }
  }

  /**
   * Class responsible to cache the list of connected networks.
   */
  public static class CacheState {
    private final List<Network> connectedNetworks;

    private CacheState(int initialCapacity) {
      connectedNetworks = new ArrayList<>(initialCapacity);
    }
  }

  /**
   * possible values of events recorded by {@link FiniteStateMachine}.
   */
  private static class Event implements FiniteStateMachine.Event {
    public static final Event CONNECTED = new Event();
    public static final Event DISCONNECTED = new Event();
    public static final Event CONNECTING = new Event();
    public static final Event NETWORK_DATA_RECEIVED = new Event();
    public static final Event NETWORK_DATA_LOST = new Event();
    public static final Event NODE_STATUS_ONLINE = new Event();
    public static final Event NODE_STATUS_OFFLINE = new Event();
    public static final Event TIMEOUT = new Event();
    public static final Event RESET = new Event();
    public static final Event BLOCKED = new Event();
    public static final Event NOT_PERMITTED = new Event();

    private Event() {
    }

    @Override
    public String toString() {
      if (CONNECTED == this) {
        return "CONNECTED";
      }

      if (DISCONNECTED == this) {
        return "DISCONNECTED";
      }

      if (CONNECTING == this) {
        return "CONNECTING";
      }

      if (NETWORK_DATA_RECEIVED == this) {
        return "NETWORK_DATA_RECEIVED";
      }

      if (NETWORK_DATA_LOST == this) {
        return "NETWORK_DATA_LOST";
      }

      if (NODE_STATUS_ONLINE == this) {
        return "NODE_STATUS_ONLINE";
      }

      if (NODE_STATUS_OFFLINE == this) {
        return "NODE_STATUS_OFFLINE";
      }

      if (TIMEOUT == this) {
        return "TIMEOUT";
      }

      if (RESET == this) {
        return "RESET";
      }

      if (BLOCKED == this) {
        return "BLOCKED";
      }

      if (NOT_PERMITTED == this) {
        return "NOT PERMITTED";
      }

      return "<UNKNOWN>" + super.toString();
    }
  }

  private static class InternalState implements FiniteStateMachine.State {
    public static final InternalState BLOCKED = new InternalState(State.BLOCKED);
    public static final InternalState NOT_PERMITTED = new InternalState(State.NOT_PERMITTED);
    public static final InternalState DELETED = new InternalState(State.DELETED);
    public static final InternalState OFFLINE = new InternalState(State.OFFLINE);
    // nmn offline | connected | data NOT received
    public static final InternalState LOCAL_1 = new InternalState(State.LOCAL);
    // nmn online  | connected | data NOT received
    public static final InternalState LOCAL_2 = new InternalState(State.LOCAL);
    // nmn offline | connected | data received
    public static final InternalState LOCAL_3 = new InternalState(State.LOCAL);
    public static final InternalState CONNECTING = new InternalState(State.CONNECTING);
    public static final InternalState ONLINE = new InternalState(State.ONLINE);
    public static final InternalState ANY = new InternalState(null);

    private final State exposedState;

    private InternalState(State exposedState) {
      this.exposedState = exposedState;
    }

    @Override
    public boolean equals(Object obj) {
      if (null == obj) {
        return false;
      }

      if (ANY == this || ANY == obj) {
        return true;
      }

      return super.equals(obj);
    }

    @Override
    public int hashCode() {
      return (null != exposedState) ? exposedState.hashCode() : 0;
    }

    @Override
    public String toString() {
      if (DELETED == this) {
        return "DELETED";
      }

      if (OFFLINE == this) {
        return "OFFLINE";
      }

      if (LOCAL_1 == this) {
        return "LOCAL_1";
      }

      if (LOCAL_2 == this) {
        return "LOCAL_2";
      }

      if (LOCAL_3 == this) {
        return "LOCAL_3";
      }

      if (CONNECTING == this) {
        return "CONNECTING";
      }

      if (ONLINE == this) {
        return "ONLINE";
      }

      if (ANY == this) {
        return "ANY";
      }

      if (BLOCKED == this) {
        return "BLOCKED";
      }

      if (NOT_PERMITTED == this) {
        return "NOT_PERMITTED";
      }

      return "<UNKNOWN>" + super.toString();
    }

    private State getExposedState() {
      return exposedState;
    }
  }

  /**
   * An enum denoting possible values of state for network object.
   */
  public enum State {
    DELETED,        // When a CocoNet is Left or Deleted.
    OFFLINE,
    LOCAL,
    CONNECTING,
    ONLINE,
    BLOCKED,
    NOT_PERMITTED;

    private InternalState getInternalState() {
      switch (this) {
        case BLOCKED:
          return InternalState.BLOCKED;
        case DELETED:
          return InternalState.DELETED;
        case OFFLINE:
          return InternalState.OFFLINE;
        case CONNECTING:
          return InternalState.CONNECTING;
        case LOCAL:
          return InternalState.LOCAL_1;
        case ONLINE:
          return InternalState.ONLINE;

        default:
      }

      return InternalState.OFFLINE;
    }
  }
}
