/*
 * @author    Krishna, krishnasarma@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.reflect.TypeToken;
import java.lang.reflect.Type;
import java.time.Instant;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;

/**
 * This is the DeepDeserializer class which can deserialize the previous
 * state of a network into the memory.
 */
class DeepDeserializer {

  private static final String TAG = "DeepDeserializer";

  private static final Gson GSON;
  private static final GsonBuilder BUILDER = new GsonBuilder();

  private static final ConcurrentHashMap<String, Network> deserializationJobs =
      new ConcurrentHashMap<>(3);

  // block written in a specific order and MUST retain BUILDER, BLOCK, GSON order
  static {
    GsonBuilder localBuilder = new GsonBuilder();

    // LOCAL BUILDER ATTACHMENTS BEGIN
    localBuilder.registerTypeAdapter(Resource.class,
        (JsonDeserializer<Resource>) (json, typeOfT, context) -> {
          JsonObject jsonObject = json.getAsJsonObject();

          String networkId = jsonObject.get(Constants.NETWORK_ID).getAsString();
          long deviceNodeId = jsonObject.get(Constants.DEVICE_NODE_ID).getAsLong();
          String resourceEui = jsonObject.get(Constants.RESOURCE_EUI).getAsString();

          Resource resource =
              Utils.getResource(deserializationJobs, networkId, deviceNodeId, resourceEui);

          if (null == resource) {
            throw new JsonParseException(
                "cannot find resource for: " + networkId + ":" + deviceNodeId + ":" + resourceEui);
          }

          return resource;
        });

    localBuilder.registerTypeAdapter(Scene.class,
        (JsonDeserializer<Scene>) (json, typeOfT, context) -> {
          JsonObject jsonObject = json.getAsJsonObject();

          String networkId = jsonObject.get(Constants.NETWORK_ID).getAsString();
          int sceneId = jsonObject.get(Constants.SCENE_ID).getAsInt();
          Network network = deserializationJobs.get(networkId);

          return network.getScene(sceneId);
        });

    localBuilder.registerTypeAdapter(ResourceAction.class,
        (JsonDeserializer<ResourceAction>) (json, typeOfT, context) -> {
          JsonObject jsonObject = json.getAsJsonObject();

          String networkId = jsonObject.get(Constants.NETWORK_ID).getAsString();
          long deviceNodeId = jsonObject.get(Constants.DEVICE_NODE_ID).getAsLong();
          String resourceEui = jsonObject.get(Constants.RESOURCE_EUI).getAsString();
          int capabilityId = jsonObject.get(Constants.CAPABILITY_ID).getAsInt();

          Capability capability = Utils.getCapability(deserializationJobs, networkId,
              deviceNodeId, resourceEui, capabilityId);

          if (null == capability) {
            throw new JsonParseException(
                "cannot find capability for: " + networkId + ":" + deviceNodeId + ":" + resourceEui
                    + ":" + capabilityId);
          }

          int resourceActionId = jsonObject.get(Constants.RESOURCE_ACTION_ID).getAsInt();
          Command<? extends Capability.CommandId> command =
              capability.createCommand(jsonObject.get(Constants.COMMAND_ID).getAsInt(),
                  jsonObject.get(Constants.COMMAND_PARAMS));

          return new ResourceAction(resourceActionId, capability, command);
        });

    localBuilder.registerTypeAdapter(ScheduleCondition.class,
        (JsonDeserializer<ScheduleCondition>) (json, typeOfT, context) -> {
          JsonObject jsonObject = json.getAsJsonObject();

          int id = jsonObject.get(Constants.SCHEDULE_CONDITION_ID).getAsInt();
          ScheduleCondition.ScheduleType scheduleType = ScheduleCondition.ScheduleType.getEnum(
              jsonObject.get(Constants.SCHEDULE_TYPE_ID).getAsInt());
          Instant startTime =
              Instant.ofEpochSecond(jsonObject.get(Constants.SCHEDULE_START_TIME).getAsLong());
          Instant expiryTime =
              Instant.ofEpochSecond(jsonObject.get(Constants.SCHEDULE_EXPIRY_TIME).getAsLong());

          return new ScheduleCondition(id, startTime, expiryTime, scheduleType);
        });

    localBuilder.registerTypeAdapter(ResourceCondition.class,
        (JsonDeserializer<ResourceCondition>) (json, typeOfT, context) -> {
          JsonObject jsonObject = json.getAsJsonObject();
          Gson gson = new Gson();

          Object thresholdAttributeValue;
          String networkId;
          String resourceEui;

          networkId = jsonObject.get(Constants.NETWORK_ID).getAsString();
          resourceEui = jsonObject.get(Constants.RESOURCE_EUI).getAsString();

          long deviceNodeId = jsonObject.get(Constants.DEVICE_NODE_ID).getAsLong();
          int capabilityId = jsonObject.get(Constants.CAPABILITY_ID).getAsInt();
          int attributeId = jsonObject.get(Constants.ATTRIBUTE_ID).getAsInt();
          // value in millis
          long conditionDuration = jsonObject.get(Constants.RULE_CONDITION_DURATION).getAsLong();

          ResourceCondition.OperatorType operatorType =
              context.deserialize(jsonObject.get(Constants.RULE_OPERATOR_ID),
                  ResourceCondition.OperatorType.class);
          Attribute.DataType attributeDataType =
              context.deserialize(jsonObject.get(Constants.RULE_VALUE_DATA_TYPE),
                  Attribute.DataType.class);

          JsonElement attributeValueJson = jsonObject.get(Constants.ATTRIBUTE_VALUE);

          switch (attributeDataType) {
            case BOOLEAN:
              thresholdAttributeValue = gson.fromJson(attributeValueJson, boolean.class);
              break;
            case BOOLEAN_ARR:
              thresholdAttributeValue = gson.fromJson(attributeValueJson, boolean[].class);
              break;
            case STRING:
            case JSON:
              thresholdAttributeValue = gson.fromJson(attributeValueJson, String.class);
              break;
            case STRING_ARR:
            case JSON_ARR:
              thresholdAttributeValue = gson.fromJson(attributeValueJson, String[].class);
              break;
            case FLOAT:
            case DOUBLE:
              thresholdAttributeValue = gson.fromJson(attributeValueJson, double.class);
              break;
            case FLOAT_ARR:
            case DOUBLE_ARR:
              thresholdAttributeValue = gson.fromJson(attributeValueJson, double[].class);
              break;
            case UINT8:
            case UINT16:
            case UINT32:
            case UINT64:
            case INT8:
            case INT16:
            case INT32:
            case INT64:
              thresholdAttributeValue = gson.fromJson(attributeValueJson, int.class);
              break;
            case UINT8_ARR:
            case UINT16_ARR:
            case UINT32_ARR:
            case UINT64_ARR:
            case INT8_ARR:
            case INT16_ARR:
            case INT32_ARR:
            case INT64_ARR:
              thresholdAttributeValue = gson.fromJson(attributeValueJson, int[].class);
              break;
            default:
              thresholdAttributeValue = null;
          }

          Network network = deserializationJobs.get(networkId);
          Device device = (null == network) ? null : network.getDevice(deviceNodeId);
          Resource resource = (null == device) ? null : device.getResource(resourceEui);
          Capability capability = (null == resource) ? null :
              resource.getCapability(Capability.CapabilityId.getEnum(capabilityId));
          int id = jsonObject.get(Constants.RESOURCE_CONDITION_ID).getAsInt();

          return new ResourceCondition(id, capability, attributeId, operatorType.getInt(),
              conditionDuration, attributeDataType, thresholdAttributeValue);
        });

    // LOCAL BUILDER ATTACHMENTS END

    Gson localGson = localBuilder.create();

    BUILDER.registerTypeAdapter(Zone.class, (JsonDeserializer<Zone>) (json, typeOfT, context) -> {

      Type type = new TypeToken<HashSet<Resource>>() {
      }.getType();
      JsonObject jsonObject = json.getAsJsonObject();

      int zoneId = jsonObject.get(Constants.ZONE_ID).getAsInt();
      String zoneName = jsonObject.get(Constants.ZONE_NAME).getAsString();
      HashSet<Resource> resources = localGson.fromJson(jsonObject.get(Constants.RESOURCES), type);

      Zone zone = Factory.createZone(zoneId, null);

      zone.internalSetName(zoneName);

      for (Resource resource : resources) {
        zone.internalAddResource(resource);
        resource.internalSetParentZone(zone);
      }

      return zone;
    });

    BUILDER.registerTypeAdapter(Scene.class, (JsonDeserializer<Scene>) (json, typeOfT, context) -> {
      String sceneName = null;
      String sceneMetadata = null;
      Type type = new TypeToken<HashMap<Integer, ResourceAction>>() {
      }.getType();

      JsonObject jsonObject = json.getAsJsonObject();

      JsonElement nameElement = jsonObject.get(Constants.SCENE_NAME);
      if (null != nameElement) {
        sceneName = nameElement.getAsString();
      }

      JsonElement metadataElement = jsonObject.get(Constants.SCENE_METADATA);
      if (null != metadataElement) {
        sceneMetadata = metadataElement.getAsString();
      }

      int sceneId = jsonObject.get(Constants.SCENE_ID).getAsInt();

      HashMap<Integer, ResourceAction> resourceActionMap =
          localGson.fromJson(jsonObject.get(Constants.RESOURCE_ACTIONS), type);

      Scene scene = Factory.createScene(sceneId, null);

      scene.internalSetName(sceneName);
      scene.internalSetMetadata(sceneMetadata);

      for (ResourceAction action : resourceActionMap.values()) {
        scene.internalAddResourceAction(action);
      }

      return scene;
    });

    BUILDER.registerTypeAdapter(Rule.class, (JsonDeserializer<Rule>) (json, typeOfT, context) -> {
      Type resourceConditionsType = new TypeToken<List<ResourceCondition>>() {
      }.getType();
      Type scheduleConditionsType = new TypeToken<List<ScheduleCondition>>() {
      }.getType();
      Type resourceActionsType = new TypeToken<List<ResourceAction>>() {
      }.getType();
      Type sceneActionsType = new TypeToken<List<Scene>>() {
      }.getType();

      JsonObject jsonObject = json.getAsJsonObject();
      JsonElement nameEle = jsonObject.get(Constants.RULE_NAME);

      int ruleId = jsonObject.get(Constants.RULE_ID).getAsInt();
      String ruleName = (null == nameEle || !nameEle.isJsonPrimitive()
          || !nameEle.getAsJsonPrimitive().isString()) ? null : nameEle.getAsString();

      List<ResourceCondition> resourceConditions =
          localGson.fromJson(jsonObject.get(Constants.RESOURCE_CONDITIONS),
              resourceConditionsType);
      List<ScheduleCondition> scheduleConditions =
          localGson.fromJson(jsonObject.get(Constants.SCHEDULE_CONDITIONS),
              scheduleConditionsType);
      List<ResourceAction> resourceActions =
          localGson.fromJson(jsonObject.get(Constants.RESOURCE_ACTIONS), resourceActionsType);
      List<Scene> sceneActions =
          localGson.fromJson(jsonObject.get(Constants.SCENE_ACTIONS), sceneActionsType);

      Rule rule = Factory.createRule(ruleId, null);

      rule.internalSetName(ruleName);

      rule.internalClearAddSceneActions(sceneActions);
      rule.internalClearAddResourceActions(resourceActions);
      rule.internalClearAddResourceConditions(resourceConditions);
      rule.internalClearAddScheduleConditions(scheduleConditions);

      return rule;
    });

    BUILDER.registerTypeAdapter(Network.class, new JsonDeserializer<Network>() {
      @Override
      public Network deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context)
          throws JsonParseException {
        Network.UserRole userRole;
        Network.AccessType accessType;
        Network.NetworkType networkType;

        JsonObject jsonObject = json.getAsJsonObject();

        userRole = GSON.fromJson(jsonObject.get(Constants.NETWORK_USER_ROLE).getAsString(),
            Network.UserRole.class);
        accessType = GSON.fromJson(jsonObject.get(Constants.NETWORK_ACCESS_TYPE).getAsString(),
            Network.AccessType.class);
        networkType = GSON.fromJson(jsonObject.get(Constants.NETWORK_TYPE).getAsString(),
            Network.NetworkType.class);

        String networkId = jsonObject.get(Constants.NETWORK_ID).getAsString();
        String networkName = jsonObject.get(Constants.NETWORK_NAME).getAsString();

        if (deserializationJobs.containsKey(networkId)) {
          throw new JsonParseException("Network deserialization already in progress");
        }

        Network network = Factory.createNetwork(networkId);
        network.internalSetName(networkName);
        network.internalSetUserRole(userRole);
        network.internalSetAccessType(accessType);
        network.internalSetNetworkType(networkType);

        deserializationJobs.put(networkId, network);

        // the order of execution is extremely important
        Type deviceMapType = new TypeToken<HashMap<Long, Device>>() {
        }.getType();
        HashMap<Long, Device> deviceMap =
            GSON.fromJson(jsonObject.get(Constants.NETWORK_DEVICES), deviceMapType);

        if (null != deviceMap) {
          for (Device device : deviceMap.values()) {
            network.internalAddDevice(device);
            device.internalSetParent(network);
          }
        }

        // zone deserialization needs the resources to be in the network object,
        // Thus, the necessity to add all devices before deserialize of zones
        Type zoneMapType = new TypeToken<HashMap<Integer, Zone>>() {
        }.getType();
        HashMap<Integer, Zone> zoneMap =
            GSON.fromJson(jsonObject.get(Constants.NETWORK_ZONES), zoneMapType);
        if (null != zoneMap) {
          for (Zone zone : zoneMap.values()) {
            network.internalAddZone(zone);
            zone.internalSetParent(network);
          }
        }

        if (null != deviceMap) {
          Zone defaultZone = network.getZone(Zone.DEFAULT_ZONE_ID);

          Objects.requireNonNull(defaultZone);

          for (Device d : deviceMap.values()) {
            for (Resource r : d) {
              Zone parentZone = r.getParentZone();

              if (null != parentZone && parentZone.containsResource(r)) {
                continue;
              }

              Log.d(TAG, "inconsistency detected: resource:" + r + ", zone: " + r.getParentZone()
                  + ", assigning to default zone");

              defaultZone.internalAddResource(r);
              r.internalSetParentZone(defaultZone);
            }
          }
        }

        Type sceneMapType = new TypeToken<HashMap<Integer, Scene>>() {
        }.getType();
        HashMap<Integer, Scene> sceneMap =
            GSON.fromJson(jsonObject.get(Constants.NETWORK_SCENES), sceneMapType);
        if (null != sceneMap) {
          for (Scene scene : sceneMap.values()) {
            network.internalAddScene(scene);
            scene.internalSetParent(network);
          }
        }

        Type ruleMapType = new TypeToken<HashMap<Integer, Rule>>() {
        }.getType();
        HashMap<Integer, Rule> ruleMap =
            GSON.fromJson(jsonObject.get(Constants.NETWORK_RULES), ruleMapType);
        if (null != ruleMap) {
          for (Rule rule : ruleMap.values()) {
            network.internalAddRule(rule);
            rule.internalSetParent(network);
          }
        }

        deserializationJobs.remove(networkId);

        return network;
      }
    });

    BUILDER.registerTypeAdapter(Device.class, new JsonDeserializer<Device>() {

      @Override
      public Device deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context)
          throws JsonParseException {

        long deviceNodeId;
        PowerSource powerSource;
        ReceiverType receiverType;

        JsonObject jsonObject = json.getAsJsonObject();

        deviceNodeId = jsonObject.get(Constants.DEVICE_NODE_ID).getAsLong();
        powerSource = GSON.fromJson(jsonObject.get(Constants.POWER_SOURCE), PowerSource.class);
        receiverType = GSON.fromJson(jsonObject.get(Constants.RECEIVER_TYPE), ReceiverType.class);

        JsonElement jsonName = jsonObject.get(Constants.DEVICE_NAME);
        JsonElement jsonDevicePsn = jsonObject.get(Constants.DEVICE_PSN);
        JsonElement jsonProductName = jsonObject.get(Constants.PRODUCT_NAME);

        String name = (null == jsonName) ? null : jsonName.getAsString();
        String devicePsn = (null == jsonDevicePsn) ? null : jsonDevicePsn.getAsString();
        String productName = (null == jsonProductName) ? null : jsonProductName.getAsString();

        JsonElement jsonMake = jsonObject.get(Constants.MAKE);
        JsonElement jsonModel = jsonObject.get(Constants.MODEL);
        JsonElement jsonFirmwareVersion = jsonObject.get(Constants.FIRMWARE_VERSION);
        JsonElement jsonIsExtendable = jsonObject.get(Constants.IS_EXTENDABLE);

        String make = (null == jsonMake) ? null : jsonMake.getAsString();
        String model = (null == jsonModel) ? null : jsonModel.getAsString();
        String firmwareVersion =
            (null == jsonFirmwareVersion) ? null : jsonFirmwareVersion.getAsString();
        boolean isExtendable = (null != jsonIsExtendable) && jsonIsExtendable.getAsBoolean();

        Log.d(TAG, "deserialize: Device name: " + name + " ,psn: " + devicePsn
            + " ,productName: " + productName + " ,make: " + make + " ,model: " + model
            + " ,firmwareVersion: " + firmwareVersion + " ,isExtendable: " + isExtendable);

        int[] protocolsSupported =
            GSON.fromJson(jsonObject.get(Constants.PROTOCOL_ID_ARR), int[].class);

        Device device = Factory.createDevice(deviceNodeId, null);

        device.internalSetExtendable(isExtendable);
        device.internalSetProtocolSupported(protocolsSupported);

        if (null != name) {
          device.internalSetName(name);
        }

        if (null != devicePsn) {
          device.internalSetDevicePsn(devicePsn);
        }

        if (null != productName) {
          device.internalSetProductName(productName);
        }

        if (null != make) {
          device.internalSetMake(make);
        }

        if (null != model) {
          device.internalSetModel(model);
        }

        if (null != firmwareVersion) {
          device.internalSetFirmwareVersion(firmwareVersion);
        }

        if (null != powerSource) {
          device.internalSetPowerSource(powerSource);
        }

        if (null != receiverType) {
          device.internalSetReceiverType(receiverType);
        }

        Type resourceMapType = new TypeToken<HashMap<String, Resource>>() {
        }.getType();
        HashMap<String, Resource> resourceMap =
            GSON.fromJson(jsonObject.get(Constants.DEVICE_RESOURCES_MAP), resourceMapType);

        if (null != resourceMap) {
          for (Resource resource : resourceMap.values()) {
            device.internalAddResource(resource);
            resource.internalSetParentDevice(device);
          }
        }

        return device;
      }
    });

    BUILDER.registerTypeAdapter(Resource.class, new JsonDeserializer<Resource>() {

      @Override
      public Resource deserialize(JsonElement json, Type typeOfT,
                                  JsonDeserializationContext context) throws JsonParseException {
        JsonObject jsonObject = json.getAsJsonObject();

        JsonElement jsonMetadata = jsonObject.get(Constants.RESOURCE_METADATA);
        JsonElement jsonName = jsonObject.get(Constants.RESOURCE_NAME);

        String name = (null == jsonName) ? null : jsonName.getAsString();
        String metadata = (null == jsonMetadata) ? null : jsonMetadata.getAsString();

        JsonElement jsonManufacturer = jsonObject.get(Constants.MANUFACTURER);
        JsonElement jsonModel = jsonObject.get(Constants.MODEL);
        JsonElement jsonFirmware = jsonObject.get(Constants.FIRMWARE_VERSION);

        String manufacturer = (null == jsonManufacturer) ? null : jsonManufacturer.getAsString();
        String model = (null == jsonModel) ? null : jsonModel.getAsString();
        String firmware = (null == jsonFirmware) ? null : jsonFirmware.getAsString();

        Log.d(TAG, "deserialize: Resource manufacturer: " + manufacturer + " ,model: "
            + model + " ,firmware: " + firmware);

        String resourceEui = jsonObject.get(Constants.RESOURCE_EUI).getAsString();

        Resource resource = Factory.createResource(resourceEui, null, null);

        if (null != name) {
          resource.internalSetName(name);
        }

        if (null != metadata) {
          resource.internalSetMetadata(metadata);
        }

        if (null != manufacturer) {
          resource.internalSetManufacturer(manufacturer);
        }

        if (null != model) {
          resource.internalSetModel(model);
        }

        if (null != firmware) {
          resource.internalSetFirmware(firmware);
        }

        PowerSource powerSource =
            GSON.fromJson(jsonObject.get(Constants.POWER_SOURCE), PowerSource.class);
        ReceiverType receiverType =
            GSON.fromJson(jsonObject.get(Constants.RECEIVER_TYPE), ReceiverType.class);
        RadioProtocol protocol =
            GSON.fromJson(jsonObject.get(Constants.PROTOCOL_ID), RadioProtocol.class);

        if (null != powerSource) {
          resource.internalSetPowerSource(powerSource);
        }

        if (null != receiverType) {
          resource.internalSetReceiverType(receiverType);
        }

        if (null != protocol) {
          resource.internalSetProtocol(protocol);
        }

        Type capabilityMapType = new TypeToken<HashMap<Capability.CapabilityId, Capability>>() {
        }.getType();
        HashMap<Capability.CapabilityId, Capability> capabilityMap =
            GSON.fromJson(jsonObject.get(Constants.RESOURCE_CAPABILITY_MAP), capabilityMapType);

        if (null != capabilityMap) {
          for (Capability capability : capabilityMap.values()) {
            resource.internalAddCapability(capability);
            capability.internalSetParent(resource);
          }
        }

        return resource;
      }
    });

    BUILDER.registerTypeAdapter(Capability.class, new JsonDeserializer<Capability>() {

      @Override
      public Capability deserialize(JsonElement json, Type typeOfT,
                                    JsonDeserializationContext context) throws JsonParseException {
        JsonObject jsonObject = json.getAsJsonObject();

        Capability.CapabilityId capabilityId =
            Capability.CapabilityId.valueOf(jsonObject.get(Constants.CAPABILITY_ID).getAsString());
        JsonElement jsonName = jsonObject.get(Constants.CAPABILITY_NAME);
        String name = (null == jsonName) ? null : jsonName.getAsString();

        Capability capability = Factory.createCapability(capabilityId.getInt(), null);

        if (null != name) {
          capability.internalSetName(name);
        }

        Type commandArrayType = new TypeToken<HashSet<Integer>>() {
        }.getType();
        HashSet<Integer> commandArray =
            GSON.fromJson(jsonObject.get(Constants.CAPABILITY_COMMAND_ARRAY), commandArrayType);

        int[] commandArr = new int[commandArray.size()];

          {
            int index = 0;

            for (Integer cmd : commandArray) {
              commandArr[index] = cmd;
              ++index;
            }
          }

        capability.clearAndSetStandardCommandSet(commandArr);

        Type attributeMapType = new TypeToken<HashMap<Integer, Attribute>>() {
        }.getType();
        HashMap<Integer, Attribute> attributeMap =
            GSON.fromJson(jsonObject.get(Constants.CAPABILITY_ATTRIBUTE_MAP), attributeMapType);

        if (null != attributeMap) {
          for (Attribute attribute : attributeMap.values()) {
            capability.internalAddAttribute(attribute);
            attribute.internalSetParent(capability);
          }
        }

        return capability;
      }
    });

    BUILDER.registerTypeAdapter(Attribute.class, new JsonDeserializer<Attribute>() {

      @Override
      public Attribute deserialize(JsonElement json, Type typeOfT,
                                   JsonDeserializationContext context) throws JsonParseException {
        Class<?> defaultValueClass;
        Class<?> currentValueClass;
        long minReportingInterval;
        long maxReportingInterval;

        JsonObject jsonObject = json.getAsJsonObject();

        JsonElement idJson = jsonObject.get(Constants.ATTRIBUTE_ID);
        JsonElement nameJson = jsonObject.get(Constants.ATTRIBUTE_NAME);
        JsonElement descJson = jsonObject.get(Constants.ATTRIBUTE_DESCRIPTION);
        JsonElement minReportJson = jsonObject.get(Constants.ATTRIBUTE_MIN_REPORTING_INTERVAL);
        JsonElement maxReportJson = jsonObject.get(Constants.ATTRIBUTE_MAX_REPORTING_INTERVAL);

        if (null == idJson) {
          throw new JsonParseException("unknown attribute: " + json);
        }

        int id = idJson.getAsInt();
        String name = (null == nameJson) ? null : nameJson.getAsString();
        String description = (null == descJson) ? null : descJson.getAsString();

        minReportingInterval = (null == minReportJson) ? Long.MAX_VALUE : minReportJson.getAsLong();
        maxReportingInterval = (null == maxReportJson) ? Long.MAX_VALUE : maxReportJson.getAsLong();

        int arrayLength = jsonObject.get(Constants.ATTRIBUTE_ARRAY_LENGTH).getAsInt();

        Log.d(TAG, "deserialize: id: " + id + ", name: " + name + " ,description: "
            + description + " ,arrayLength: " + arrayLength);

        Attribute.DataType dataType =
            GSON.fromJson(jsonObject.get(Constants.ATTRIBUTE_DATA_TYPE), Attribute.DataType.class);

        switch (dataType) {
          case BOOLEAN:
            defaultValueClass = currentValueClass = boolean.class;
            break;
          case BOOLEAN_ARR:
            defaultValueClass = boolean.class;
            currentValueClass = boolean[].class;
            break;
          case STRING:
          case JSON:
            defaultValueClass = currentValueClass = String.class;
            break;
          case STRING_ARR:
          case JSON_ARR:
            defaultValueClass = String.class;
            currentValueClass = String[].class;
            break;
          case FLOAT:
          case DOUBLE:
            defaultValueClass = currentValueClass = double.class;
            break;
          case FLOAT_ARR:
          case DOUBLE_ARR:
            defaultValueClass = double.class;
            currentValueClass = double[].class;
            break;
          case UINT8:
          case UINT16:
          case INT8:
          case INT16:
          case INT32:
            defaultValueClass = currentValueClass = int.class;
            break;
          case UINT32:
          case UINT64:
          case INT64:
            defaultValueClass = currentValueClass = long.class;
            break;
          case UINT8_ARR:
          case UINT16_ARR:
          case INT8_ARR:
          case INT16_ARR:
          case INT32_ARR:
            defaultValueClass = int.class;
            currentValueClass = int[].class;
            break;
          case UINT32_ARR:
          case UINT64_ARR:
          case INT64_ARR:
            defaultValueClass = long.class;
            currentValueClass = long[].class;
            break;
          default:
            defaultValueClass = currentValueClass = null;
        }

        if (null == defaultValueClass) {
          throw new JsonParseException("unknown data type");
        }

        Object minValue =
            GSON.fromJson(jsonObject.get(Constants.ATTRIBUTE_MIN_VALUE), defaultValueClass);
        Object maxValue =
            GSON.fromJson(jsonObject.get(Constants.ATTRIBUTE_MAX_VALUE), defaultValueClass);
        Object defaultValue =
            GSON.fromJson(jsonObject.get(Constants.ATTRIBUTE_DEFAULT_VALUE), defaultValueClass);
        Object currentValue =
            GSON.fromJson(jsonObject.get(Constants.ATTRIBUTE_VALUE), currentValueClass);

        Log.d(TAG, "deserialize: minValue: " + minValue + ", maxValue: " + maxValue
            + ", defaultValue: " + defaultValue + ", currentValue: " + currentValue);

        Attribute attribute = Factory.createAttribute(id, null);

        if (null != name) {
          attribute.internalSetName(name);
        }

        if (null != description) {
          attribute.internalSetDescription(description);
        }

        attribute.internalSetDataType(dataType.getInt());

        attribute.internalSetArrayLength(arrayLength);
        attribute.internalSetMinValue(minValue);
        attribute.internalSetMaxValue(maxValue);

        attribute.internalSetCurrentValue(currentValue);
        attribute.internalSetDefaultValue(defaultValue);
        attribute.internalSetMinReportingInterval(minReportingInterval);
        attribute.internalSetMaxReportingInterval(maxReportingInterval);

        return attribute;
      }
    });

    GSON = BUILDER.create();
  }

  /**
   * A function to deserialize the NetworkJson string.
   *
   * @param json The stringified json which is returned by {@link DeepSerializer#serialize(Network)}
   * @return The network object with the state restored.
   * @apiNote Must deserialize zone objects only after device and resource
   */
  static Network deserialize(String json) {
    return GSON.fromJson(json, Network.class);
  }
}
