/*
 * @author    Krishna, krishnasarma@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import com.google.gson.reflect.TypeToken;
import java.lang.reflect.Type;
import java.util.List;
import java.util.Set;

/**
 * This is the DeepSerializer class which can serialize the whole memory
 * of the network under current state.
 */
// NOTE: serializer don't work with custom maps
// TODO: make serializer work with custom maps
class DeepSerializer {
  private static final String TAG = "DeepSerializer";

  private static final Gson GSON;
  private static final GsonBuilder BUILDER = new GsonBuilder().enableComplexMapKeySerialization();

  // block written in a specific order and MUST retain BUILDER, BLOCK, GSON order
  static {
    // Mustn't use {@link BUILDER}  since it needs to convert resource to the
    // complete resource object.
    GsonBuilder localBuilder = new GsonBuilder();

    // LOCAL BUILDER START
    localBuilder.registerTypeAdapter(Resource.class,
        (JsonSerializer<Resource>) (resource, typeOfSrc, context) -> {
          JsonObject jsonObject = new JsonObject();

          Log.d(TAG,
              "local serializing resource:" + resource.getParent().getId() + ":"
                  + resource.getId());

          jsonObject.addProperty(Constants.NETWORK_ID, resource.getParent().getParent().getId());
          jsonObject.addProperty(Constants.DEVICE_NODE_ID, resource.getParent().getId());
          jsonObject.addProperty(Constants.RESOURCE_EUI, resource.getId());

          return jsonObject;
        });

    localBuilder.registerTypeAdapter(Scene.class,
        (JsonSerializer<Scene>) (src, typeOfSrc, context) -> {
          JsonObject jsonObject = new JsonObject();

          jsonObject.addProperty(Constants.NETWORK_ID, src.getParent().getId());
          jsonObject.addProperty(Constants.SCENE_ID, src.getId());

          return jsonObject;
        });

    localBuilder.registerTypeAdapter(ResourceAction.class,
        (JsonSerializer<ResourceAction>) (resourceAction, typeOfSrc, context) -> {
          JsonObject jsonObject = new JsonObject();

          Capability c = resourceAction.getCapability();
          Resource r = (null == c) ? null : c.getParent();
          Device d = (null == r) ? null : r.getParent();
          Network n = (null == d) ? null : d.getParent();

          jsonObject.addProperty(Constants.NETWORK_ID, (null == n) ? null : n.getId());
          jsonObject.addProperty(Constants.RESOURCE_EUI, (null == r) ? null : r.getId());
          jsonObject.addProperty(Constants.DEVICE_NODE_ID, (null == d) ? null : d.getId());
          jsonObject.addProperty(Constants.CAPABILITY_ID, (null == c) ? null : c.getId().getInt());

          jsonObject.addProperty(Constants.COMMAND_ID,
              resourceAction.getCommand().getCommandId().getInt());

          jsonObject.add(Constants.COMMAND_PARAMS, resourceAction.getCommand().toJsonObject());

          jsonObject.addProperty(Constants.RESOURCE_ACTION_ID, resourceAction.getId());

          return jsonObject;
        });

    localBuilder.registerTypeAdapter(ScheduleCondition.class,
        (JsonSerializer<ScheduleCondition>) (src, typeOfSrc, context) -> {
          JsonObject jsonObject = new JsonObject();

          jsonObject.addProperty(Constants.SCHEDULE_CONDITION_ID, src.getId());
          jsonObject.addProperty(Constants.SCHEDULE_TYPE_ID, src.getScheduleType().getInt());
          jsonObject.addProperty(Constants.SCHEDULE_START_TIME,
              src.getStartTime().getEpochSecond());
          jsonObject.addProperty(Constants.SCHEDULE_EXPIRY_TIME,
              src.getInstantExpiryTime().getEpochSecond());

          return jsonObject;
        });

    localBuilder.registerTypeAdapter(ResourceCondition.class,
        (JsonSerializer<ResourceCondition>) (src, typeOfSrc, context) -> {
          JsonObject jsonObject = new JsonObject();

          jsonObject.addProperty(Constants.RESOURCE_CONDITION_ID, src.getId());

          jsonObject.addProperty(Constants.NETWORK_ID,
              src.getParentCapability().getParent().getParent().getParent().getId());
          jsonObject.addProperty(Constants.RESOURCE_EUI,
              src.getParentCapability().getParent().getId());
          jsonObject.addProperty(Constants.DEVICE_NODE_ID,
              src.getParentCapability().getParent().getParent().getId());
          jsonObject.addProperty(Constants.CAPABILITY_ID,
              src.getParentCapability().getId().getInt());
          jsonObject.addProperty(Constants.ATTRIBUTE_ID, src.getAttributeId().getInt());
          jsonObject.addProperty(Constants.RULE_CONDITION_DURATION, src.getConditionDuration());

          jsonObject.add(Constants.RULE_OPERATOR_ID, context.serialize(src.getOperatorType()));
          jsonObject.add(Constants.RULE_VALUE_DATA_TYPE,
              context.serialize(src.getAttributeDataType()));

          jsonObject.add(Constants.ATTRIBUTE_VALUE,
              context.serialize(src.getThresholdAttributeValue()));

          return jsonObject;
        });
    // LOCAL BUILDER END

    Gson localGson = localBuilder.create();

    BUILDER.registerTypeAdapter(Zone.class, (JsonSerializer<Zone>) (zone, typeOfSrc, context) -> {
      JsonObject jsonObject = new JsonObject();
      Type resourceListType = new TypeToken<Set<Resource>>() {
      }.getType();

      jsonObject.addProperty(Constants.ZONE_ID, zone.getId());
      jsonObject.addProperty(Constants.ZONE_NAME, zone.getName());

      jsonObject.add(Constants.RESOURCES,
          localGson.toJsonTree(zone.getResources(), resourceListType));

      return jsonObject;
    });

    BUILDER.registerTypeAdapter(Scene.class, new JsonSerializer<Scene>() {
      @Override
      public JsonElement serialize(Scene scene, Type typeOfSrc, JsonSerializationContext context) {
        JsonObject jsonObject = new JsonObject();

        jsonObject.addProperty(Constants.SCENE_ID, scene.getId());
        jsonObject.addProperty(Constants.SCENE_NAME, scene.getName());
        jsonObject.addProperty(Constants.SCENE_METADATA, scene.getMetadata());
        jsonObject.add(Constants.RESOURCE_ACTIONS,
            localGson.toJsonTree(scene.getResourceActionMap()));

        return jsonObject;
      }
    });

    BUILDER.registerTypeAdapter(Rule.class, new JsonSerializer<Rule>() {
      @Override
      public JsonElement serialize(Rule src, Type typeOfSrc, JsonSerializationContext context) {
        JsonObject jsonObject = new JsonObject();

        jsonObject.addProperty(Constants.RULE_ID, src.getId());
        jsonObject.addProperty(Constants.RULE_NAME, src.getName());

        jsonObject.add(Constants.RESOURCE_CONDITIONS,
            localGson.toJsonTree(src.getResourceConditions()));
        jsonObject.add(Constants.SCHEDULE_CONDITIONS,
            localGson.toJsonTree(src.getScheduleConditions()));
        jsonObject.add(Constants.RESOURCE_ACTIONS, localGson.toJsonTree(src.getResourceActions()));
        jsonObject.add(Constants.SCENE_ACTIONS,
            localGson.toJsonTree(src.getSceneActions(), new TypeToken<List<Scene>>() {
            }.getType()));

        return jsonObject;
      }
    });

    GSON = BUILDER.serializeSpecialFloatingPointValues().create();
  }

  static String serialize(Network network) {
    return GSON.toJson(network);
  }
}
