/*
 * @authors   Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.Gson;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;

class DefaultCallbacksHandler extends NativeCallbacksInterface {

  private static final String TAG = "CallbacksHandler";

  private static final PlatformInterface platformInterface = PlatformCallbacksHandler.getInstance();
  private static final CallbackMultiplexer multiplexer = CallbackMultiplexer.getInstance();

  /**
   * A function to call connectStatusCallback in callbacksInstance and
   * update the Object with appropriate data.
   * This function is used to relay the control to the developer when the
   * status of a network connect is received and the developer can
   * implement his logic depending on the callback.
   *
   * @param networkId            The Id of the network to which the callback is
   *                             being received.
   * @param networkState               The state of the network for which the callback
   *                             is being received.
   * @param nativeNetworkContext The context of the network which has been passed
   *                             by the developer during connect call.
   */
  @Override
  protected void connectStatusCallback(String networkId, int networkState,
                                       Object nativeNetworkContext) {

    Network network;
    Network.State state;

    network = Utils.addMissingNetwork(networkId);

    // updating the state in Network object
    state = Network.State.getEnum(networkState);
    network.internalSetState(state);

    multiplexer.connectStatusCallback(network);

    if (Network.containsClearFlag(state)) {
      // deleting from map on failure
      // since connect should be called anyways. So, will be added to map
      CocoClient.getInstance().internalRemoveNetwork(networkId);
      network.internalRemoveNetwork();
    }
  }

  /**
   * The status of {@link Network#leave}.
   *
   * @param status        status of the command
   * @param nativeContext context which is passed with leave call
   */
  @Override
  protected void leaveNetworkStatusCallback(int status, Object nativeContext) {
    Command.State commandStatus = Command.State.getEnum(status);
    Network.LeaveStatusListener listener = null;
    Network network = null;

    if (null != nativeContext) {
      Context context = (Context) nativeContext;

      listener = cast(context.developerContext);
      network = (Network) context.sdkContext;
    }

    if (null != listener) {
      listener.onResponse(commandStatus,
          Command.State.SUCCESS == commandStatus ? null : new StateException(commandStatus));
    }

    multiplexer.leaveNetworkStatusCallback(network, commandStatus);
  }

  /**
   * The callback describing a device present in the network.
   * This function creates or updates a device object as per the information received and then
   * relays controller over to the interface for implementation of logic via callbacks interface.
   *
   * @param networkId            The Id of the network to which the device belongs
   *                             to.
   * @param deviceNodeId         The unique Id of the device.
   * @param name                 The name of the device.
   * @param extendable           The flag showing if it is extendable.
   * @param protocolsSupported   The list of supported protocols
   * @param resourceEuis         The Id of resources which belong to the device
   * @param nativeNetworkContext The network context of the network to which the
   *                             device belongs to.
   */
  @Override
  protected void deviceInfoCallback(boolean trigger, boolean append, String networkId,
                                    long deviceNodeId, String name, String devicePsn,
                                    String productName,
                                    String make, String model, String firmwareVersion,
                                    int powerSource, int receiverType, boolean extendable,
                                    int[] protocolsSupported, String[] resourceEuis,
                                    Object nativeNetworkContext) {

    Device device;
    Object context = null;
    if (nativeNetworkContext != null) {
      context = ((Context) nativeNetworkContext).developerContext;
    }

    Network network = Utils.addMissingNetwork(networkId);

    // Creating new device if it is not existent
    if (!network.containsDevice(deviceNodeId)) {

      device = Factory.createDevice(deviceNodeId, network);

      network.internalAddDevice(device);
    } else {
      device = network.getDevice(deviceNodeId);
    }

    device.internalSetName(name);

    if (!append) {
      device.internalSetExtendable(extendable);
      device.internalSetDevicePsn(devicePsn);
      device.internalSetProductName(productName);
      device.internalSetMake(make);
      device.internalSetModel(model);
      device.internalSetFirmwareVersion(firmwareVersion);
      device.internalSetPowerSource(PowerSource.getEnum(powerSource));
      device.internalSetReceiverType(ReceiverType.getEnum(receiverType));
      device.internalSetProtocolSupported(protocolsSupported);

      for (String resourceEui : resourceEuis) {
        if (!device.containsResource(resourceEui)) {
          Utils.addMissingResource(networkId, deviceNodeId, resourceEui, context);
        }
      }

      if (resourceEuis.length != device.getResourceMap().size()) {
        HashSet<String> resourceEuiSet = new HashSet<>(Arrays.asList(resourceEuis));

        // NOT maintaining zone here.
        // zone_info_cb should take care of it.
        // OR
        // resource_removed_cb should take care of it
        for (Resource resource : device) {
          if (resourceEuiSet.contains(resource.getId())) {
            continue;
          }

          device.internalRemoveResource(resource.getId());
          resource.internalRemoveResource();
        }
      }

      device.internalSetReady(true);
    }

    if (trigger) {
      multiplexer.deviceInfoCallback(device);
    }
  }

  /**
   * The callback describes a resource present in a device.
   * This callback will create and add the resource to the resourceMap
   * update the object accordingly and calls the resourceSummary callback
   * in the callbacksInstance.
   *
   * @param networkId            The of the network to which the resource belongs.
   * @param deviceNodeId         The Id of device to which the resource belongs.
   * @param resourceEui          The unique Id of the resource.
   * @param name                 The name of the resource.
   * @param metadata             The metadata of the resource.
   * @param manufacturer         The manufacturer of the resource.
   * @param model                The model of the resource.
   * @param firmware             The firmware the resource is running.
   * @param powerSource          The power source off which the resource is being
   *                             powered.
   * @param receiverType         The receiver type which the device uses.
   * @param nativeNetworkContext The context of the network to which was passed
   *                             during connect call.
   */
  @Override
  protected void resourceSummaryCallback(boolean trigger, String networkId, long deviceNodeId,
                                         String resourceEui, String name,
                                         String metadata, String manufacturer, String model,
                                         String firmware, int powerSource,
                                         int receiverType, Object nativeNetworkContext) {

    Resource resource;

    Device parentDevice = Utils.addMissingDevice(networkId, deviceNodeId);
    Zone defaultZone = parentDevice.getParent().getZone(Zone.DEFAULT_ZONE_ID);

    // creating new resource if it's not yet created
    if (!parentDevice.containsResource(resourceEui)) {
      resource = Factory.createResource(resourceEui, parentDevice, defaultZone);

      parentDevice.internalAddResource(resource);

      defaultZone.internalAddResource(resource);
    } else {
      resource = parentDevice.getResource(resourceEui);
    }

    if (!Objects.equals(resource.getName(), name)) {
      resource.internalSetName(name);
    }

    if (!Objects.equals(resource.getMetadata(), metadata)) {
      resource.internalSetMetadata(metadata);
    }

    if (!Objects.equals(resource.getManufacturer(), manufacturer)) {
      resource.internalSetManufacturer(manufacturer);
    }

    if (!Objects.equals(resource.getModel(), model)) {
      resource.internalSetModel(model);
    }

    if (!Objects.equals(resource.getFirmware(), firmware)) {
      resource.internalSetFirmware(firmware);
    }

    if (!Objects.equals(resource.getPowerSource(), PowerSource.getEnum(powerSource))) {
      resource.internalSetPowerSource(PowerSource.getEnum(powerSource));
    }

    if (!Objects.equals(resource.getReceiverType(), ReceiverType.getEnum(receiverType))) {
      resource.internalSetReceiverType(ReceiverType.getEnum(receiverType));
    }

    resource.internalSetParentDevice(parentDevice);

    // This case must not happen, but is a fail safe in case of an incorrect state retrieval
    if (!resource.getParentZone().containsResource(resource)) {
      defaultZone.internalAddResource(resource);
      resource.internalSetParentZone(defaultZone);
    }

    if (!resource.isReady()) {
      resource.internalMarkAsReady();
    }

    if (trigger) {
      multiplexer.resourceCallback(resource);
    }
  }

  /**
   * This callback describes the capabilities a resource has.
   * This callback will create and add the capability to the
   * capabilityMap update the object accordingly and calls the capability
   * callback in the callbacks instance.
   *
   * @param networkId            The of the network to which the capability
   *                             belongs.
   * @param deviceNodeId         The Id of device to which the capability
   *                             belongs.
   * @param resourceEui          The unique Id of the resource to which the
   *                             capability belongs.
   * @param capabilityId         The unique Id of the capability.
   * @param name                 The name of the capability.
   * @param standardCommandArr   The commands which are supported as a
   *                             standard.
   * @param nativeNetworkContext The context that was passed during the connect
   *                             call.
   */
  @Override
  protected void resourceCapabilityCallback(boolean trigger, String networkId, long deviceNodeId,
                                            String resourceEui,
                                            int capabilityId, String name, int[] standardCommandArr,
                                            Object nativeNetworkContext) {

    Capability capability;
    Object context = null;
    if (nativeNetworkContext != null) {
      context = ((Context) nativeNetworkContext).developerContext;
    }

    Resource parentResource =
        Utils.addMissingResource(networkId, deviceNodeId, resourceEui, context);

    Capability.CapabilityId capId = Capability.CapabilityId.getEnum(capabilityId);

    // creating new capability if it's not yet created
    if (!parentResource.containsCapability(capId)) {
      capability = Factory.createCapability(capabilityId, parentResource);

      parentResource.internalAddCapability(capability);
    } else {
      capability = parentResource.getCapability(capId);
    }

    if (!Objects.equals(capability.getName(), name)) {
      capability.internalSetName(name);
    }

    capability.clearAndSetStandardCommandSet(standardCommandArr);

    capability.internalMarkAsReady();

    if (trigger) {
      multiplexer.resourceCapabilityCallback(capability);
    }
  }

  /**
   * This callback describes the attributes each capability has.
   * This callback will create and add the attribute to the attributeMap
   * update the object accordingly and calls the attribute callback in
   * the callbacks instance.
   *
   * @param networkId            The of the network to which the capability
   *                             belongs.
   * @param deviceNodeId         The Id of device to which the capability belongs.
   * @param resourceEui          The unique Id of the resource to which the
   *                             capability belongs.
   * @param capabilityId         The capability Id which is the parent of the
   *                             attribute.
   * @param attributeId          The Id of the attribute.
   * @param name                 The name of the attribute.
   * @param description          The description of the attribute.
   * @param minReportingInterval The minimum reporting time interval.
   * @param maxReportingInterval The maximum reporting time interval.
   * @param dataType             The dataType of the attribute.
   * @param arrayLength          The arrayLength of the attribute If it's an
   *                             array.
   * @param minValue             The minimum value the attribute can take.
   * @param maxValue             The maximum value the attribute can take.
   * @param defaultValue         The default value of the attribute.
   * @param currentValue         The currentValue of the attribute.
   * @param nativeNetworkContext The context of the network which was passed
   *                             during connect().
   * @see ResourceCondition.ResourceConditionParser#deserialize
   */
  @Override
  protected void resourceAttributeCallback(boolean trigger, String networkId, long deviceNodeId,
                                           String resourceEui, int capabilityId,
                                           int attributeId, String name, String description,
                                           long minReportingInterval, long maxReportingInterval,
                                           int dataType, int arrayLength, Object minValue,
                                           Object maxValue, Object defaultValue,
                                           Object currentValue, boolean realtimeUpdate,
                                           Object nativeNetworkContext) {

    long timeTaken = System.currentTimeMillis();
    Log.d(TAG, "resourceAttributeCallback: started at: " + timeTaken);

    Attribute attribute;
    Object context = null;

    if (nativeNetworkContext != null) {
      context = ((Context) nativeNetworkContext).developerContext;
    }

    Capability parentCapability =
        Utils.addMissingCapability(networkId, deviceNodeId, resourceEui, capabilityId, context);

    // creating object if it's not created yet
    if (!parentCapability.containsAttribute(attributeId)) {
      attribute = Factory.createAttribute(attributeId, parentCapability);

      parentCapability.internalAddAttribute(attribute);
    } else {
      attribute = parentCapability.getAttribute(attributeId);
    }

    attribute.internalSetRealtimeUpdate(realtimeUpdate);

    if (!Objects.equals(attribute.getName(), name)) {
      attribute.internalSetName(name);
    }

    if (!Objects.equals(attribute.getDescription(), description)) {
      attribute.internalSetDescription(description);
    }

    if (!Objects.equals(attribute.getDataType(), Attribute.DataType.getEnum(dataType))) {
      attribute.internalSetDataType(dataType);
    }

    if (!Objects.equals(attribute.getArrayLength(), arrayLength)) {
      attribute.internalSetArrayLength(arrayLength);
    }

    if (!Objects.deepEquals(attribute.getMinValue(), minValue)) {
      attribute.internalSetMinValue(minValue);
    }

    if (!Objects.deepEquals(attribute.getMaxValue(), maxValue)) {
      attribute.internalSetMaxValue(maxValue);
    }

    if (!Objects.deepEquals(attribute.getCurrentValue(), currentValue)) {
      attribute.internalSetCurrentValue(currentValue);
    }

    if (!Objects.deepEquals(attribute.getDefaultValue(), defaultValue)) {
      attribute.internalSetDefaultValue(defaultValue);
    }

    if (attribute.getMinReportingInterval() != minReportingInterval) {
      attribute.internalSetMinReportingInterval(minReportingInterval);
    }

    if (attribute.getMaxReportingInterval() != maxReportingInterval) {
      attribute.internalSetMaxReportingInterval(maxReportingInterval);
    }

    attribute.internalMarkAsReady();

    timeTaken = System.currentTimeMillis() - timeTaken;
    Log.d(TAG, "resourceAttributeCallback: completed in: " + timeTaken + "ms");

    if (trigger) {
      multiplexer.resourceAttributeCallback(attribute);
    }
  }

  /**
   * This is a callback which is issued after the command is executed.
   * This function changes the status of a command and reports it in callbacks instance.
   *
   * @param status               The status of the connection.
   * @param nativeNetworkContext The context of the network which was passed
   *                             during connect call.
   * @param nativeCommandContext The command context which was passed during the
   *                             {@link NativeInterface#sendResourceCommand} call.
   */
  @Override
  protected void commandStatusCallback(int status, String responseJson, Object nativeNetworkContext,
                                       Object nativeCommandContext) {

    Capability.CommandStatusListener<Capability.CommandId> listener;
    Command<Capability.CommandId> command;
    CommandResponse<Capability.CommandId> response;
    Command.State commandStatus = Command.State.getEnum(status);

    if (null != nativeCommandContext) {
      listener = cast(((Context) nativeCommandContext).developerContext);
      command = cast(((Context) nativeCommandContext).sdkContext);
    } else {
      Log.e(TAG,
          "cannot find context. This might mean a wrong implementation of c-sdk OR java-sdk");
      Log.e(TAG, "IGNORING Command status callback");
      return;
    }

    Log.d(TAG, "commandStatusCallback: status: " + status);

    response = CommandResponse
        .createResponse(command, responseJson)
        .setState(commandStatus);

    if (null != listener) {
      listener.onCommandStatus(CommandResponse.cast(response),
          Command.State.SUCCESS == commandStatus ? null : new StateException(commandStatus));
    }

    multiplexer.commandStatusCallback(response);
  }

  /**
   * This function is a callback to the getAllNetwork() and contains the
   * available networks to connect to.
   *
   * @param networkIds           The Ids of the networks.
   * @param networkNames         The names of the networks.
   * @param userRoles            The role of user in individual network.
   * @param accessTypes          The type of access the app has in a network.
   * @param nativeRequestContext The contexts of each and individual network
   *                             objects.
   */
  @Override
  protected void networkListCallback(String[] networkIds, String[] networkNames, int[] netTypes,
                                     int[] userRoles, int[] accessTypes,
                                     Object nativeRequestContext) {

    CocoClient.NetworkListListener commandContext = null;
    List<Network> networksToRemove = null;

    if (null != nativeRequestContext) {
      commandContext = cast(((Context) nativeRequestContext).developerContext);
    }

    if (null == networkIds) {
      if (null != commandContext) {
        commandContext.onResponse(null, new IOException("Internal Error"));
      }

      multiplexer.networkListCallback(null);
      return;
    }

    ArrayList<Network> networksList = new ArrayList<>();

    // if network is already connected then the network Object is reused.
    for (int i = 0; i < networkIds.length; i++) {
      Network network;

      if (null != (network = CocoClient.getInstance().getNetwork(networkIds[i]))) {
        Network.NetworkType netType;
        Network.UserRole userRole;
        Network.AccessType accessType;

        netType = Network.NetworkType.getEnum(netTypes[i]);
        userRole = Network.UserRole.getEnum(userRoles[i]);
        accessType = Network.AccessType.getEnum(accessTypes[i]);

        networksList.add(network);

        if (!Objects.equals(network.getName(), networkNames[i])) {
          network.internalSetName(networkNames[i]);
        }

        if (!Objects.equals(network.getUserRole(), userRole)) {
          network.internalSetUserRole(userRole);
        }

        if (!Objects.equals(network.getAccessType(), accessType)) {
          network.internalSetAccessType(accessType);
        }

        if (!Objects.equals(network.getNetworkType(), netType)) {
          network.internalSetNetworkType(netType);
        }
      } else {
        network = Factory.createNetwork(networkIds[i]);

        network.internalSetName(networkNames[i]);
        network.internalSetNetworkType(Network.NetworkType.getEnum(netTypes[i]));
        network.internalSetUserRole(Network.UserRole.getEnum(userRoles[i]));
        network.internalSetAccessType(Network.AccessType.getEnum(accessTypes[i]));

        networksList.add(network);
      }
    }

    Arrays.sort(networkIds, String.CASE_INSENSITIVE_ORDER);

    for (Network network : CocoClient.getInstance().getNetworkMap().values()) {
      int index = Arrays.binarySearch(networkIds, network.getId(), String.CASE_INSENSITIVE_ORDER);

      if (0 > index) {
        // not present in networkList. So, removing from networkMap

        if (null == networksToRemove) {
          networksToRemove = new ArrayList<>();
        }

        networksToRemove.add(network);
      }
    }

    if (null != networksToRemove) {
      for (Network network : networksToRemove) {
        CocoClient.getInstance().internalRemoveNetwork(network.getId());
        network.internalRemoveNetwork();
      }
    }

    if (null != commandContext) {
      commandContext.onResponse(networksList, null);
    }

    multiplexer.networkListCallback(networksList);
  }

  /**
   * This function starts an OAuth routine when the c-SDK is unable to
   * get access tokens from refresh tokens.
   *
   * @param authorizationEndpoint The authorization endpoint.
   * @param tokenEndpoint         The token end point.
   */
  @Override
  protected void authCallback(String authorizationEndpoint, String tokenEndpoint) {
    platformInterface.authCallback(authorizationEndpoint, tokenEndpoint);
  }

  /**
   * Status callback for functions discoverResource(), addResource() and removeResource().
   *
   * @param networkId             The networkId to which the device belongs.
   * @param deviceNodeId          The deviceNodeId which uniquely identifies
   *                              devices.
   * @param status                The status of the command.
   * @param impactedResourcesEuiArr The resourceEuis of the resources which got effected
   * @param nativeNetworkContext  The Context which was passes during connect() of
   *                              a network.
   * @param nativeCommandContext  The Context Which is passed along the command.
   */
  @Override
  protected void deviceManagementStatusCallback(String networkId, long deviceNodeId, int status,
                                                String[] impactedResourcesEuiArr,
                                                Object nativeNetworkContext,
                                                Object nativeCommandContext) {

    Object commandContext = null;
    Command<Device.CommandId> command = null;
    Device device = Utils.getDevice(networkId, deviceNodeId);
    Resource[] resourcesImpacted =
        (null == impactedResourcesEuiArr) ? null : new Resource[impactedResourcesEuiArr.length];


    if (null == device) {
      throw new RuntimeException("device missing during device management callback");
    }

    if (nativeCommandContext != null) {
      command = cast(((Context) nativeCommandContext).sdkContext);
      commandContext = ((Context) nativeCommandContext).developerContext;
    }

    if (null != impactedResourcesEuiArr) {
      for (int i = 0; i < impactedResourcesEuiArr.length; i++) {
        resourcesImpacted[i] =
            Utils.getResource(networkId, deviceNodeId, impactedResourcesEuiArr[i]);
      }
    }

    Command.State commandStatus = Command.State.getEnum(status);
    CommandResponse<Device.CommandId> response = CommandResponse.createResponse(command, null)
        .setState(commandStatus);

    if (commandContext instanceof Device.DeviceManagementStatusListener) {
      ((Device.DeviceManagementStatusListener) commandContext).onStatusChanged(response,
          resourcesImpacted,
          Command.State.SUCCESS == commandStatus ? null : new StateException(commandStatus));
    }

    multiplexer.deviceManagementStatusCallback(device, response, resourcesImpacted);
  }

  /**
   * InfoResponse callback is issued to determine the status of a sendInfoResponse call.
   *
   * @param status        The value showing the status.
   * @param nativeContext The context passed along with the sendInfoResponse().
   * @see Device#sendInfoResponse(InfoResponse, Device.InfoResponseStatusListener)
   */
  @Override
  protected void infoResponseStatusCallback(int status, Object nativeContext) {

    InfoResponse infoResponse;
    Device.InfoResponseStatusListener listener;
    Context context = (Context) nativeContext;

    if (null == context) {
      Log.e(TAG, "sdk error: null sdk context in infoResponseStatusCallback");
      return;
    }

    infoResponse = (InfoResponse) context.sdkContext;
    listener = (Device.InfoResponseStatusListener) context.developerContext;

    if (null == infoResponse) {
      Log.e(TAG, "sdk error: null info response in infoResponseStatusCallback");
      return;
    }

    infoResponse.state = Command.State.getEnum(status);

    if (null != listener) {
      listener.onInfoResponseStatus(infoResponse,
          Command.State.SUCCESS == infoResponse.state ? null :
              new StateException(infoResponse.state));
    }

    multiplexer.infoResponseStatusCallback(infoResponse.state, infoResponse);
  }

  /**
   * This callback is issued when an addResources or similar call needs
   * additional information to be passed. This callback must be responded with sendInfoResponse()
   * which is a reply to this callback.
   *
   * @param networkId            The networkId to which the device belongs.
   * @param requestNodeId        The ID of the node to which needs additional
   *                             info to be sent.
   * @param requestId            The requestId which must be addressed to.
   * @param infoRequestJson      The stringified json of infoRequest
   * @param nativeNetworkContext The context which is passed with connect()
   * @param nativeCommandContext The context which is passed during
   *                             sendResourceCommand().
   *
   */
  @Override
  protected void infoRequestCallback(String networkId, long requestNodeId, long responseNodeId,
                                     long requestId, long cmdSeqNum,
                                     String infoRequestJson, Object nativeNetworkContext,
                                     Object nativeCommandContext) {

    InfoRequest infoRequest;
    Object commandContext = null;

    if (null != nativeCommandContext) {
      commandContext = ((Context) nativeCommandContext).developerContext;
    }

    infoRequest = Command.GSON_BUILDER.create().fromJson(infoRequestJson, InfoRequest.class);

    infoRequest.requestId = requestId;
    infoRequest.cmdSeqNum = cmdSeqNum;

    Log.d(TAG, "requestNodeId: " + requestNodeId + ", responseNodeId: " + responseNodeId);

    infoRequest.deviceNodeId = requestNodeId;
    infoRequest.networkId = networkId;

    if (commandContext instanceof Device.InfoRequestListener) {
      Device.InfoRequestListener listener = (Device.InfoRequestListener) commandContext;
      listener.onInfoRequest(infoRequest);
    }

    multiplexer.infoRequestCallback(infoRequest);
  }

  /**
   * This callback is invoked after {@link Device#sendInfoRequest} is called.
   *
   * @param networkId            The network Id of the device on which infoRequest is sent
   * @param requestNodeId        The nodeId which was requested for info
   * @param requestId            The id which is passed during infoRequest
   * @param infoResponseJson     The stringified form of infoRequest
   * @param nativeNetworkContext The context that has been passed with connect() of network
   * @param nativeCommandContext The context that has been passed with infoRequest
   */
  @Override
  protected void infoResponseCallback(String networkId, long requestNodeId, long requestId,
                                      long cmdSeqNum,
                                      String infoResponseJson, Object nativeNetworkContext,
                                      Object nativeCommandContext) {

    Device.InfoRequestStatusListener listener;
    InfoResponse infoResponse;
    InfoRequest request;

    request = (InfoRequest) ((Context) nativeCommandContext).sdkContext;
    listener = (Device.InfoRequestStatusListener) ((Context) nativeCommandContext).developerContext;

    infoResponse = Command.GSON_BUILDER.create().fromJson(infoResponseJson, InfoResponse.class);

    infoResponse.networkId = networkId;
    infoResponse.setRequestNodeId(requestNodeId);
    infoResponse.setInfoRequestId(requestId);
    infoResponse.setCmdSeqNum(cmdSeqNum);

    if (null != listener) {
      listener.onInfoResponse(infoResponse);
    }

    multiplexer.infoResponseCallback(request, infoResponse);
  }

  /**
   * This callback is invoked after {@link Device#sendInfoRequest} is called.
   *
   * @param status        The status of infoRequest that was sent
   * @param nativeContext The context which was passed with infoRequest
   * @brief
   */
  @Override
  protected void infoRequestStatusCallback(int status, Object nativeContext) {
    Context context = (Context) nativeContext;
    InfoRequest request = (InfoRequest) context.sdkContext;
    Device.InfoRequestStatusListener listener =
        (Device.InfoRequestStatusListener) context.developerContext;

    request.state = Command.State.getEnum(status);

    if (null != listener) {
      listener.onInfoRequestStatus(request,
          Command.State.SUCCESS == request.state ? null : new StateException(request.state));
    }

    multiplexer.infoRequestStatusCallback(request.state, request);
  }

  /**
   * This function is used to show messages to user,
   * before infoRequestCb or even at any point of time.
   *
   * @param message              The message that has to be shown.
   * @param nativeNetworkContext The context that has been passed during
   *                             connect().
   * @param nativeCommandContext The context which is passed during
   *                             sendResourceCommand().
   */
  @Override
  protected void messageCallback(String title, String message, int messageType,
                                 Object nativeNetworkContext, Object nativeCommandContext) {
    Object listener = null;

    if (null != nativeCommandContext) {
      listener = ((Context) nativeCommandContext).developerContext;
    }

    if (listener instanceof Device.MessageListener) {
      ((Device.MessageListener) listener).onMessage(title, message,
          MessageType.getEnum(messageType));
    }

    multiplexer.messageCallback(title, message, MessageType.getEnum(messageType));
  }

  /**
   * A callback advertising the available resources of an extendable device.
   * This callback is issued after discoverResources() of an extendable device.
   *
   * @param resourceEui          The resource name which is present.
   * @param resourceName         the name of the resource.
   * @param protocol             The protocol which the resource is using
   * @param nativeNetworkContext The context that was passed during connect().
   */
  @Override
  protected void advertiseResourceCallback(String networkId, long deviceNodeId, String resourceEui,
                                           String resourceName, int protocol,
                                           Object nativeNetworkContext) {
    Device parentDevice = Utils.addMissingDevice(networkId, deviceNodeId);

    Resource resourceAd = Factory.createAdvertResource(resourceEui, parentDevice);

    resourceAd.internalSetName(resourceName);
    resourceAd.internalSetProtocol(RadioProtocol.getEnum(protocol));

    multiplexer.advertiseResourceCallback(resourceAd);
  }

  /**
   * This callback is issued after successful inclusion which was done by
   * calling <b>addResource()</b> method in {@link Device} class.
   * NOTE: during resource inclusion. The lower layer will trigger individual resourceSummary,
   * capability and attribute CBs. So, using {@link Utils#addMissingResource} is just a formality.
   *
   * @param networkId            The network to which the resource belongs.
   * @param deviceNodeIds        The device to which the resource belongs.
   * @param resourceEuis         The unique Id of the resource.
   * @param nativeNetworkContext The context that was passed during addResource().
   */
  @Override
  protected void resourceIncludedCallback(String networkId, long[] deviceNodeIds,
                                          String[] resourceEuis, Object nativeNetworkContext) {

    Object networkContext = null;

    if (null != nativeNetworkContext) {
      networkContext = ((Context) nativeNetworkContext).developerContext;
    }

    ArrayList<Resource> resources = new ArrayList<>(deviceNodeIds.length);
    for (int i = 0; i < deviceNodeIds.length; i++) {
      Resource resource =
          Utils.addMissingResource(networkId, deviceNodeIds[i], resourceEuis[i], networkContext);
      resources.add(resource);
    }

    multiplexer.resourceIncludedCallback(resources);
  }

  /**
   * This callback is issued after successful exclusion which was done by calling
   * {@link Device#removeResource}.
   *
   * @param networkId            The Id of network to which the resource belongs.
   * @param deviceNodeId         The Id of device to which the resource belongs.
   * @param resourceEui          The unique Id of the resource.
   * @param nativeNetworkContext The context that has been passed during the
   *                             removeResource().
   */
  @Override
  protected void resourceExcludedCallback(String networkId, long deviceNodeId, String resourceEui,
                                          Object nativeNetworkContext) {

    Device parentDevice = Utils.getDevice(networkId, deviceNodeId);

    Resource resource = Utils.getResource(networkId, deviceNodeId, resourceEui);

    if (null == resource) {
      return;
    }

    multiplexer.resourceExcludedCallback(resource);

    // removing from the resourceMap and zoneMap
    // anyways, a ZoneInfoCB should do the job. which doesn't trigger in case of default-zone
    parentDevice.internalRemoveResource(resourceEui);

    resource.internalRemoveResource();

    if (null != resource.getParentZone()) {
      resource.getParentZone().internalRemoveResource(resource);
    }
  }

  /**
   * A callback to indicate the status of the tunnel and the port over
   * which the tunnel is open.This callback is issued when the tunnel status is changed.
   *
   * @param tunnelHandle        The pointer to the tunnel
   * @param status              The status of the tunnel
   * @param port                The port which the tunnel is using
   * @param nativeTunnelContext The context passed with tunnelOpen()
   */
  @Override
  protected void tunnelStatusCallback(long tunnelHandle, int status, int port,
                                      Object nativeTunnelContext) {

    CapabilityTunnel tunnel = (CapabilityTunnel) ((Context) nativeTunnelContext).sdkContext;
    CapabilityTunnel.TunnelStatusListener listener =
        (CapabilityTunnel.TunnelStatusListener) ((Context) nativeTunnelContext).developerContext;

    CapabilityTunnel.StatePort statePort = tunnel.getTunnelHandleMap().get(tunnelHandle);

    if (null == listener) {
      Log.d(TAG, "tunnelStatusCallback: listener = null");
    }

    if (null == statePort) {
      statePort = new CapabilityTunnel.StatePort();
      tunnel.getTunnelHandleMap().put(tunnelHandle, statePort);
    }

    statePort.port = port;
    statePort.state = CapabilityTunnel.State.getEnum(status);

    if (CapabilityTunnel.State.CLOSED == statePort.state
        || CapabilityTunnel.State.TIMEOUT == statePort.state
        || CapabilityTunnel.State.OPEN_FAILED == statePort.state) {

      tunnel.getTunnelHandleMap().remove(tunnelHandle);
    }

    if (null != listener) {
      try {
        listener.onStatusChanged(tunnelHandle, statePort.port, statePort.state);
      } catch (Exception e) {
        errorCallback(e);
      } catch (Throwable tr) {
        errorCallback(new Exception(tr));
      }
    }

    multiplexer.tunnelStatusCallback(tunnel, tunnelHandle, statePort.state, statePort.port);
  }

  /**
   * This callback is issued when the zones are modified, it notifies the zone information.
   *
   * @param networkId     The network to which the zone belongs to.
   * @param zoneId        The unique identifier of the zone
   * @param zoneName      The name of the zone
   * @param deviceNodeIds The deviceNodeIds to which resources belong to.
   * @param resourceEuis  The resourceEui of the respective resources.
   * @param resourceNames The names of the corresponding resources.
   * @param context       The networkContext.
   */
  @Override
  protected void zoneInfoCallback(boolean trigger, String networkId, int zoneId, String zoneName,
                                  long[] deviceNodeIds,
                                  String[] resourceEuis, String[] resourceNames, Object context) {

    long timeTaken = System.currentTimeMillis();
    Log.d(TAG, "zoneInfoCallback: started at: " + timeTaken);

    Object developerContext = null;
    List<Resource> resourcesToRemove = null;
    HashSet<Zone> effectedZones = new HashSet<>();
    HashSet<Resource> expectedResources = new HashSet<>();

    if (null != context) {
      developerContext = ((Context) context).developerContext;
    }

    Zone zone;
    Network network = Utils.addMissingNetwork(networkId);

    if (!network.containsZone(zoneId)) {
      network.internalAddZone(zone = Factory.createZone(zoneId, network));
    } else {
      zone = network.getZone(zoneId);
    }

    zone.internalSetName(zoneName);

    for (int i = 0; i < deviceNodeIds.length; i++) {
      Zone resourceParentZone;
      Resource resource =
          Utils.addMissingResource(networkId, deviceNodeIds[i], resourceEuis[i], zone,
              developerContext);
      expectedResources.add(resource);

      if (null != resourceNames[i]) {
        resource.internalSetName(resourceNames[i]);
      }

      resourceParentZone = resource.getParentZone();

      if (!zone.equals(resourceParentZone)) {
        resourceParentZone.internalRemoveResource(resource);
        effectedZones.add(resourceParentZone);

        resource.internalSetParentZone(zone);
      }

      if (!zone.containsResource(resource)) {
        zone.internalAddResource(resource);
      }
    }

    for (Resource resource : zone) {
      if (expectedResources.contains(resource)) {
        continue;
      }

      if (null == resourcesToRemove) {
        resourcesToRemove = new ArrayList<>();
      }

      resourcesToRemove.add(resource);
    }

    // not setting the parent zone of these resources. So, there can be a situation
    // where the resource:
    //  R says the parent zone as X
    //  and
    //  resource set of X doesn't contain R

    if (null != resourcesToRemove) {
      for (Resource resource : resourcesToRemove) {
        Log.d(TAG, "removing id: " + resource.getId());
        zone.internalRemoveResource(resource);
      }
    }

    if (trigger) {
      for (Zone effectedZone : effectedZones) {
        multiplexer.zoneInfoCallback(effectedZone);
      }
    }

    zone.internalMarkAsReady();

    timeTaken = System.currentTimeMillis() - timeTaken;
    Log.d(TAG, "zoneInfoCallback: completed in: " + timeTaken + "ms");

    if (trigger) {
      multiplexer.zoneInfoCallback(zone);
    }
  }

  /**
   * This callback is issued when a zone is deleted.
   *
   * @param networkId The networkId of network to which the zone belongs.
   * @param zoneId    The zoneId of the zone.
   * @param context   The context that has been passed with {@link Network#connect()} call.
   */
  @Override
  protected void zoneDeletedCallback(String networkId, int zoneId, Object context) {
    Network network = Utils.getNetwork(networkId);
    if (null == network) {
      return;
    }

    if (network.containsZone(zoneId)) {
      network.internalRemoveZone(zoneId);
    }

    multiplexer.zoneDeletedCallback(network, zoneId);
  }

  /**
   * A function to notify the scene information.
   *
   * @param networkId        The networkId to which the scene belongs to
   * @param sceneId          The unique identifier of the scene
   * @param sceneName        The name of the scene
   * @param metadata         The metadata of the scene
   * @param resourceCommands The commands which the scene executes
   * @param context          The context passed with {@link Network#connect()} call
   */
  @Override
  protected void sceneInfoCallback(boolean trigger, String networkId, int sceneId, String sceneName,
                                   String metadata, String[] resourceCommands, Object context) {

    long timeTaken = System.currentTimeMillis();
    Log.d(TAG, "sceneInfoCallback: started at: " + timeTaken);

    Scene scene;
    HashMap<Integer, ResourceAction> backupActionMap;
    HashMap<Integer, ResourceAction> resourceActionMap = new HashMap<>();
    Gson gson = Command.GSON_BUILDER.create();

    Network parent = Utils.addMissingNetwork(networkId);

    if (parent.containsScene(sceneId)) {
      scene = parent.getScene(sceneId);
    } else {
      parent.internalAddScene(scene = Factory.createScene(sceneId, parent));
    }

    backupActionMap = new HashMap<>(scene.getResourceActionMap());

    scene.internalSetMetadata(metadata);
    scene.internalSetName(sceneName);

    for (String resourceCommand : resourceCommands) {
      JsonObject jsonCommand = JsonParser.parseString(resourceCommand).getAsJsonObject();
      jsonCommand.addProperty(Constants.NETWORK_ID, networkId);

      ResourceAction resourceAction = gson.fromJson(jsonCommand, ResourceAction.class);

      if (null == resourceAction) {
        Log.d(TAG, "cannot form resourceAction for: " + sceneId);
        Log.d(TAG, "resourceActionJson: " + jsonCommand);
        continue;
      }

      resourceActionMap.put(resourceAction.getId(), resourceAction);
    }

    // updating the same resourceAction object instead of using a new resourceAction
    // this loop will also remove resourceActions from backupMap once added
    for (ResourceAction action : resourceActionMap.values()) {
      if (backupActionMap.containsKey(action.getId())) {
        backupActionMap.remove(action.getId());
        scene.internalUpdateResourceAction(action);
      } else {
        scene.internalAddResourceAction(action);
      }
    }

    for (ResourceAction action : backupActionMap.values()) {
      scene.internalRemoveResourceAction(action.getId());
    }

    scene.internalMarkAsReady();

    timeTaken = System.currentTimeMillis() - timeTaken;
    Log.d(TAG, "sceneInfoCallback: completed in: " + timeTaken + "ms");

    if (trigger) {
      multiplexer.sceneInfoCallback(scene);
    }
  }

  /**
   * A function notifying that the scene is deleted.
   *
   * @param networkId The networkId to which the scene belongs to
   * @param sceneId   The uniqueId of the scene
   * @param context   The context passed with {@link Network#connect()} call of the network
   */
  @Override
  protected void sceneDeletedCallback(String networkId, int sceneId, Object context) {
    Network parent = Utils.getNetwork(networkId);
    if (null == parent) {
      return;
    }

    parent.internalRemoveScene(sceneId);
    multiplexer.sceneDeletedCallback(parent, sceneId);
  }

  /**
   * A function to notify the scene information.
   *
   * @param networkId                The networkId to which the rule belongs to
   * @param ruleId                   The unique identifier of the rule
   * @param ruleName                 The name of the rule
   * @param resourceConditions       Array of conditions of resource for which rule should trigger
   * @param scheduleConditions       Array os Schedules decided by user for rule to trigger
   * @param resourceActions          Array of actions on the resource decided by the user
   * @param sceneActions             Array of actions on the scene decided by the user
   * @param nativeContext            The context passed with {@link Network#connect()} call
   */
  @Override
  protected void ruleInfoCallback(boolean trigger, String networkId, int ruleId, String ruleName,
                                  String[] resourceConditions, String[] scheduleConditions,
                                  String[] resourceActions, String[] sceneActions,
                                  Object nativeContext) {

    long timeTaken = System.currentTimeMillis();
    Log.d(TAG, "ruleInfoCallback: started at: " + timeTaken);

    Gson gson = Command.GSON_BUILDER.create();
    Rule rule;
    Network parent;
    ArrayList<ResourceAction> resourceActionList = new ArrayList<>(resourceActions.length);
    ArrayList<Scene> sceneActionList = new ArrayList<>(sceneActions.length);
    ArrayList<ResourceCondition> resourceConditionList = new ArrayList<>(resourceConditions.length);
    ArrayList<ScheduleCondition> scheduleConditionList = new ArrayList<>(scheduleConditions.length);

    parent = Utils.addMissingNetwork(networkId);

    if (null == (rule = parent.getRule(ruleId))) {
      parent.internalAddRule(rule = Factory.createRule(ruleId, parent));
    }

    for (String resourceActionString : resourceActions) {
      try {
        JsonObject resourceActionJson =
            JsonParser.parseString(resourceActionString).getAsJsonObject();
        resourceActionJson.addProperty(Constants.NETWORK_ID, networkId);

        ResourceAction resourceAction = gson.fromJson(resourceActionJson, ResourceAction.class);
        resourceActionList.add(resourceAction);
      } catch (Exception e) {
        errorCallback(e);
      }
    }

    for (String sceneAction : sceneActions) {
      try {
        JsonObject sceneActionJson = JsonParser.parseString(sceneAction).getAsJsonObject();
        sceneActionJson.addProperty(Constants.NETWORK_ID, networkId);

        // TODO: find a better way to handle this
        Scene scene = gson.fromJson(sceneActionJson, Scene.class);
        sceneActionList.add(scene);
      } catch (Exception e) {
        errorCallback(e);
      }
    }

    for (String resourceConditionString : resourceConditions) {
      try {
        JsonObject resourceConditionJson =
            JsonParser.parseString(resourceConditionString).getAsJsonObject();
        resourceConditionJson.addProperty(Constants.NETWORK_ID, networkId);

        ResourceCondition resourceCondition =
            gson.fromJson(resourceConditionJson, ResourceCondition.class);
        resourceConditionList.add(resourceCondition);
      } catch (Exception e) {
        errorCallback(e);
      }
    }

    for (String scheduleConditionString : scheduleConditions) {
      try {
        ScheduleCondition scheduleCondition =
            gson.fromJson(scheduleConditionString, ScheduleCondition.class);
        scheduleConditionList.add(scheduleCondition);
      } catch (Exception e) {
        errorCallback(e);
      }
    }

    rule.internalSetName(ruleName);

    rule.internalClearAddResourceConditions(resourceConditionList);
    rule.internalClearAddResourceActions(resourceActionList);
    rule.internalClearAddSceneActions(sceneActionList);
    rule.internalClearAddScheduleConditions(scheduleConditionList);

    rule.internalMarkAsReady();

    timeTaken = System.currentTimeMillis() - timeTaken;
    Log.d(TAG, "ruleInfo: completed in: " + timeTaken + "ms");

    if (trigger) {
      multiplexer.ruleInfoCallback(rule);
    }
  }

  /**
   * A function notifying that the rule is deleted.
   *
   * @param networkId The networkId to which the rule belongs to
   * @param ruleId   The uniqueId of the rule
   * @param nativeContext   The context passed with {@link Network#connect()} call of the network
   */
  @Override
  protected void ruleDeletedCallback(String networkId, int ruleId, Object nativeContext) {
    Network network = Utils.getNetwork(networkId);

    if (null == network) {
      return;
    }

    network.internalRemoveRule(ruleId);
    multiplexer.ruleDeletedCallback(network, ruleId);
  }

  /**
   * This function is used to get the accessTokens of the current instance.
   *
   * @param accessToken The stringified JSON Access Token
   * @param status      The status of the response
   * @param context     The context that has been passed along with the getAccessToken()
   */
  @Override
  protected void accessTokenCallback(String accessToken, int status, Object context) {
    CocoClient.AccessTokensListener tokensListener = null;
    Command.State commandStatus = Command.State.getEnum(status);

    if (null != context) {
      tokensListener = (CocoClient.AccessTokensListener) ((Context) context).developerContext;
    }

    if (null != tokensListener) {
      tokensListener.onResponse(accessToken,
          Command.State.SUCCESS == commandStatus ? null : new StateException(commandStatus));
    }

    platformInterface.accessTokensCallback(accessToken, commandStatus);
  }

  /**
   * This function is used to get the networkManagementStatus of commands sent.
   *
   * @param networkId      The networkId to which the command is sent
   * @param commandStatus  The status of the command that has been sent
   * @param commandContext The context that's passed with the command
   * @param networkContext The context that has been passed with {@link Network#connect()}
   */
  @Override
  protected void networkManagementStatusCallback(String networkId, int commandStatus, int errorCode,
                                                 String errorMessage, String fieldName,
                                                 Object commandContext,
                                                 Object networkContext) {
    Command<Network.CommandId> command = null;
    Network.NetworkManagementStatusListener listener = null;
    CommandResponse<Network.CommandId> response;
    Command.State commandState = Command.State.getEnum(commandStatus);

    if (null != commandContext) {
      command = cast(((Context) commandContext).sdkContext);
      listener =
          (Network.NetworkManagementStatusListener) ((Context) commandContext).developerContext;
    }

    response = CommandResponse.createResponse(command, null)
        .setState(commandState)
        .setError(new CommandResponse.Error(errorCode, errorMessage, fieldName));

    if (null != listener) {
      listener.onStatusChanged(response,
          Command.State.SUCCESS == commandState ? null : new StateException(commandState));
    }

    multiplexer.networkManagementCommandCallback(Utils.addMissingNetwork(networkId), response);
  }

  /**
   * A function to determine if the node is online or offline.
   *
   * @param networkId      The networkId to which the node belongs to.
   * @param nodeId         The nodeId whose status has changed.
   * @param nativeNodeType The type of the node (device / client / NMN).
   * @param isOnline       A flag showing if the node is online or offline
   * @param nativeContext  The context that's passed with {@link Network#connect()}
   */
  @Override
  protected void nodeConnectionStatusCallback(String networkId, long nodeId, int nativeNodeType,
                                              boolean isOnline, Object nativeContext) {
    Device device = Utils.getDevice(networkId, nodeId);
    NodeType nodeType = NodeType.getEnum(nativeNodeType);

    if (null != device) {
      device.internalSetReady(isOnline);
    }

    multiplexer.nodeConnectionStatusCallback(Utils.getNetwork(networkId), nodeId, nodeType,
        isOnline);
  }

  /**
   * This function is used to inform errors during callbacks,
   * this possibly mean something near fatal(yet recoverable) has occurred.
   *
   * @param exception The exception which has occurred
   */
  @Override
  protected void errorCallback(Exception exception) {
    try {
      exception.printStackTrace();
      multiplexer.errorCallback(exception);
    } catch (Throwable tr) {
      Log.w(TAG, "wth: error in errorCallback", tr);
    }
  }

  /**
   * This function is used to deliver the whole data of the network.
   *
   * @param networkId The id of network whose data has been received
   * @param deviceIds The ids of devices present in the network
   * @param sceneIds  The ids of scenes present in the network
   * @param ruleIds   The ids of rules present in the network
   * @param zoneIds   The ids of the zones present in the network
   */
  @Override
  protected void networkDataCallback(String networkId, int[] resTempZoneIds, String[] resTempIds,
                                     String[] resTempNames, String[] resTempIcons, long[] deviceIds,
                                     int[] zoneIds, int[] sceneIds, int[] ruleIds, Object context) {

    long timeTaken = System.currentTimeMillis();
    Log.d(TAG, "networkDataCallback: started at: " + timeTaken);

    // Just to be sure with not deleting default zone
    zoneIds = getZonesIdsWithDefaultIncluded(zoneIds);

    Network network = Utils.addMissingNetwork(networkId);
    List<Network.ResourceTemplate> resourceTemplates = new ArrayList<>();

    int resTempArrCnt = resTempIds.length;

    for (int i = 0; i < resTempArrCnt; i++) {
      resourceTemplates.add(
          new Network.ResourceTemplate(resTempIds[i],
              resTempZoneIds[i],
              resTempNames[i],
              resTempIcons[i]));
    }

    network.internalSetResourceTemplates(resourceTemplates);

    List<Long> removeDeviceIds = null;
    List<Integer> removeZoneIds = null;
    List<Integer> removeSceneIds = null;
    List<Integer> removeRuleIds = null;
    List<Resource> removeResources = null;

    // 1. add any items which are present in callback but, not in network
    // NOTE: This step is purely pedantic. Because, deviceInfoCB will be triggered by this point
    // on all the devices.
    for (long deviceId : deviceIds) {
      if (network.containsDevice(deviceId)) {
        continue;
      }

      Device device = Utils.addMissingDevice(networkId, deviceId);

      if (null == device) {
        Log.w(TAG,
            "cannot find device for networkId: " + networkId + ", deviceNodeId: " + deviceId);
        continue;
      }

      network.internalAddDevice(device);
    }

    for (int zoneId : zoneIds) {
      if (network.containsZone(zoneId)) {
        continue;
      }

      Zone zone = Utils.getZone(networkId, zoneId);

      if (null == zone) {
        Log.w(TAG, "cannot find zone for network: " + networkId + ", zoneId: " + zoneId);
        continue;
      }

      network.internalAddZone(zone);
    }

    for (int sceneId : sceneIds) {
      if (network.containsScene(sceneId)) {
        continue;
      }

      Scene scene = Utils.getScene(networkId, sceneId);

      if (null == scene) {
        Log.w(TAG, "cannot find scene for network: " + networkId + ", sceneId: " + sceneId);
        continue;
      }

      network.internalAddScene(scene);
    }

    for (int ruleId : ruleIds) {
      if (network.containsRule(ruleId)) {
        continue;
      }

      Rule rule = Utils.getRule(networkId, ruleId);

      if (null == rule) {
        Log.w(TAG, "cannot find rule for network: " + networkId + ", ruleId: " + ruleId);
        continue;
      }

      network.internalAddRule(rule);
    }

    // 2. mark items which are present in network. But, not in list
    if (network.getDeviceMap().size() != deviceIds.length) {

      HashSet<Long> deviceIdSet = new HashSet<>();

      // 2.1. populate sets for o(1) access

      for (long deviceId : deviceIds) {
        deviceIdSet.add(deviceId);
      }

      for (Device device : network.getDeviceIterable()) {
        if (deviceIdSet.contains(device.getId())) {
          continue;
        }

        for (Resource r : device) {
          Zone parentZone = r.getParentZone();

          r.internalRemoveResource();
          device.internalRemoveResource(r.getId());

          if (null == parentZone) {
            Log.w(TAG, "zone not found for resource: " + r);
            continue;
          }

          parentZone.internalRemoveResource(r);
        }

        if (null == removeDeviceIds) {
          removeDeviceIds = new ArrayList<>();
        }

        removeDeviceIds.add(device.getId());
      }
    }

    if (network.getZoneMap().size() != zoneIds.length) {

      HashSet<Integer> zoneIdSet = new HashSet<>();

      for (int zoneId : zoneIds) {
        zoneIdSet.add(zoneId);
      }

      for (Zone zone : network.getZoneIterable()) {
        if (zoneIdSet.contains(zone.getId())) {
          continue;
        }

        if (null == removeZoneIds) {
          removeZoneIds = new ArrayList<>();
        }

        removeZoneIds.add(zone.getId());
      }
    }

    if (network.getSceneMap().size() != sceneIds.length) {
      HashSet<Integer> sceneIdSet = new HashSet<>();

      for (int sceneId : sceneIds) {
        sceneIdSet.add(sceneId);
      }

      for (Scene scene : network.getSceneIterable()) {
        if (sceneIdSet.contains(scene.getId())) {
          continue;
        }

        if (null == removeSceneIds) {
          removeSceneIds = new ArrayList<>();
        }

        removeSceneIds.add(scene.getId());
      }
    }

    if (network.getRuleMap().size() != ruleIds.length) {
      HashSet<Integer> ruleIdSet = new HashSet<>();

      for (int ruleId : ruleIds) {
        ruleIdSet.add(ruleId);
      }

      for (Rule rule : network.getRuleIterable()) {
        if (ruleIdSet.contains(rule.getId())) {
          continue;
        }

        if (null == removeRuleIds) {
          removeRuleIds = new ArrayList<>();
        }

        removeRuleIds.add(rule.getId());
      }
    }

    // 3. remove items which are marked
    if (null != removeDeviceIds) {
      for (Long deviceId : removeDeviceIds) {

        Device device = network.getDevice(deviceId);

        if (null != device) {
          device.internalRemoveDevice();
        }

        network.internalRemoveDevice(deviceId);
      }
    }

    if (null != removeZoneIds) {
      for (Integer zoneId : removeZoneIds) {
        network.internalRemoveZone(zoneId);
      }
    }

    if (null != removeSceneIds) {
      for (Integer sceneId : removeSceneIds) {
        network.internalRemoveScene(sceneId);
      }
    }

    if (null != removeRuleIds) {
      for (Integer ruleId : removeRuleIds) {
        network.internalRemoveRule(ruleId);
      }
    }

    // 4. after clean up if the device is not present in any of the zones.
    //     Remove it from the device
    for (Device d : network) {
      for (Resource r : d) {
        Zone z = r.getParentZone();

        if (null == z) {
          continue;
        }

        if (z.containsResource(r)) {
          continue;
        }

        if (null == removeResources) {
          removeResources = new ArrayList<>();
        }

        removeResources.add(r);
      }
    }

    if (null != removeResources) {
      for (Resource r : removeResources) {
        Device d = r.getParent();

        Log.d(TAG, "removing res: " + d.getId() + " : " + r.getId());

        d.internalRemoveResource(r.getId());
        r.internalRemoveResource();
      }
    }

    timeTaken = System.currentTimeMillis() - timeTaken;
    Log.d(TAG, "networkDataCallback: completed in: " + timeTaken + "ms");

    multiplexer.networkDataCallback(network);
  }

  /**
   * This function is used deliver the status of the channel connection.
   *
   * @param streamHandle   The pointer to the streaming struct
   * @param channelPort    The port on which the status is delivered
   * @param status         The status of the connection
   * @param networkContext The Context which has been passed with {@link Network#connect()}
   * @param streamContext  The context which has been passed with {@link CapabilityTunnel#open}
   */
  @Override
  protected void mediaStreamStatusCallback(long streamHandle, int channelPort, int status,
                                           Object networkContext, Object streamContext) {
    CapabilityMediaStreaming.MediaStreamListener mediaStreamCallback = null;

    if (null != streamContext) {
      mediaStreamCallback =
          (CapabilityMediaStreaming.MediaStreamListener) ((Context) streamContext).developerContext;
    }

    if (null == mediaStreamCallback) {
      throw new IllegalStateException(
          "null context has been received in mediaStreamStatusCallback, context: " + streamContext);
    }

    Log.d(TAG, "mediaStreamStatusCallback: networkContext: " + networkContext);

    try {
      mediaStreamCallback.onStatusChanged(streamHandle, channelPort,
          CapabilityMediaStreaming.Status.getEnum(status));
    } catch (Throwable tr) {
      multiplexer.errorCallback(tr);
    }
  }

  /**
   * This function is used to deliver the snapshot file.
   *
   * @param filePath        The location of the file
   * @param status          The status of the snapshot
   * @param snapshotContext The context passed with {@link CapabilitySnapshot#captureSnapshot}
   */
  @Override
  protected void snapshotReceiveCallback(String filePath, int status, Object snapshotContext) {
    CapabilitySnapshot.SnapshotListener listener = null;

    if (null != snapshotContext) {
      listener = cast(((Context) snapshotContext).developerContext);
    }

    if (null != listener) {
      listener.onSnapshotCaptured(filePath, status);
    } else {
      Log.d(TAG, "snapshotReceiveCallback: listener is null");
    }
  }

  /**
   * This function is used deliver the data via the channel.
   *
   * @param streamHandle   The pointer to the streaming struct
   * @param channelPort    The port on which the status is delivered
   * @param frameIndex     The index of this frame
   * @param frameType      The type of this frame (KEY or NONE)
   * @param frameDuration  The duration of the frame
   * @param framePts       The presentation timestamp of the frame
   * @param data           The data which has been transmitted by the other end
   * @param networkContext The Context which has been passed with {@link Network#connect()}
   * @param streamContext  The context which has been passed with {@link CapabilityTunnel#open}
   */
  @Override
  protected void mediaStreamReceiveCallback(long streamHandle, int channelPort, long frameIndex,
                                            int frameType, long frameDuration, long framePts,
                                            ByteBuffer data, Object networkContext,
                                            Object streamContext) {
    CapabilityMediaStreaming.MediaStreamListener mediaStreamCallback = null;

    if (null != streamContext) {
      mediaStreamCallback =
          (CapabilityMediaStreaming.MediaStreamListener) ((Context) streamContext).developerContext;
    }

    if (null == mediaStreamCallback) {
      throw new IllegalStateException(
          "null context has been received in mediaStreamStatusCallback, context: " + streamContext);
    }

    Log.d(TAG, "mediaStreamReceiveCallback: networkContext: " + networkContext);

    try {
      mediaStreamCallback.onDataReceived(streamHandle, channelPort, frameIndex, frameType,
          frameDuration, framePts, data);
    } catch (Throwable tr) {
      multiplexer.errorCallback(tr);
    }
  }

  @Override
  protected void receiveDataCallback(long sourceNodeId, String data, Object nativeNetworkContext) {
    Network.SdkContext sdkContext =
        (Network.SdkContext) ((Context) nativeNetworkContext).sdkContext;

    Network network = Utils.getNetwork(sdkContext.getNetworkId());

    multiplexer.receiveDataCallback(network, sourceNodeId, data);
  }

  @Override
  protected void contentInfoCallback(long sourceNodeId, long contentTimestamp, String data,
                                     Object nativeNetworkContext) {
    Network.SdkContext sdkContext =
        (Network.SdkContext) ((Context) nativeNetworkContext).sdkContext;

    Network network = Utils.getNetwork(sdkContext.getNetworkId());

    multiplexer.contentInfoCallback(network, sourceNodeId, contentTimestamp, data);
  }

  @Override
  protected void networkMetadataCallback(String metadata, Object nativeNetworkContext) {
    Network.SdkContext sdkContext =
        (Network.SdkContext) ((Context) nativeNetworkContext).sdkContext;

    Network network = Utils.getNetwork(sdkContext.getNetworkId());

    if (null == network) {
      return;
    }

    network.internalSetMetadata(metadata);

    multiplexer.networkMetadataCallback(network);
  }

  private static int[] getZonesIdsWithDefaultIncluded(int[] zoneIds) {

    for (int zoneId : zoneIds) {
      if (Zone.DEFAULT_ZONE_ID == zoneId) {
        return zoneIds;
      }
    }

    int[] zoneIdsWithDefault = new int[zoneIds.length + 1];

    System.arraycopy(zoneIds, 0, zoneIdsWithDefault, 0, zoneIds.length);

    zoneIdsWithDefault[zoneIds.length] = Zone.DEFAULT_ZONE_ID;

    return zoneIdsWithDefault;
  }

  @SuppressWarnings("unchecked")
  private <T, U> U cast(T t) {
    return (U) t;
  }
}
