/*
 * @author    Krishna, krishnasarma@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.JsonElement;

/**
 * This interface provides default implementation of {@link Creator}.
 */
public interface DefaultCreator extends Creator {

  @Override
  default Network createNetwork(String id) {
    return new Network(id);
  }

  @Override
  default Device createDevice(long deviceNodeId, Network parent) {
    return new Device(deviceNodeId, parent);
  }

  @Override
  default Zone createZone(int zoneId, Network parent) {
    return new Zone(zoneId, parent);
  }

  @Override
  default Scene createScene(int sceneId, Network parent) {
    return new Scene(sceneId, parent);
  }

  @Override
  default Rule createRule(int id, Network parent) {
    return new Rule(id, parent);
  }

  @Override
  default Resource createResource(String eui, Device parentDevice, Zone parentZone) {
    return new Resource(eui, parentDevice, parentZone);
  }

  @Override
  default Resource createAdvertResource(String eui, Device parentDevice) {
    return createResource(eui, parentDevice, null);
  }

  @Override
  default Attribute createAttribute(int id, Capability parent) {
    return new Attribute(id, parent);
  }

  @Override
  default Capability createCapability(int capabilityId, Resource parent) {
    Capability capability;

    switch (Capability.CapabilityId.getEnum(capabilityId)) {
      case ON_OFF_CONTROL:
        capability = new CapabilityOnOff(capabilityId, parent);
        break;
      case LEVEL_CTRL:
        capability = new CapabilityLevel(capabilityId, parent);
        break;
      case COLOR_CTRL:
        capability = new CapabilityColor(capabilityId, parent);
        break;
      case LOCK_CONTROL:
        capability = new CapabilityDoorLock(capabilityId, parent);
        break;
      case ENERGY_METERING:
        capability = new CapabilityEnergyMetering(capabilityId, parent);
        break;
      case MOTION_SENSING:
        capability = new CapabilityMotionSensing(capabilityId, parent);
        break;
      case OCCUPANCY_SENSING:
        capability = new CapabilityOccupancySensing(capabilityId, parent);
        break;
      case CONTACT_SENSING:
        capability = new CapabilityContactSensing(capabilityId, parent);
        break;
      case FLUID_LEVEL_MEASUREMENT:
        capability = new CapabilityFluidLevelSensing(capabilityId, parent);
        break;
      case FIRE_SENSING:
        capability = new CapabilityFireSensing(capabilityId, parent);
        break;
      case TEMPERATURE_MEASUREMENT:
        capability = new CapabilityTemperatureSensing(capabilityId, parent);
        break;
      case ILLUMINANCE_MEASUREMENT:
        capability = new CapabilityIlluminance(capabilityId, parent);
        break;
      case POWER_LEVEL_MEASUREMENT:
        capability = new CapabilityPowerLevels(capabilityId, parent);
        break;
      case TUNNEL_CONTROL:
        capability = new CapabilityTunnel(capabilityId, parent);
        break;
      case REL_HUMIDITY_MEASUREMENT:
        capability = new CapabilityHumiditySensing(capabilityId, parent);
        break;
      case KEY_PRESS_SENSING:
        capability = new CapabilityKeyPressSensing(capabilityId, parent);
        break;
      case WARNING_DEV_CONTROL:
        capability = new CapabilityWarning(capabilityId, parent);
        break;
      case NETWORK_CONFIGURATION:
        capability = new CapabilityNetworkConfiguration(capabilityId, parent);
        break;
      case MEDIA_STREAM:
        capability = new CapabilityMediaStreaming(capabilityId, parent);
        break;
      case MOTOR_CONTROL:
        capability = new CapabilityMotorControl(capabilityId, parent);
        break;
      case STORAGE_CONTROL:
        capability = new CapabilityStorageControl(capabilityId, parent);
        break;
      case STORAGE_CONFIG:
        capability = new CapabilityStorageConfig(capabilityId, parent);
        break;
      case SNAPSHOT:
        capability = new CapabilitySnapshot(capabilityId, parent);
        break;
      case IMAGE_CONTROL:
        capability = new CapabilityImageControl(capabilityId, parent);
        break;
      case REAL_TIME_POS:
        capability = new CapabilityRealTimePosition(capabilityId, parent);
        break;
      case STATIONARY_POS:
        capability = new CapabilityStationaryPosition(capabilityId, parent);
        break;
      case AIR_QUALITY_SENSING:
        capability = new CapabilityAirQualitySensing(capabilityId, parent);
        break;
      case VIBRATION_SENSING:
        capability = new CapabilityVibrationSensing(capabilityId, parent);
        break;
      case WINDOW_COVERING:
        capability = new CapabilityWindowCovering(capabilityId, parent);
        break;
      case HVAC_CONTROL:
        capability = new CapabilityHvacControl(capabilityId, parent);
        break;
      case REMOTE_CONTROL:
        capability = new CapabilityRemoteControl(capabilityId, parent);
        break;
      default:
        capability = new Capability(capabilityId, parent) {

          @Override
          protected Command<? extends CommandId> extendedCreateCommand(int commandId,
                                                                       JsonElement commandParams) {
            return null;
          }
        };
    }

    return capability;
  }
}
