/*
 * @authors   Rohan Sahay, rohansahay@elear.solutions
 *            Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.annotations.SerializedName;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Device model class is a structure of the device present in the {@link Network}.
 */
public class Device implements Iterable<Resource> {
  static {
    assert PowerSource.values().length <= Long.SIZE;
  }

  /**
   * CommandId aids in generation of Command to be sent using
   * {@link #sendDeviceManagementCommand(Command, DeviceManagementStatusListener)}.
   */
  public enum CommandId implements CommandIdInterface {
    DISCOVER_RES,
    ADD_RES,
    REM_RES,
    REBOOT,
    UPD_RES_NAME,
    UPD_RES_META,
    FORCE_REM_RES,
    CONFIGURE_WIFI;

    static CommandId getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    @Override
    public int getInt() {
      return ordinal();
    }
  }

  private static final String TAG = "Device";
  private static final String NOT_AN_EXTENDABLE_DEVICE_MESSAGE =
      "the selected device is not an extendable device to send commands";

  @SerializedName(Constants.DEVICE_NODE_ID)
  private final long deviceNodeId;
  @SerializedName(Constants.DEVICE_NAME)
  private String name;
  @SerializedName(Constants.DEVICE_PSN)
  private String devicePsn;
  @SerializedName(Constants.PRODUCT_NAME)
  private String productName;
  @SerializedName(Constants.MAKE)
  private String make;
  @SerializedName(Constants.MODEL)
  private String model;
  @SerializedName(Constants.FIRMWARE_VERSION)
  private String firmwareVersion;
  @SerializedName(Constants.POWER_SOURCE)
  private PowerSource powerSource;
  @SerializedName(Constants.RECEIVER_TYPE)
  private ReceiverType receiverType;
  @SerializedName(Constants.IS_EXTENDABLE)
  private boolean isExtendable = false;
  @SerializedName(Constants.PROTOCOL_ID_ARR)
  private int[] supportedProtocols;

  private transient Network parent;
  private transient boolean ready = false;
  private transient long supportedProtocolFlags = -1; // assuming support for all by default

  @SerializedName(Constants.DEVICE_RESOURCES_MAP)
  private Map<String, Resource> resourceMap;

  /*
   * @param deviceNodeId The unique id of the device.
   * @param parent       The parent network of the device.
   */
  protected Device(long deviceNodeId, Network parent) {
    this.deviceNodeId = deviceNodeId;
    this.parent = parent;
  }

  /**
   * A function to send commands to device.
   *
   * @param command  Command to be sent to Device.
   * @param listener The listener which will be triggered after the callback
   */
  public void sendDeviceManagementCommand(Command<CommandId> command,
                                          DeviceManagementStatusListener listener) {
    if (CommandId.ADD_RES == command.getCommandId()
        || CommandId.DISCOVER_RES == command.getCommandId()) {

      if (!isExtendable) {
        throw new IllegalArgumentException(NOT_AN_EXTENDABLE_DEVICE_MESSAGE);
      }
    }

    CocoClient.getInstance().getNativeHandler()
        .sendDeviceManagementCommand(this, command, listener);
  }

  public void reboot(DeviceManagementStatusListener listener) {
    sendDeviceManagementCommand(new Command<>(CommandId.REBOOT), listener);
  }

  protected void setResourceName(Resource resource, String name,
                                 DeviceManagementStatusListener listener) {
    sendDeviceManagementCommand(new SetResourceName(resource, name), listener);
  }

  protected void setResourceMetadata(Resource resource, String metadata,
                                     DeviceManagementStatusListener listener) {
    sendDeviceManagementCommand(new SetResourceMetadata(resource, metadata), listener);
  }

  /**
   * This function is used to get the unique identifier of the device.
   *
   * @return long: The unique number that identifies the Devices of the given
   *         network.
   */
  public final long getId() {
    return this.deviceNodeId;
  }

  /**
   * This function is used to get the Name of the device.
   *
   * @return String: The name of the device.
   */
  public String getName() {
    return this.name;
  }

  /**
   * This function is used to indicate that the device is extendable or not.
   *
   * @return boolean: This is a flag that is used to identify extendable devices
   *         from non-extendable
   */
  public boolean isExtendable() {
    return this.isExtendable;
  }

  /**
   * This function is used to get the parent Network of the given device.
   *
   * @return Network: The parent Network of the given device.
   */
  public final Network getParent() {
    return this.parent;
  }

  /**
   * This function is used to return the HashMap that the sdk uses to map
   * the resourceEuis to the Resource Objects.
   * This function can be used to databind the sdk to the application.
   *
   * @return The key being the resourceEui and the value
   *         being the Resource Object. NOTE: This hashmap is for read-only any
   *         misuse may cause the program to misbehave and even crash.
   */
  public Map<String, Resource> getResourceMap() {
    if (null == this.resourceMap) {
      this.resourceMap = new ConcurrentHashMap<>();
    }

    return this.resourceMap;
  }

  /**
   * This function is used to get the resource with the given EUI.
   *
   * @return The Resource
   */
  public final <T extends Resource> T getResource(String resourceEui) {
    return Utils.castUp(getResourceMap().get(resourceEui));
  }

  /**
   *  A function suggesting if the device is made from {@link NativeCallbacks#deviceInfoCallback}
   *  or because of state retrieval.
   *
   * @return boolean: state if the device is dummy or not.
   */
  public boolean isReady() {
    return ready;
  }

  /**
   * This function is used to discover resources of the given device which are present.
   *
   * @param protocol       The protocol which will be used
   * @param listener       The listener which will be triggered after the callback
   */
  public void discoverResources(RadioProtocol protocol, DiscoverResourceListener listener) {
    discoverResources(protocol, Command.defaultTimeOut, listener);
  }

  /**
   * This function is used to discover resources of the given device which are present.
   *
   * @param protocol       The protocol which will be used
   * @param timeout        The timeout for this command
   * @param listener       The listener which will be triggered after the callback
   */
  public void discoverResources(RadioProtocol protocol, long timeout,
                                DiscoverResourceListener listener) {
    if (!isExtendable) {
      throw new IllegalArgumentException(NOT_AN_EXTENDABLE_DEVICE_MESSAGE);
    }

    sendDeviceManagementCommand(new DiscoverResource(protocol).setTimeout(timeout), listener);
  }

  /**
   * This function is used to add the discovered resources to the resource list.
   *
   * @param metadata       The metadata that has to be set on this resource
   * @param parameters     The parameters to be passed while adding
   * @param timeout        The timeout for this command
   * @param listener       The listener which will be triggered after the callback
   */
  public void addZwaveResource(String metadata, List<Parameter<InfoResponse.Value>> parameters,
                               long timeout, AddResourceListener listener) {
    if (!isExtendable) {
      throw new IllegalArgumentException(NOT_AN_EXTENDABLE_DEVICE_MESSAGE);
    }

    sendDeviceManagementCommand(new AddResource(metadata, parameters).setTimeout(timeout),
        listener);
  }

  /**
   * This function is used to add the discovered resources to the resource list.
   *
   * @param resourceEui    The ID of the resource to be added.
   * @param metadata       The metadata that has to be set on this resource
   * @param parameters     The parameters to be passed while adding
   * @param timeout        The timeout for this command
   * @param listener       The listener which will be triggered after the callback
   * @throws RuntimeException Throws exception on communication failures etc..,
   */
  public void addZigbeeResource(String resourceEui, String metadata,
                                List<Parameter<InfoResponse.Value>> parameters, long timeout,
                                AddResourceListener listener) {
    if (!isExtendable) {
      throw new IllegalArgumentException(NOT_AN_EXTENDABLE_DEVICE_MESSAGE);
    }

    sendDeviceManagementCommand(
        new AddResource(resourceEui, metadata, parameters).setTimeout(timeout), listener);
  }

  /**
   * This function is used to add the discovered resources to the resource list.
   *
   * @param metadata       The metadata that has to be set on this resource
   * @param parameters     The parameters to be passed while adding
   * @param timeout        The timeout for this command
   * @param listener       The listener which will be triggered after the callback
   * @throws RuntimeException Throws exception on communication failures etc..,
   */
  public void addTuyaResource(String metadata, List<Parameter<InfoResponse.Value>> parameters,
                              long timeout, AddResourceListener listener) {
    if (!isExtendable) {
      throw new IllegalArgumentException(NOT_AN_EXTENDABLE_DEVICE_MESSAGE);
    }

    sendDeviceManagementCommand(
        new AddResource(RadioProtocol.TUYA, metadata, parameters).setTimeout(timeout), listener);
  }

  /**
   * This function is used to remove the previously added resources.
   *
   * @param resource       The resource to be removed.
   * @param parameters     The parameters to be passed while adding
   * @param listener       The listener which will be triggered after the callback
   * @throws RuntimeException Throws an exception on communication failures and other
   *                          cases like removing resources which haven't been added.
   */
  public void removeResource(Resource resource, List<Parameter<InfoResponse.Value>> parameters,
                             long timeout, RemoveResourceListener listener) {
    if (!isExtendable) {
      throw new IllegalArgumentException(NOT_AN_EXTENDABLE_DEVICE_MESSAGE);
    }
    sendDeviceManagementCommand(new RemoveResource(resource, parameters).setTimeout(timeout),
        listener);
  }

  /**
   * FORCE Remove the resource from the device.
   * This function is used to remove the previously added resources.
   * NOTE: this should be performed on resources which are lost.
   *
   * @param resource       The resource to be removed.
   * @param listener       The listener which will be triggered after the callback
   * @throws RuntimeException Throws an exception on communication failures and other
   *                          cases like removing resources which haven't been added.
   */
  public void forceRemoveResource(Resource resource, RemoveResourceListener listener) {
    if (!isExtendable) {
      throw new IllegalArgumentException(NOT_AN_EXTENDABLE_DEVICE_MESSAGE);
    }
    sendDeviceManagementCommand(new ForceRemoveResource(resource), listener);
  }

  /**
   * Change the connected Wi-Fi of the device.
   *
   * @param ssid         SSID of the Wi-Fi to connect
   * @param password     PASSWORD of the Wi-Fi to connect
   * @param listener     The listener which will be triggered after the callback
   * @throws RuntimeException Throws an exception on communication failures and other
   *                          cases like removing resources which haven't been added.
   */
  public void configureWiFi(String ssid, String password, DeviceManagementStatusListener listener) {
    sendDeviceManagementCommand(new ConfigureWiFi(ssid, password), listener);
  }

  /**
   * A call to respond to the commandInfoRequest.
   * This function is primarily designed to send some additional information
   * during the addResource process.
   *
   * @param infoResponse   The response object to be sent.
   * @param listener       The listener which will be triggered after the callback
   */
  public void sendInfoResponse(InfoResponse infoResponse, InfoResponseStatusListener listener) {
    if (0 > infoResponse.getRequestNodeId()) {
      infoResponse.setRequestNodeId(deviceNodeId);
    }

    CocoClient.getInstance().getNativeHandler().sendInfoResponse(infoResponse, listener);
  }

  /**
   * A function to send InfoRequest to a device.
   * The {@link InfoResponse} will be returned in {@link CallbacksInterface#infoResponseCallback}.
   *
   * @param infoRequest    The infoRequest object which will contain info to be queried on device
   * @param listener       The listener which will be triggered with
   *                       {@link CallbacksInterface#infoResponseCallback}
   */
  public void sendInfoRequest(InfoRequest infoRequest, InfoRequestStatusListener listener) {
    infoRequest.deviceNodeId = deviceNodeId;
    infoRequest.networkId = parent.getId();

    CocoClient.getInstance().getNativeHandler().sendInfoRequest(infoRequest, listener);
  }

  /**
   * A function to get the PSN of the device.
   *
   * @return String: the PSN of the device
   */
  public String getDevicePsn() {
    return devicePsn;
  }

  /**
   * A function to get the product name of the device.
   *
   * @return String: The name of the product
   */
  public String getProductName() {
    return productName;
  }

  /**
   * A function to get the Manufacturer name of the device.
   *
   * @return String: The name of the manufacturer
   */
  public String getMake() {
    return make;
  }

  /**
   * A function to get the model name of the device.
   *
   * @return String: The model name of the device
   */
  public String getModel() {
    return model;
  }

  /**
   * A function to get the version of firmware the device is running.
   *
   * @return String: The Firmware version
   */
  public String getFirmwareVersion() {
    return firmwareVersion;
  }

  /**
   * A function to get the {@link PowerSource} on which the device is running on.
   *
   * @return PowerSource: The PowerSource of the device
   */
  public PowerSource getPowerSource() {
    return powerSource;
  }

  /**
   * A function to get the {@link ReceiverType} of the device.
   *
   * @return ReceiverType: The type of the receiver on the device
   */
  public ReceiverType getReceiverType() {
    return receiverType;
  }

  /**
   * A function to get free ports to talk to this device.
   *
   * @return int[]: The ports over which communication can take place
   */
  public int[] getAvailableChannelPorts(int portCount) {
    return CocoClient.getInstance().getNativeHandler().getAvailableChannelPorts(this, portCount);
  }

  private boolean supportsProtocol(int protocol) {
    if (null == this.supportedProtocols) {
      return true;
    }

    for (int supportedProtocol : this.supportedProtocols) {
      if (protocol == supportedProtocol) {
        return true;
      }
    }

    return false;
  }

  /**
   * This function offers an easy check if the Device in interest supports any of
   * {@link RadioProtocol}.
   *
   * @param radioProtocol An enum with possible values of protocols for communication. Ex: Zigbee
   * @return True if device supports protocol entered.
   */
  public boolean supportsProtocol(RadioProtocol radioProtocol) {
    int protocolInt = radioProtocol.getInt();

    long protocolFlag = (1L << protocolInt);
    return ((this.supportedProtocolFlags & protocolFlag) == protocolFlag);

  }

  protected void internalSetProtocolSupported(int[] supportedProtocols) {
    if (null == supportedProtocols) {
      Log.d(TAG, "ignoring supported protocols");
      return;
    }

    this.supportedProtocols = supportedProtocols;

    this.supportedProtocolFlags = 0;

    for (int sp : this.supportedProtocols) {
      this.supportedProtocolFlags |= (1L << sp);
    }
  }

  /**
   * A getter of list of supported radio protocols.
   *
   * @return The supported protocols in {@link RadioProtocol}
   */
  public List<RadioProtocol> getSupportedProtocols() {
    if (null == supportedProtocols) {
      return Collections.emptyList();
    }

    List<RadioProtocol> radioProtocols = new ArrayList<>(3);

    for (int sp : this.supportedProtocols) {
      radioProtocols.add(RadioProtocol.getEnum(sp));
    }

    return radioProtocols;
  }

  protected void internalRemoveDevice() {}

  protected void internalSetName(String deviceName) {
    this.name = deviceName;
  }

  protected void internalSetDevicePsn(String devicePsn) {
    this.devicePsn = devicePsn;
  }

  protected void internalSetProductName(String productName) {
    this.productName = productName;
  }

  protected void internalSetMake(String make) {
    this.make = make;
  }

  protected void internalSetModel(String model) {
    this.model = model;
  }

  protected void internalSetFirmwareVersion(String firmwareVersion) {
    this.firmwareVersion = firmwareVersion;
  }

  protected void internalSetPowerSource(PowerSource powerSource) {
    this.powerSource = powerSource;
  }

  protected void internalSetReceiverType(ReceiverType receiverType) {
    this.receiverType = receiverType;
  }

  protected void internalSetExtendable(boolean extendable) {
    isExtendable = extendable;
  }

  protected void internalSetParent(Network parent) {
    this.parent = parent;
  }

  protected void internalSetReady(boolean ready) {
    this.ready = ready;
  }

  /**
   * This function checks membership of resource in resource map.
   *
   * @param resourceEui The ID of the resource
   * @return True if the resource exists in the resource map
   */
  public boolean containsResource(String resourceEui) {
    return getResourceMap().containsKey(resourceEui);
  }

  protected void internalAddResource(Resource resource) {
    getResourceMap().put(resource.getId(), resource);
  }

  protected void internalRemoveResource(String resourceEui) {
    getResourceMap().remove(resourceEui);
  }

  @Override
  public final int hashCode() {
    return Long.hashCode(getId());
  }

  @Override
  public String toString() {
    return "Device{"
        + "deviceNodeId=" + deviceNodeId
        + ", name='" + name + '\''
        + ", devicePsn='" + devicePsn + '\''
        + ", productName='" + productName + '\''
        + ", make='" + make + '\''
        + ", model='" + model + '\''
        + ", firmwareVersion='" + firmwareVersion + '\''
        + ", powerSource=" + powerSource
        + ", receiverType=" + receiverType
        + ", isExtendable=" + isExtendable
        + ", supportedProtocols=" + Arrays.toString(supportedProtocols)
        + ", parent=" + parent
        + ", ready=" + ready
        + ", supportedProtocolFlags=" + supportedProtocolFlags
        + ", resourceMap=" + resourceMap
        + '}';
  }

  @Override
  public Iterator<Resource> iterator() {
    return getResourceMap().values().iterator();
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendDeviceManagementCommand} while sending set resource name commands.
   */
  public static class SetResourceName extends Command<CommandId> {
    @SerializedName(Constants.RESOURCE_EUI)
    public String resourceEui;
    @SerializedName(Constants.RESOURCE_NAME)
    public String resourceName;

    /**
     * A constructor for this command.
     *
     * @param resource  resource which is to be named
     * @param newName   new name to replace existing one
     */
    public SetResourceName(Resource resource, String newName) {
      super(CommandId.UPD_RES_NAME);

      this.resourceName = newName;
      this.resourceEui = resource.getId();
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendDeviceManagementCommand} while sending set resource metadata commands.
   */
  public static class SetResourceMetadata extends Command<CommandId> {
    @SerializedName(Constants.RESOURCE_EUI)
    public String resourceEui;
    @SerializedName(Constants.RESOURCE_METADATA)
    public String resourceMetadata;

    /**
     * An overloaded constructor for this command.
     *
     * @param resource   Resource in interest whose metadata change is needed
     * @param metadata   The metadata to replace the existing metadata
     */
    public SetResourceMetadata(Resource resource, String metadata) {
      super(CommandId.UPD_RES_META);

      this.resourceMetadata = metadata;
      this.resourceEui = resource.getId();
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendDeviceManagementCommand} while sending discover commands.
   *
   * @see Device#discoverResources(RadioProtocol, long, DiscoverResourceListener)
   */
  public static class DiscoverResource extends Command<CommandId> {
    @SerializedName(Constants.PROTOCOL_ID)
    public RadioProtocol protocol;

    /**
     * Constructor for this command.
     *
     * @param radioProtocol protocol over which discoverResources() is called
     */
    public DiscoverResource(RadioProtocol radioProtocol) {
      super(CommandId.DISCOVER_RES);

      this.protocol = radioProtocol;
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendDeviceManagementCommand} while sending add resource commands.
   *
   * @see Device#addZigbeeResource(String, String, List, long, AddResourceListener)
   * @see Device#addTuyaResource(String, List, long, AddResourceListener)
   * @see Device#addZwaveResource(String, List, long, AddResourceListener)
   */
  public static class AddResource extends Command<CommandId> {
    @SerializedName(Constants.PROTOCOL_ID)
    public RadioProtocol protocolId;
    @SerializedName(Constants.RESOURCE_EUI)
    public String resourceEui;
    @SerializedName(Constants.RESOURCE_METADATA)
    public String metadata;
    @SerializedName(Constants.PARAMETERS)
    public List<Parameter<InfoResponse.Value>> parameters;

    private AddResource(RadioProtocol protocol, String metadata,
                        List<Parameter<InfoResponse.Value>> parameters) {
      super(CommandId.ADD_RES);

      this.resourceEui = null;
      this.protocolId = protocol;
      this.metadata = metadata;
      this.parameters =
          (null == parameters) ? Collections.emptyList() : new ArrayList<>(parameters);
    }

    public AddResource(String metadata, List<Parameter<InfoResponse.Value>> parameters) {
      this(RadioProtocol.ZWAVE, metadata, parameters);
      this.resourceEui = null;
    }

    public AddResource(String resourceEui, String metadata,
                       List<Parameter<InfoResponse.Value>> parameters) {
      this(RadioProtocol.ZIGBEE, metadata, parameters);
      this.resourceEui = resourceEui;
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendDeviceManagementCommand} while sending remove resource commands.
   *
   * @see Device#removeResource(Resource, List, long, RemoveResourceListener)
   */
  public static class RemoveResource extends Command<CommandId> {
    @SerializedName(Constants.RESOURCE_EUI)
    public String resourceEui;
    @SerializedName(Constants.PARAMETERS)
    public List<Parameter<InfoResponse.Value>> parameters;

    /**
     * Constructor for this command.
     *
     * @param resource       resource in interest to be removed forcefully
     * @param parameters     list of implementations of {@link InfoResponse.Value}
     */
    public RemoveResource(Resource resource, List<Parameter<InfoResponse.Value>> parameters) {
      super(CommandId.REM_RES);

      this.resourceEui = resource.getId();
      this.parameters =
          (null == parameters) ? Collections.emptyList() : new ArrayList<>(parameters);
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendDeviceManagementCommand} while sending force remove resource commands.
   *
   * @see Device#forceRemoveResource(Resource, RemoveResourceListener)
   */
  public static class ForceRemoveResource extends Command<CommandId> {
    @SerializedName(Constants.RESOURCE_EUI)
    public String resourceEui;

    /**
     * A constructor for this command.
     *
     * @param resource resource in interest to be removed forcefully
     */
    public ForceRemoveResource(Resource resource) {
      super(CommandId.FORCE_REM_RES);

      this.resourceEui = resource.getId();
    }
  }

  /**
   * This class can be sent as an argument to
   * {@link #sendDeviceManagementCommand} while sending configure Wi-Fi commands.
   *
   * @see Device#configureWiFi(String, String, DeviceManagementStatusListener)
   */
  public static class ConfigureWiFi extends Command<CommandId> {
    @SerializedName(Constants.SSID)
    public String ssid;
    @SerializedName(Constants.PASSWORD)
    public String password;

    public ConfigureWiFi() {
      super(CommandId.CONFIGURE_WIFI);
    }

    /**
     * An overloaded constructor for configure Wi-Fi command.
     *
     * @param ssid       SSID of the Wi-Fi to connect
     * @param password   PASSWORD of the Wi-Fi to connect
     */
    public ConfigureWiFi(String ssid, String password) {
      this();
      this.ssid = ssid;
      this.password = password;
    }
  }

  /**
   * Listener responsible to update developer with status of
   * {@link Device#sendDeviceManagementCommand(Command, DeviceManagementStatusListener)}.
   */
  public interface DeviceManagementStatusListener extends Listener {
    void onStatusChanged(CommandResponse<Device.CommandId> response, Resource[] resourcesImpacted,
                         Throwable tr);
  }

  /**
   * Listener responsible to trigger with
   * {@link NativeCallbacks#messageCallback}.
   */
  public interface MessageListener extends Listener {
    void onMessage(String title, String message, MessageType messageType);
  }

  /**
   * Listener responsible to trigger with
   * {@link NativeCallbacks#infoRequestCallback}.
   */
  public interface InfoRequestListener extends Listener {
    void onInfoRequest(InfoRequest infoRequest);
  }

  /**
   * Listener used to listen status of:
   * {@link Device#sendInfoRequest(InfoRequest, InfoRequestStatusListener)}.
   */
  public interface InfoRequestStatusListener extends Listener {
    void onInfoRequestStatus(InfoRequest request, Throwable tr);

    void onInfoResponse(InfoResponse response);
  }

  /**
   * Listener used to listen status of
   * {@link Device#sendInfoResponse(InfoResponse, InfoResponseStatusListener)}.
   */
  public interface InfoResponseStatusListener extends Listener {
    void onInfoResponseStatus(InfoResponse response, Throwable tr);
  }

  /**
   * Listener used to listen updates of:
   * {@link Device#discoverResources(RadioProtocol, long, DiscoverResourceListener)}.
   */
  public interface DiscoverResourceListener
      extends DeviceManagementStatusListener, InfoRequestListener, MessageListener {
  }

  /**
   * Listener used to listen updates of:
   * <ul>
   *   <li>{@link Device#addTuyaResource(String, List, long, AddResourceListener)};</li>
   *   <li>{@link Device#addZigbeeResource(String, String, List, long, AddResourceListener)};</li>
   *   <li>{@link Device#addZwaveResource(String, List, long, AddResourceListener)}.</li>
   * </ul>.
   */
  public interface AddResourceListener
      extends DeviceManagementStatusListener, InfoRequestListener, MessageListener {
  }

  /**
   * Listener used to listen updates of
   * {@link Device#removeResource(Resource, List, long, RemoveResourceListener)}.
   */
  public interface RemoveResourceListener extends DeviceManagementStatusListener, MessageListener {
  }
}
