/*
 * @authors   Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

/**
 * Factory design pattern that abstracts creation logic and invokes creation of objects.
 */
public class Factory {
  private static final String TAG = "Factory";

  private static final Creator DEFAULT_CREATOR = new DefaultCreator() {
  };
  static Creator creator = DEFAULT_CREATOR;

  /**
   * A utility function to create Network.
   *
   * @param id The Id of the network
   * @return Network which has been created
   */
  static Network createNetwork(String id) {
    Network network = null;

    try {
      network = creator.createNetwork(id);
    } catch (Throwable tr) {
      Log.e(TAG, "cannot create network", tr);
    }

    return (null != network) ? network : DEFAULT_CREATOR.createNetwork(id);
  }

  /**
   * A Utility function to create Device.
   *
   * @param deviceNodeId The unique id of the device in this network
   * @param parent       The parent network of the device
   * @return Device which has been created
   */
  static Device createDevice(long deviceNodeId, Network parent) {

    Device device = null;

    try {
      device = creator.createDevice(deviceNodeId, parent);
    } catch (Throwable tr) {
      Log.e(TAG, "cannot create device", tr);
    }

    return (null != device) ? device : DEFAULT_CREATOR.createDevice(deviceNodeId, parent);
  }

  /**
   * A Utility function to create Zone.
   *
   * @param zoneId The uniqueId of the zone in the network
   * @param parent The parent network of the zone
   * @return Zone which has been created
   */
  static Zone createZone(int zoneId, Network parent) {
    Zone zone = null;

    try {
      zone = creator.createZone(zoneId, parent);
    } catch (Throwable tr) {
      Log.e(TAG, "cannot create zone", tr);
    }

    return (null != zone) ? zone : DEFAULT_CREATOR.createZone(zoneId, parent);
  }

  /**
   * A utility function to create scene.
   *
   * @param id     The unique ID of scene in the network
   * @param parent The parent network of this scene
   * @return Scene which has been created
   */
  static Scene createScene(int id, Network parent) {
    Scene scene = null;

    try {
      scene = creator.createScene(id, parent);
    } catch (Throwable tr) {
      Log.e(TAG, "cannot create scene", tr);
    }

    return (null != scene) ? scene : DEFAULT_CREATOR.createScene(id, parent);
  }

  /**
   * A utility function to create Rule.
   *
   * @param id     The unique ID of rule in the network
   * @param parent The parent network of this rule
   * @return Rule which has been created
   */
  static Rule createRule(int id, Network parent) {

    Rule rule = null;

    try {
      rule = creator.createRule(id, parent);
    } catch (Throwable tr) {
      Log.e(TAG, "cannot create rule", tr);
    }

    return (null != rule) ? rule : DEFAULT_CREATOR.createRule(id, parent);
  }

  /**
   * A utility function to create resource.
   *
   * @param eui          The unique Id of the resource.
   * @param parentDevice The parent device which contains this resource.
   * @return             Resource which has been created
   */
  static Resource createResource(String eui, Device parentDevice, Zone parentZone) {

    Resource resource = null;

    try {
      resource = creator.createResource(eui, parentDevice, parentZone);
    } catch (Throwable tr) {
      Log.e(TAG, "cannot create resource", tr);
    }

    return (null != resource) ? resource :
        DEFAULT_CREATOR.createResource(eui, parentDevice, parentZone);
  }

  /**
   * A utility function to create an advertisement.
   *
   * @param eui          The unique Id of the resource.
   * @param parentDevice The parent device which contains this resource.
   * @return Resource which has been created
   */
  static Resource createAdvertResource(String eui, Device parentDevice) {

    Resource resource = null;

    try {
      resource = creator.createAdvertResource(eui, parentDevice);
    } catch (Throwable tr) {
      Log.e(TAG, "cannot create resource", tr);
    }

    return (null != resource) ? resource : DEFAULT_CREATOR.createAdvertResource(eui, parentDevice);
  }

  /**
   * A utility function to create attributes.
   *
   * @param id     The id of the attribute which is unique per attribute
   * @param parent The capability to which the attribute belongs
   * @return Attribute which has been created
   */
  static Attribute createAttribute(int id, Capability parent) {

    Attribute attribute = null;

    try {
      attribute = creator.createAttribute(id, parent);
    } catch (Throwable tr) {
      Log.e(TAG, "cannot create attribute", tr);
    }

    return (null != attribute) ? attribute : DEFAULT_CREATOR.createAttribute(id, parent);
  }

  /**
   * A utility function to create capabilities.
   *
   * @param capabilityId The unique id Of the capability
   * @param parent       The parent Resource of the capability.
   * @return Capability The capability that has been created.
   */
  static Capability createCapability(int capabilityId, Resource parent) {

    Capability capability = null;

    try {
      capability = creator.createCapability(capabilityId, parent);
    } catch (Throwable tr) {
      Log.e(TAG, "cannot create capability", tr);
    }

    return (null != capability) ? capability :
        DEFAULT_CREATOR.createCapability(capabilityId, parent);
  }
}
