/*
 * @author    Krishna, krishnasarma@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Finite State Machine manages states that an object can be based on the events in
 * {@link EventTable} that trigger these state changes.
 *
 * @param <S> type can be of any class which is an immediate subclass
 *            of {@link FiniteStateMachine.State}
 * @param <E> type can be of any class which is an immediate subclass
 *            of {@link FiniteStateMachine.Event}
 * @param <A> type of args for action callbacks
 */
public class FiniteStateMachine
    <S extends FiniteStateMachine.State, E extends FiniteStateMachine.Event, A> {
  private static final String TAG = "FiniteStateMachine";

  private int runCount = 0; // not MT-SAFE, just for logging purposes
  private S currentState;
  private final TimerCallback<A> timerCallback;
  private final List<EventTable<S, E, A>> eventTables;
  private final List<StateTable<S, E, A>> stateTables;

  /**
   * A marker interface that defines a class that implements it
   * to be an event that can cause a state change.
   */
  public interface Event {
  }

  /**
   * A marker interface that defines a class that implements it
   * to be a state of an object.
   */
  public interface State {
  }

  /**
   * A callback defining an action upon entry to state.
   *
   * @param <E> type can be of any class which is an immediate subclass
   *            of {@link FiniteStateMachine.Event}
   * @param <A> type of args for action callbacks
   */
  public interface EntryActionCallback<E, A> {
    E triggerEntry(A val);
  }

  /**
   * A callback defining an action upon exit from a state.
   *
   * @param <A> type of args for action callbacks
   */
  public interface ExitActionCallback<A> {
    void triggerExit(A val);
  }

  /**
   * A callback defining an action for an event.
   *
   * @param <A> type of args for action callbacks
   */
  public interface ActionCallback<A> {
    void triggerAction(A val);
  }

  /**
   * A callback defining an action upon timeout reached.
   *
   * @param <A> type of args for action callbacks
   */
  public interface TimerCallback<A> {
    void triggerTimer(long timeout, A val);
  }

  /**
   * An interface defining timeout for an event.
   */
  public interface FetchTimeout {
    long getTimeout();
  }

  /**
   * An Event table encapsulates complete information about event such as
   * previous state, next state and action callback etc.
   *
   * @param <S> type can be of any class which is an immediate subclass
   *            of {@link FiniteStateMachine.State}
   * @param <E> type can be of any class which is an immediate subclass
   *            of {@link FiniteStateMachine.Event}
   * @param <A> type of args for action callbacks
   */
  public static class EventTable<S extends State, E extends Event, A> {
    public final E event;
    public final S conditionState;
    public final S nextState;
    private final ActionCallback<A> actionCallback;

    /**
     * Constructor for an event.
     *
     * @param event          Event under interest to create an event table
     * @param conditionState state of an object for which event is defined to alter.
     * @param nextState      final state to which the event is defined to drive.
     * @param actionCallback callback to trigger action for the event.
     */
    public EventTable(E event, S conditionState, S nextState,
                      ActionCallback<A> actionCallback) {
      this.event = event;
      this.conditionState = conditionState;
      this.nextState = nextState;
      this.actionCallback = actionCallback;
    }

    @Override
    public String toString() {
      return "EventTable{"
          + "event=" + event
          + ", conditionState=" + conditionState
          + ", nextState=" + nextState
          + ", actionCallback=" + actionCallback
          + '}';
    }
  }

  /**
   * State table encapsulates complete information about the state such as timeout,
   * entry and exit callback actions etc.
   *
   * @param <S> type can be of any class which is an immediate subclass
   *            of {@link FiniteStateMachine.State}
   * @param <E> type can be of any class which is an immediate subclass
   *            of {@link FiniteStateMachine.Event}
   * @param <A> type of args for action callbacks
   */
  public static class StateTable<S extends State, E extends Event, A> {
    public final S state;
    public final FetchTimeout fetch;
    private final EntryActionCallback<E, A> entryCallback;
    private final ExitActionCallback<A> exitCallback;

    /**
     * A constructor for state table.
     *
     * @param state         State under interest to create a state table
     * @param timeout       timeout period defined for an action
     * @param entryCallback action callback that triggers when entry to this state happens
     * @param exitCallback  action callback that triggers when exit from this state happens
     */
    public StateTable(S state, long timeout, EntryActionCallback<E, A> entryCallback,
                      ExitActionCallback<A> exitCallback) {
      this.state = state;
      this.entryCallback = entryCallback;
      this.exitCallback = exitCallback;

      this.fetch = () -> timeout;
    }

    /**
     * An overloaded constructor for state table.
     *
     * @param state         State under interest to create a state table
     * @param fetch         FetchTimeout instance to define or retrieve timeout
     * @param entryCallback action callback that triggers when entry to this state happens
     * @param exitCallback  action callback that triggers when exit from this state happens
     */
    public StateTable(S state, FetchTimeout fetch, EntryActionCallback<E, A> entryCallback,
                      ExitActionCallback<A> exitCallback) {
      this.state = state;
      this.fetch = fetch;
      this.entryCallback = entryCallback;
      this.exitCallback = exitCallback;
    }

    @Override
    public String toString() {
      return "StateTable{"
          + "state=" + state
          + ", timeout=" + fetch.getTimeout()
          + ", fetch=" + fetch
          + ", entryCallback=" + entryCallback
          + ", exitCallback=" + exitCallback
          + '}';
    }
  }

  /**
   * Constructor for current class.
   *
   * @param initialState  primary state from which event triggers initially
   * @param timerCallback essential in triggering timer and set timeout conditions for
   *                      the finite state machine
   * @param eventTables   List of all {@link EventTable} possible for the machine
   * @param stateTables   List of all {@link StateTable} possible for the machine
   */
  public FiniteStateMachine(S initialState, TimerCallback<A> timerCallback,
                            List<EventTable<S, E, A>> eventTables,
                            List<StateTable<S, E, A>> stateTables) {
    this.currentState = initialState;
    this.timerCallback = timerCallback;
    this.eventTables = new ArrayList<>(eventTables);
    this.stateTables = new ArrayList<>(stateTables);
  }

  /**
   * This function handles an event using information encapsulated in event table to change states
   * of a finite state machine as defined.
   *
   * @param event Event under interest to be handled
   * @param args  args to trigger action callback for event
   */
  public synchronized void handleEvent(E event, A args) {
    int etIndex;
    int stIndex;
    E newEvent;

    Log.d(TAG, "handleEvent: started");

    if (null == event) {
      throw new NullPointerException();
    }

    do {
      ++runCount;

      Log.d(TAG, "event: " + event + ", currentState: " + currentState + ", runCount: " + runCount);

      StateTable<S, E, A> curStateTable;
      StateTable<S, E, A> oldStateTable;
      EventTable<S, E, A> eventTable;

      if (0 > (etIndex = getIndex(event))) {
        throw new
            IllegalArgumentException("unknown event: " + event + ", onState: " + currentState);
      }

      newEvent = null;
      eventTable = this.eventTables.get(etIndex);

      Log.v(TAG, "eventTable: " + eventTable);

      if (null == eventTable.nextState) {

        if (null != eventTable.actionCallback) {
          eventTable.actionCallback.triggerAction(args);
        }

        Log.v(TAG, "no next state, exiting");
        return;
      } else {
        if (0 > (stIndex = getIndex(this.currentState))) {
          throw new IllegalArgumentException("unknown current state " + this.currentState);
        }

        oldStateTable = this.stateTables.get(stIndex);

        Log.v(TAG, "oldTable: " + oldStateTable);

        if (null != eventTable.actionCallback) {
          eventTable.actionCallback.triggerAction(args);
        }

        if (null != oldStateTable.exitCallback) {
          oldStateTable.exitCallback.triggerExit(args);
        }

        this.currentState = eventTable.nextState;

        Log.v(TAG, "nextState: " + this.currentState);

        if (0 > (stIndex = getIndex(this.currentState))) {
          throw new IllegalArgumentException("unknown current state " + this.currentState);
        }

        curStateTable = this.stateTables.get(stIndex);

        Log.v(TAG, "curTable: " + curStateTable);

        if (null != timerCallback) {
          timerCallback.triggerTimer(curStateTable.fetch.getTimeout(), args);
        }

        if (null != curStateTable.entryCallback) {
          newEvent = curStateTable.entryCallback.triggerEntry(args);
        }
      }

      event = newEvent;

    } while (null != event);

    Log.d(TAG, "handleEvent: completed");
  }

  /**
   * To forcefully set the state of this machine.
   *
   * @param state desired state
   */
  public void unsafeSetState(S state) {
    this.currentState = state;
  }

  public S getCurrentState() {
    return currentState;
  }

  private int getIndex(E event) {

    for (int i = 0; i < eventTables.size(); i++) {
      EventTable<S, E, A> eventTable = eventTables.get(i);

      boolean isSameEvent = Objects.equals(eventTable.event, event);
      boolean isSameState = Objects.equals(eventTable.conditionState, currentState);

      if (isSameEvent && isSameState) {
        return i;
      }
    }

    return -1;
  }

  private int getIndex(S state) {

    for (int i = 0; i < stateTables.size(); i++) {
      if (stateTables.get(i).state == state
          || stateTables.get(i).state.equals(state)) {
        return i;
      }
    }

    return -1;
  }
}
