/*
 * @authors   Krishna, krishnasarma@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;

/**
 * Formatters for Date and Time.
 */
class Formatters {
  private static final String TAG = "Formatters";

  private static final String TIME_FORMAT = "HH:mm:ss";
  private static final String DATE_FORMAT = "yyyy-MM-dd";
  private static final String DATE_TIME_FORMAT = "yyyy-MM-dd HH:mm:ss";

  private static final Object ACCESS_LOCK = new Object();

  private static ZoneId zoneId = ZoneOffset.UTC;

  private static DateTimeFormatter dateFormatter;
  private static DateTimeFormatter timeFormatter;
  private static DateTimeFormatter dateTimeFormatter;

  static {
    Log.d(TAG, "using zone: " + zoneId);
  }

  // NOT exposed. Since, class is not public
  public static void setTimezone(ZoneId zoneId) {
    if (null == zoneId) {
      throw new IllegalArgumentException("invalid zone id");
    }

    Log.d(TAG, "changing to zone: " + zoneId + ", from zone: " + Formatters.getZoneId());

    if (zoneId.equals(getZoneId())) {
      return;
    }

    synchronized (ACCESS_LOCK) {
      Formatters.dateFormatter = null;
      Formatters.timeFormatter = null;
      Formatters.dateTimeFormatter = null;

      Formatters.zoneId = zoneId;
    }
  }

  static ZoneId getZoneId() {
    return zoneId;
  }

  static DateTimeFormatter getDateFormatter() {
    DateTimeFormatter copy;

    if (null != (copy = dateFormatter)) {
      return copy;
    }

    synchronized (ACCESS_LOCK) {
      DateTimeFormatter dateFormatter = Formatters.dateFormatter;

      if (null != dateFormatter) {
        return dateFormatter;
      }

      dateFormatter = DateTimeFormatter.ofPattern(DATE_FORMAT).withZone(getZoneId());

      return (Formatters.dateFormatter = dateFormatter);
    }
  }

  static DateTimeFormatter getTimeFormatter() {
    DateTimeFormatter copy;

    if (null != (copy = timeFormatter)) {
      return copy;
    }

    synchronized (ACCESS_LOCK) {
      DateTimeFormatter timeFormatter = Formatters.timeFormatter;

      if (null != timeFormatter) {
        return timeFormatter;
      }

      timeFormatter = DateTimeFormatter.ofPattern(TIME_FORMAT).withZone(getZoneId());

      return (Formatters.timeFormatter = timeFormatter);
    }
  }

  static DateTimeFormatter getDateTimeFormatter() {
    DateTimeFormatter copy;

    if (null != (copy = dateFormatter)) {
      return copy;
    }

    synchronized (ACCESS_LOCK) {
      DateTimeFormatter dateTimeFormatter = Formatters.dateTimeFormatter;

      if (null != dateTimeFormatter) {
        return dateTimeFormatter;
      }

      dateTimeFormatter = DateTimeFormatter.ofPattern(DATE_TIME_FORMAT).withZone(getZoneId());

      return (Formatters.dateTimeFormatter = dateTimeFormatter);
    }
  }
}
