/*
 * @authors   Krishna, krishnasarma@elear.solutions
 *            Akshay Mende, akshaymende@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.annotations.SerializedName;
import java.io.Serializable;

/**
 * This class encapsulates ids for Network, Device, Resource, Capability,
 * Attribute, Zone, Scene, Rule.
 */
public class Identifier {
  private static final Gson GSON = new Gson();

  @SerializedName(Constants.NETWORK_ID)
  public final String networkId;

  @SerializedName(value = Constants.NODE_ID, alternate = { Constants.DEVICE_NODE_ID })
  public final long deviceNodeId;

  @SerializedName(Constants.RESOURCE_EUI)
  public final String resourceEui;
  @SerializedName(Constants.CAPABILITY_ID)
  public final int capabilityId;
  @SerializedName(Constants.ATTRIBUTE_ID)
  public final int attributeId;
  @SerializedName(Constants.ZONE_ID)
  public final int zoneId;
  @SerializedName(Constants.SCENE_ID)
  public final int sceneId;
  @SerializedName(Constants.RULE_ID)
  public final int ruleId;

  /**
   * Constructor for current class.
   *
   * @param id identifier with information of all ids.
   */
  public Identifier(Identifier id) {

    this.networkId = (null == id) ? null : id.networkId;
    this.resourceEui = (null == id) ? null : id.resourceEui;

    this.deviceNodeId = (null == id) ? -1 : id.deviceNodeId;
    this.capabilityId = (null == id) ? -1 : id.capabilityId;
    this.attributeId = (null == id) ? -1 : id.attributeId;
    this.zoneId = (null == id) ? -1 : id.zoneId;
    this.sceneId = (null == id) ? -1 : id.sceneId;
    this.ruleId = (null == id) ? -1 : id.ruleId;
  }

  private Identifier(
      String networkId, long deviceNodeId, String resourceEui, int capabilityId, int attributeId) {
    this(networkId, deviceNodeId, resourceEui, capabilityId, attributeId, -1, -1, -1);
  }

  /**
   * An overloaded constructor for current class.
   *
   * @param networkId       The network to which the device belongs will respond
   * @param deviceNodeId    The device which has to respond to this request
   * @param resourceEui     The ID corresponding to the resource
   * @param capabilityId    CapabilityID of the recipient resource
   * @param attributeId     ID of the attribute value as in {@link Capability.AttributeId}
   * @param zoneId          ID of the zone in the network
   * @param sceneId         ID of the scene created by user in network
   * @param ruleId          ID of the rule created by user in network
   */
  public Identifier(
      String networkId, long deviceNodeId, String resourceEui,
      int capabilityId, int attributeId, int zoneId, int sceneId, int ruleId) {
    this.networkId = networkId;
    this.deviceNodeId = deviceNodeId;
    this.resourceEui = resourceEui;
    this.capabilityId = capabilityId;
    this.attributeId = attributeId;
    this.zoneId = zoneId;
    this.sceneId = sceneId;
    this.ruleId = ruleId;
  }

  /**
   * A function to get all identification information for device param.
   *
   * @param network Network object in interest, for an Identifier
   * @return Identifier object corresponding to the network
   *         with default deviceNodeId, resourceEui, capabilityId and attributeId values.
   */
  public static Identifier getIdentifier(Network network) {
    return new Identifier(network.getId(), -1,
        null, -1, -1);
  }

  /**
   * A function to get all identification information for device param.
   *
   * @param device Device object in interest, for an Identifier
   * @return Identifier object corresponding to the device
   *         with default resourceEui, capabilityId and attributeId values.
   */
  public static Identifier getIdentifier(Device device) {
    return new Identifier(device.getParent().getId(), device.getId(),
        null, -1, -1);
  }

  /**
   * A function to get all identification information for resource param.
   *
   * @param resource Resource object in interest, for an Identifier
   * @return Identifier object corresponding to the resource
   *         with default capabilityId and attributeId values.
   */
  public static Identifier getIdentifier(Resource resource) {
    return new Identifier(
        resource.getParent().getParent().getId(), resource.getParent().getId(),
        resource.getId(), -1, -1);
  }

  /**
   * A function to get all identification information for capability param.
   *
   * @param capability Capability object in interest, for an Identifier
   * @return Identifier object corresponding to the capability
   *         with default attributeId value.
   */
  public static Identifier getIdentifier(Capability capability) {
    return new Identifier(
        capability.getParent().getParent().getParent().getId(),
        capability.getParent().getParent().getId(), capability.getParent().getId(),
        capability.getId().getInt(), -1);
  }

  /**
   * A function to get all identification information for attribute param.
   *
   * @param attribute Attribute object in interest, for an Identifier
   * @return Identifier object corresponding to the attribute.
   */
  public static Identifier getIdentifier(Attribute attribute) {
    return new Identifier(attribute.getParent().getParent().getParent().getParent().getId(),
        attribute.getParent().getParent().getParent().getId(),
        attribute.getParent().getParent().getId(), attribute.getParent().getId().getInt(),
        attribute.getMapKey());
  }

  /**
   * A function to get all identification information for zone param.
   *
   * @param zone Zone object in interest, for an Identifier
   * @return Identifier object corresponding to the zone
   *         with default deviceNodeId, resourceEui, capabilityId and attributeId values.
   */
  public static Identifier getIdentifier(Zone zone) {
    return new Identifier(zone.getParent().getId(), -1, null, -1, -1, zone.getId(), -1, -1);
  }

  /**
   * A function to get all identification information for scene param.
   *
   * @param scene Scene object in interest, for an Identifier
   * @return Identifier object corresponding to the scene
   *         with default deviceNodeId, resourceEui, capabilityId and attributeId values.
   */
  public static Identifier getIdentifier(Scene scene) {
    return new Identifier(scene.getParent().getId(), -1, null, -1, -1, -1, scene.getId(), -1);
  }

  /**
   * A function to get all identification information for rule param.
   *
   * @param rule Rule object in interest, for an Identifier
   * @return Identifier object corresponding to the rule
   *         with default deviceNodeId, resourceEui, capabilityId and attributeId values.
   */
  public static Identifier getIdentifier(Rule rule) {
    return new Identifier(rule.getParent().getId(), -1, null, -1, -1, -1, -1, rule.getId());
  }

  /**
   * A generic method to fetch network for the identifier instance.
   *
   * @param <T> any immediate subclass of Network class
   * @return    Network corresponding to networkId for the Identifier instance
   */
  public <T extends Network> T getNetwork() {
    CocoClient cocoClient = CocoClient.getInstance();
    return (null == cocoClient || null == networkId) ? null : cocoClient.getNetwork(networkId);
  }

  /**
   * A generic method to fetch device for the identifier instance.
   *
   * @param <T> any immediate subclass of Device class
   * @return    Device corresponding to deviceNodeId for the Identifier instance
   */
  public <T extends Device> T getDevice() {
    Network network = getNetwork();
    return (null == network) ? null : network.getDevice(deviceNodeId);
  }

  /**
   * A generic method to fetch Resource for the identifier instance.
   *
   * @param <T> any immediate subclass of Resource class
   * @return    Resource corresponding to resourceEui for the Identifier instance
   */
  public <T extends Resource> T getResource() {
    Device device = getDevice();
    return (null == device || null == resourceEui) ? null : device.getResource(resourceEui);
  }

  /**
   * A generic method to fetch capability for the identifier instance.
   *
   * @param <T> any immediate subclass of Capability class
   * @return    Capability corresponding to capabilityId for the Identifier instance
   */
  public <T extends Capability> T getCapability() {
    Resource resource = getResource();
    return (null == resource) ? null :
        resource.getCapability(Capability.CapabilityId.getEnum(capabilityId));
  }

  /**
   * A generic method to fetch attribute for the identifier instance.
   *
   * @param <T> any immediate subclass of Attribute class
   * @return    Attribute corresponding to attributeId for the Identifier instance
   */
  public <T extends Attribute> T getAttribute() {
    Capability capability = getCapability();
    return (null == capability) ? null : capability.getAttribute(attributeId);
  }

  /**
   * A generic method to fetch zone for the identifier instance.
   *
   * @param <T> any immediate subclass of Zone class
   * @return    Zone corresponding to zoneId for the Identifier instance
   */
  public <T extends Zone> T getZone() {
    Network network = getNetwork();
    T zone = (null == network) ? null : network.getZone(zoneId);

    if (null == zone) {
      Resource resource = getResource();

      if (null != resource) {
        zone = resource.getParentZone();
      }
    }

    return zone;
  }

  /**
   * A generic method to fetch scene for the identifier instance.
   *
   * @param <T> any immediate subclass of Scene class
   * @return    Scene corresponding to sceneId for the Identifier instance
   */
  public <T extends Scene> T getScene() {
    Network network = getNetwork();
    return (null == network) ? null : network.getScene(sceneId);
  }

  /**
   * A generic method to fetch rule for the identifier instance.
   *
   * @param <T> any immediate subclass of Rule class
   * @return    Rule corresponding to ruleId for the Identifier instance
   */
  public <T extends Rule> T getRule() {
    Network network = getNetwork();
    return (null == network) ? null : network.getRule(ruleId);
  }

  @Override
  public boolean equals(Object obj) {
    if (!(obj instanceof Identifier)) {
      return false;
    }

    Identifier other = (Identifier) obj;
    boolean val;

    val = (equalsExtended(this.networkId, other.networkId));
    val &= (equalsExtended(this.resourceEui, other.resourceEui));
    val &= (this.deviceNodeId == other.deviceNodeId);
    val &= (this.capabilityId == other.capabilityId);
    val &= (this.attributeId == other.attributeId);
    val &= (this.zoneId == other.zoneId);
    val &= (this.sceneId == other.sceneId);

    return val;
  }

  @Override
  public int hashCode() {
    int result = 0;

    result = (31 * result) + (networkId != null ? networkId.hashCode() : 0);
    result = (31 * result) + (int) (deviceNodeId ^ (deviceNodeId >>> 32));
    result = (31 * result) + (resourceEui != null ? resourceEui.hashCode() : 0);
    result = (31 * result) + capabilityId;
    result = (31 * result) + attributeId;
    result = (31 * result) + zoneId;
    result = (31 * result) + sceneId;

    return result;
  }

  public JsonElement toJsonTree() {
    return toJsonTree(GSON);
  }

  public JsonElement toJsonTree(Gson gson) {
    return gson.toJsonTree(this);
  }

  public String toJson() {
    return toJson(GSON);
  }

  public String toJson(Gson gson) {
    return gson.toJson(this);
  }

  public Serializable toSerializable() {
    return new SerializedId(networkId, deviceNodeId, resourceEui, capabilityId, attributeId, zoneId,
        sceneId, ruleId);
  }

  /**
   * A function to make an identifier out of serializable param.
   *
   * @param serializable Serializable instance in interest, for an Identifier instance
   * @return Identifier object
   */
  public static Identifier fromSerializable(Serializable serializable) {
    SerializedId seId = null;

    if (serializable instanceof SerializedId) {
      seId = (SerializedId) serializable;
    }

    if (null == seId) {
      throw new IllegalArgumentException();
    }

    return new Identifier(seId.networkId, seId.deviceNodeId, seId.resourceEui,
        seId.capabilityId, seId.attributeId, seId.zoneId, seId.sceneId, seId.ruleId);
  }

  public static Identifier fromJson(String jsonIdentifier) {
    return fromJson(GSON, jsonIdentifier);
  }

  public static Identifier fromJson(Gson gson, String jsonIdentifier) {
    return gson.fromJson(jsonIdentifier, Identifier.class);
  }

  public static Identifier fromJsonTree(JsonElement jsonElement) {
    return fromJsonTree(GSON, jsonElement);
  }

  public static Identifier fromJsonTree(Gson gson, JsonElement jsonElement) {
    return gson.fromJson(jsonElement, Identifier.class);
  }

  @Override
  public String toString() {
    return "Identifier{"
        + "networkId='" + networkId + '\''
        + ", deviceNodeId=" + deviceNodeId
        + ", resourceEui='" + resourceEui + '\''
        + ", capabilityId=" + capabilityId
        + ", attributeId=" + attributeId
        + ", zoneId=" + zoneId
        + ", sceneId=" + sceneId
        + ", ruleId=" + ruleId
        + '}';
  }

  private static boolean equalsExtended(Object a, Object b) {
    if (a == null && b == null) {
      return true;
    } else if (a == null || b == null) {
      return false;
    }
    return a.equals(b);
  }

  /**
   * A helper class for generating Identifier object using {@link Identifier#fromSerializable}.
   */
  public static class SerializedId implements Serializable {
    private static final long serialVersionUID = 5863470550794984612L;

    private final String networkId;
    private final long deviceNodeId;
    private final String resourceEui;
    private final int capabilityId;
    private final int attributeId;
    private final int zoneId;
    private final int sceneId;
    private final int ruleId;

    private SerializedId(String networkId, long deviceNodeId, String resourceEui, int capabilityId,
                         int attributeId, int zoneId, int sceneId, int ruleId) {
      this.networkId = networkId;
      this.deviceNodeId = deviceNodeId;
      this.resourceEui = resourceEui;
      this.capabilityId = capabilityId;
      this.attributeId = attributeId;
      this.zoneId = zoneId;
      this.sceneId = sceneId;
      this.ruleId = ruleId;
    }
  }

  /**
   * This class serves as a source URI in storage content metadata{@link StorageContentMetadata}.
   */
  public static class SourceIdentifier extends Identifier {

    public SourceIdentifier(Identifier id) {
      super(id);
    }

    public SourceIdentifier(String networkId, long deviceNodeId, String resourceEui) {
      this(networkId, deviceNodeId, resourceEui, -1, -1);
    }

    public SourceIdentifier(String networkId, long deviceNodeId, String resourceEui,
                            int capabilityId, int attributeId) {
      super(networkId, deviceNodeId, resourceEui, capabilityId, attributeId, -1, -1, -1);
    }

    @Override
    public <T extends Zone> T getZone() {
      throw new IllegalArgumentException();
    }

    @Override
    public <T extends Scene> T getScene() {
      throw new IllegalArgumentException();
    }

    @Override
    public <T extends Rule> T getRule() {
      throw new IllegalArgumentException();
    }
  }

  /**
   * This class serves as an attribute in
   * {@link StorageContentMetadata} as upload triggered identifier.
   */
  public static class TriggerIdentifier extends Identifier {

    public TriggerIdentifier(Identifier id) {
      super(id);
    }

    public TriggerIdentifier(String networkId, long deviceNodeId, String resourceEui,
                             int capabilityId, int attributeId) {
      this(networkId, deviceNodeId, resourceEui, capabilityId, attributeId, 0, 0, 0);
    }

    public TriggerIdentifier(String networkId, long deviceNodeId, String resourceEui,
                             int capabilityId, int attributeId, int zoneId, int sceneId,
                             int ruleId) {
      super(networkId, deviceNodeId, resourceEui, capabilityId, attributeId, zoneId, sceneId,
          ruleId);
    }
  }
}
