/*
 * @author    Krishna, krishnasarma@elear.solutions
 * @copyright Copyright (c) 2019-2020 Elear Solutions Tech Private Limited. All rights
 *            reserved.
 * @license   To any person (the "Recipient") obtaining a copy of this software and
 *            associated documentation files (the "Software"):\n
 *            All information contained in or disclosed by this software is
 *            confidential and proprietary information of Elear Solutions Tech
 *            Private Limited and all rights therein are expressly reserved.
 *            By accepting this material the recipient agrees that this material and
 *            the information contained therein is held in confidence and in trust
 *            and will NOT be used, copied, modified, merged, published, distributed,
 *            sublicensed, reproduced in whole or in part, nor its contents revealed
 *            in any manner to others without the express written permission of
 *            Elear Solutions Tech Private Limited.
 */

package buzz.getcoco.iot;

import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParseException;
import com.google.gson.JsonPrimitive;
import com.google.gson.JsonSerializationContext;
import com.google.gson.JsonSerializer;
import com.google.gson.annotations.SerializedName;
import java.lang.reflect.Type;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * This class holds request in form of key-value pairs for keys as in the {@link Parameter.Key} and
 * sent to a device.
 */
public class InfoRequest {
  private static final Class<?>[] PARAM_CLASS_LUT;

  static {
    PARAM_CLASS_LUT = new Class<?>[Parameter.Key.values().length];

    PARAM_CLASS_LUT[Parameter.Key.USER_DEFINED.ordinal()]         = UserDefined.class;

    PARAM_CLASS_LUT[Parameter.Key.USERNAME.ordinal()]             = null;
    PARAM_CLASS_LUT[Parameter.Key.PASSWORD.ordinal()]             = null;
    PARAM_CLASS_LUT[Parameter.Key.PIN.ordinal()]                  = null;
    PARAM_CLASS_LUT[Parameter.Key.OTP.ordinal()]                  = null;
    PARAM_CLASS_LUT[Parameter.Key.INSTALL_CODE.ordinal()]         = null;
    PARAM_CLASS_LUT[Parameter.Key.CSA.ordinal()]                  = null;
    PARAM_CLASS_LUT[Parameter.Key.DSK.ordinal()]                  = null;
    PARAM_CLASS_LUT[Parameter.Key.RESOURCE_NAME.ordinal()]        = null;

    PARAM_CLASS_LUT[Parameter.Key.AVAIL_LISTEN_PORT.ordinal()]   = ListenPort.class;
    PARAM_CLASS_LUT[Parameter.Key.CONTENT_SEARCH.ordinal()]      = ContentSearch.class;
    PARAM_CLASS_LUT[Parameter.Key.CONTENT_PLAYBACK.ordinal()]    = ContentPlayback.class;
    PARAM_CLASS_LUT[Parameter.Key.STREAM_BUFFER_SIZES.ordinal()] = Parameter.BufferSizeValue.class;
    PARAM_CLASS_LUT[Parameter.Key.PAIRING_TYPE.ordinal()]        = Parameter.PairingTypeValue.class;
    PARAM_CLASS_LUT[Parameter.Key.TIMEZONE.ordinal()]            = Parameter.TimeZoneValue.class;

    PARAM_CLASS_LUT[Parameter.Key.REMOTE_PAIRING_CODES.ordinal()]     = RemotePairingCode.class;
    PARAM_CLASS_LUT[Parameter.Key.REMOTE_PAIR_CAPABILITIES.ordinal()]
        = RemotePairingCapabilities.class;
    PARAM_CLASS_LUT[Parameter.Key.REMOTE_SEARCH.ordinal()]            = Search.class;
    PARAM_CLASS_LUT[Parameter.Key.REMOTE_BUTTON_INFO.ordinal()]       = RemoteButtonInfo.class;
  }

  protected transient Command.State state;
  protected transient long requestId = 0;
  protected transient long cmdSeqNum = 0;
  protected transient String networkId;

  // when client gets  the request this will be the requestNodeId (id which requests)
  // when client sends the request this will be the responseNodeId(id which responds)
  protected transient long deviceNodeId;

  public long timeout;
  public final String messageText;
  public final List<Parameter<Value>> mandatoryParameters;
  public final List<Parameter<Value>> optionalParameters;

  /**
   * An enum denoting possible sorting order options.
   */
  public enum SortOrder {
    NONE,
    ASCENDING,
    DESCENDING;

    static SortOrder getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    int getInt() {
      return ordinal();
    }
  }

  /**
   * An enum denoting possible type of sort options.
   */
  public enum SortType {
    NONE,
    CREATED_TIME;

    static SortType getEnum(int index) {
      return Utils.findEnum(index, values());
    }

    int getInt() {
      return ordinal();
    }
  }

  /**
   * A marker interface for values corresponding to key as in {@link Parameter.Key} for InfoRequest.
   */
  public interface Value extends Parameter.Value {
  }

  /**
   * This class forms a value of request corresponding to {@link Parameter.Key#CONTENT_PLAYBACK}.
   *
   * @see InfoResponse.ContentPlayback
   */
  public static class ContentPlayback implements Value {
    @SerializedName(Constants.CONTENT_ID)
    public long contentId;
    @SerializedName(Constants.STREAM_PROTOCOL_TYPE)
    public CapabilityStorageControl.StreamProtocol streamProtocol;

    public ContentPlayback() {
    }
  }

  /**
   * This class forms a value of request corresponding to {@link Parameter.Key#AVAIL_LISTEN_PORT}.
   * Doesn't use a type adapter.
   *
   * @see InfoResponse.ListenPort
   */
  public static class ListenPort implements Value {
    @SerializedName(Constants.PORT_COUNT)
    public int portCount;
    @SerializedName(Constants.TRANSPORT_TYPE)
    public CapabilityTunnel.TransportType transportType;
    @SerializedName(Constants.CONSECUTIVE_PORT)
    public boolean consecutivePorts;

    public ListenPort() {
    }
  }

  /**
   * This class forms a value of request corresponding to {@link Parameter.Key#CONTENT_SEARCH}.
   *
   * @see InfoResponse.ContentSearch
   */
  public static class ContentSearch implements Value {

    @SerializedName(Constants.REQ_SEARCH_COUNT)
    public int reqSearchCount; // uint8_t

    @SerializedName(Constants.OFFSET_INDEX)
    public long offsetIndex;

    @SerializedName(Constants.START_EPOCH_TIME)
    public Instant startTime;

    @SerializedName(Constants.END_EPOCH_TIME)
    public Instant endTime;

    @SerializedName(Constants.CONTENT_TYPE)
    public StorageContentMetadata.ContentType contentType;

    @SerializedName(Constants.UPLOAD_TRIGGER_TYPE)
    public StorageContentMetadata.TriggerType triggerType;

    @SerializedName(Constants.UPLOAD_TRIGGERED_URI)
    public Identifier.TriggerIdentifier uploadTriggeredUri;

    @SerializedName(Constants.SOURCE_URI)
    public Identifier.SourceIdentifier sourceUri;

    @SerializedName(Constants.SORT_BY_TYPE)
    public SortType sortType;

    @SerializedName(Constants.SORT_BY_ORDER)
    public SortOrder sortOrder;

    public ContentSearch(StorageContentMetadata.ContentType contentType,
                         StorageContentMetadata.TriggerType triggerType) {
      this.contentType = contentType;
      this.triggerType = triggerType;
    }

    /**
     * This function creates an identifier for provided network.
     *
     * @param   n Network for which regex is requested
     * @return  Identifier object
     */
    public static Identifier getRegex(Network n) {
      return new Identifier(n.getId(), 0, null, -1, -1, 0, 0, 0);
    }

    /**
     * This function creates an identifier for provided device.
     *
     * @param   d Device for which regex is requested
     * @return  Identifier object
     */
    public static Identifier getRegex(Device d) {
      Network n = d.getParent();

      return new Identifier(n.getId(), d.getId(), null, -1, -1, 0, 0, 0);
    }

    /**
     * This function creates an identifier for provided resource.
     *
     * @param   r Resource for which regex is requested
     * @return  Identifier object
     */
    public static Identifier getRegex(Resource r) {
      Device d = r.getParent();
      Network n = d.getParent();

      return new Identifier(n.getId(), d.getId(), r.getId(), -1, -1, 0, 0, 0);
    }

    /**
     * This function creates an identifier for provided capability.
     *
     * @param   c Capability for which regex is requested
     * @return  Identifier object
     */
    public static Identifier getRegex(Capability c) {
      Resource r = c.getParent();
      Device d = r.getParent();
      Network n = d.getParent();

      return new Identifier(n.getId(), d.getId(), r.getId(), c.getId().getInt(), -1, 0, 0, 0);
    }

    /**
     * This function creates an identifier for provided attribute.
     *
     * @param   a Attribute for which regex is requested
     * @return  Identifier object
     */
    public static Identifier getRegex(Attribute a) {
      Capability c = a.getParent();
      Resource r = c.getParent();
      Device d = r.getParent();
      Network n = d.getParent();

      return new Identifier(n.getId(), d.getId(), r.getId(), c.getId().getInt(), a.getMapKey(), 0,
          0, 0);
    }
  }

  /**
   * A marker interface to classify search criteria.
   */
  public interface SearchCriteria {
  }

  /**
   * This class carries information whether the brand search is made for TV/AC.
   */
  public static class BrandSearchCriteria implements SearchCriteria {
    @SerializedName(Constants.CATEGORY_ID)
    public final CapabilityRemoteControl.Category category;
    @SerializedName(Constants.BRAND_SEARCH)
    public final String brandSearchQuery;

    public BrandSearchCriteria(CapabilityRemoteControl.Category category, String brandSearchQuery) {
      this.category = category;
      this.brandSearchQuery = brandSearchQuery;
    }

    @Override
    public String toString() {
      return "BrandSearchCriteria{"
          + "category=" + category
          + ", brandSearchQuery='" + brandSearchQuery + '\''
          + '}';
    }
  }

  /**
   * This class acts as a value of request corresponding to {@link Parameter.Key#REMOTE_SEARCH}.
   * aids in making a search request
   *
   * @see InfoResponse.Search
   */
  public static class Search implements Value {

    @SerializedName(Constants.REQ_SEARCH_COUNT)
    public final int requestSearchCount;
    @SerializedName(Constants.OFFSET_INDEX)
    public final int offsetIndex;
    @SerializedName(Constants.SORT_BY_TYPE)
    public final SortType sortType;
    @SerializedName(Constants.SORT_BY_ORDER)
    public final SortOrder sortOrder;
    @SerializedName(Constants.SEARCH_CRITERIA)
    public final SearchCriteria criteria;
    @SerializedName(Constants.SEARCH_TYPE)
    public final Parameter.SearchType searchType;
    @SerializedName(Constants.SOURCE_URI)
    public final Identifier.SourceIdentifier source;

    /**
     * A Constructor for this value.
     *
     * @param requestSearchCount   search count to be included in {@link InfoResponse.Search}
     * @param offsetIndex          index for request
     * @param searchType           an enum constant denoting type of search to be made, ex: brand
     *                             {@link Parameter.SearchType}
     * @param source               An Identifier object comprising complete identification of source
     * @param sortType             sort the results in provided type {@link SortType}
     * @param sortOrder            define ascending or descending order
     * @param criteria             additional search query for search request
     */
    public Search(int requestSearchCount, int offsetIndex, Parameter.SearchType searchType,
                  Identifier.SourceIdentifier source, SortType sortType, SortOrder sortOrder,
                  SearchCriteria criteria) {
      this.requestSearchCount = requestSearchCount;
      this.offsetIndex = offsetIndex;
      this.searchType = searchType;
      this.source = source;
      this.sortType = sortType;
      this.sortOrder = sortOrder;
      this.criteria = criteria;
    }
  }

  /**
   * This class forms a value of request corresponding to {@link Parameter.Key#USER_DEFINED}.
   *
   * @see InfoResponse.UserDefined
   */
  public static class UserDefined implements Value {
    // TODO: parser
    public String request;
  }

  /**
   * This class acts as a value of request corresponding to
   * {@link Parameter.Key#REMOTE_PAIRING_CODES}.
   *
   * @see InfoResponse.RemotePairingCode
   */
  public static class RemotePairingCode implements Value {
    @SerializedName(Constants.BRAND_ID)
    public int brandId;
    @SerializedName(Constants.CATEGORY_ID)
    public CapabilityRemoteControl.Category categoryId;
    @SerializedName(Constants.RESOURCE_EUI)
    public String resourceEui;

    /**
     * A constructor for this value.
     *
     * @param brandId                 integer id for brand
     * @param categoryId              integer category defined in
     *                                {@link CapabilityRemoteControl.Category}
     * @param remoteControlResource   resource for which pairing code request is made
     */
    public RemotePairingCode(int brandId, CapabilityRemoteControl.Category categoryId,
                             Resource remoteControlResource) {
      this.brandId = brandId;
      this.categoryId = categoryId;
      this.resourceEui = remoteControlResource.getId();
    }
  }

  /**
   * This class acts as a value of request corresponding to
   * {@link Parameter.Key#REMOTE_PAIR_CAPABILITIES}.
   *
   * @see InfoResponse.RemotePairingCapabilities
   */
  public static class RemotePairingCapabilities implements Value {
    @SerializedName(Constants.PAIRING_CODE)
    public int pairingCode;
    @SerializedName(Constants.CATEGORY_ID)
    public CapabilityRemoteControl.Category categoryId;
    @SerializedName(Constants.RESOURCE_EUI)
    public String resourceEui;

    /**
     * A Constructor for this value.
     *
     * @param pairingCode              Pairing code generated during pairing
     * @param category                 Category as defined in
     *                                 {@link CapabilityRemoteControl.Category}
     * @param remoteControlResource    Resource for which the request is made
     */
    public RemotePairingCapabilities(int pairingCode, CapabilityRemoteControl.Category category,
                                     Resource remoteControlResource) {
      this.pairingCode = pairingCode;
      this.categoryId = category;
      this.resourceEui = remoteControlResource.getId();
    }
  }

  /**
   * This class acts as value of request corresponding to {@link Parameter.Key#REMOTE_BUTTON_INFO}.
   *
   * @see InfoResponse.RemoteButtonInfo
   */
  public static class RemoteButtonInfo implements Value {
    @SerializedName(Constants.APPLIANCE_ID)
    public final String applianceId;
    @SerializedName(Constants.RESOURCE_EUI)
    public final String resourceEui;

    public RemoteButtonInfo(String applianceId, String resourceEui) {
      this.applianceId = applianceId;
      this.resourceEui = resourceEui;
    }
  }

  /**
   * A class to request information from a device.
   *
   * @param messageText         The message that has to be passed along with the request
   * @param mandatoryParameters The params which must be responded
   */
  public InfoRequest(String messageText, List<Parameter<Value>> mandatoryParameters) {
    this(messageText, mandatoryParameters, null);
  }

  /**
   * A class to request information from a device.
   *
   * @param messageText         The message that has to be passed along with the request
   * @param mandatoryParameters The params which must be responded
   * @param optionalParameters  The params which are optional to respond
   */
  public InfoRequest(String messageText, List<Parameter<Value>> mandatoryParameters,
                     List<Parameter<Value>> optionalParameters) {
    this(null, -1, messageText, mandatoryParameters, optionalParameters);
  }

  /**
   * A class to request information from a device.
   *
   * @param networkId           The network to which the device belongs will respond
   * @param deviceNodeId        The device which has to respond to this request
   * @param messageText         The message that has to be passed along with the request
   * @param mandatoryParameters The params which must be responded
   * @param optionalParameters  The params which are optional to respond
   */
  protected InfoRequest(String networkId, long deviceNodeId, String messageText,
                        List<Parameter<Value>> mandatoryParameters,
                        List<Parameter<Value>> optionalParameters) {
    this(networkId, deviceNodeId, messageText, 0, mandatoryParameters, optionalParameters);
  }

  /**
   * A class to request information from a device.
   *
   * @param networkId           The network to which the device belongs will respond
   * @param deviceNodeId        The device which has to respond to this request
   * @param messageText         The message that has to be passed along with the request
   * @param timeout             The timeout after which the command will return fail
   * @param mandatoryParameters The params which must be responded
   * @param optionalParameters  The params which are optional to respond
   */
  protected InfoRequest(String networkId, long deviceNodeId, String messageText, long timeout,
                        List<Parameter<Value>> mandatoryParameters,
                        List<Parameter<Value>> optionalParameters) {

    this.networkId = networkId;
    this.deviceNodeId = deviceNodeId;
    this.messageText = messageText;
    this.timeout = timeout;
    this.mandatoryParameters = (null == mandatoryParameters) ? Collections.emptyList() :
        new ArrayList<>(mandatoryParameters);
    this.optionalParameters = (null == optionalParameters) ? Collections.emptyList() :
        new ArrayList<>(optionalParameters);
  }

  static void init() {
    Command.GSON_BUILDER.registerTypeAdapter(InfoRequest.class, new InfoRequestParser());
    Command.GSON_BUILDER.registerTypeAdapter(ContentSearch.class, new ContentSearchParser());
    Command.GSON_BUILDER.registerTypeAdapter(InfoRequest.SortOrder.class,
        new InfoRequest.SortOrderParser());
    Command.GSON_BUILDER.registerTypeAdapter(InfoRequest.SortType.class,
        new InfoRequest.SortTypeParser());
  }

  public Device getDevice() {
    // TODO: check if requestNodeId to be used or responseNodeId to be used.
    return Utils.getDevice(CocoClient.getInstance().getNetworkMap(), networkId, deviceNodeId);
  }

  public long getCmdSeqNum() {
    return cmdSeqNum;
  }

  public long getRequestId() {
    return requestId;
  }

  public Command.State getState() {
    return state;
  }

  /**
   * A function to create/extract mandatory keys which should be responded
   * from mandatory parameters.
   *
   * @return keys which are mandatory to be responded.
   */
  public Parameter.Key[] mandatoryKeys() {
    if (null == mandatoryParameters) {
      return new Parameter.Key[0];
    }

    Parameter.Key[] keys = new Parameter.Key[mandatoryParameters.size()];

    for (int i = 0; i < mandatoryParameters.size(); i++) {
      keys[i] = mandatoryParameters.get(i).key;
    }

    return keys;
  }

  /**
   * A function to get optional parameters.
   *
   * @return an array of keys registered in optional parameters
   */
  public Parameter.Key[] optionalKeys() {
    if (null == optionalParameters) {
      return new Parameter.Key[0];
    }

    Parameter.Key[] keys = new Parameter.Key[optionalParameters.size()];

    for (int i = 0; i < optionalParameters.size(); i++) {
      keys[i] = optionalParameters.get(i).key;
    }

    return keys;
  }

  public long getTimeout() {
    return timeout;
  }

  public void setTimeout(long timeout) {
    this.timeout = timeout;
  }

  protected JsonElement toJson() {
    return Command.GSON_BUILDER.create().toJsonTree(this);
  }

  private static final class InfoRequestParser implements JsonSerializer<InfoRequest>,
      JsonDeserializer<InfoRequest> {

    @Override
    public JsonElement serialize(InfoRequest infoRequest, Type typeOfSrc,
                                 JsonSerializationContext context) {
      Gson gson = Command.GSON_BUILDER.create();
      JsonObject json = new JsonObject();
      List<Parameter<Value>> mandatoryParameters = infoRequest.mandatoryParameters;
      List<Parameter<Value>> optionalParameters = infoRequest.optionalParameters;

      JsonElement mandatoryParamJson = gson.toJsonTree(mandatoryParameters);
      JsonElement optionalParamJson = gson.toJsonTree(optionalParameters);

      json.addProperty(Constants.MESSAGE_TEXT, infoRequest.messageText);

      json.add(Constants.MANDATORY_INFO_REQUESTS, mandatoryParamJson);
      json.add(Constants.OPTIONAL_INFO_REQUESTS, optionalParamJson);

      return json;
    }

    @Override
    public InfoRequest deserialize(JsonElement jsonElement, Type typeOfT,
                                   JsonDeserializationContext context) throws
        JsonParseException {
      Gson gson = Command.GSON_BUILDER.create();
      JsonObject json = jsonElement.getAsJsonObject();
      List<Parameter<Value>> mandatoryParams;
      List<Parameter<Value>> optionalParams;
      JsonArray mandatoryParamsJson;
      JsonArray optionalParamsJson;

      mandatoryParamsJson = json.getAsJsonArray(Constants.MANDATORY_INFO_REQUESTS);
      optionalParamsJson = json.getAsJsonArray(Constants.OPTIONAL_INFO_REQUESTS);

      if (null != mandatoryParamsJson) {
        mandatoryParams = new ArrayList<>(mandatoryParamsJson.size());

        for (JsonElement paramJsonElement : mandatoryParamsJson) {
          Value value = null;
          JsonObject paramJson = paramJsonElement.getAsJsonObject();
          Parameter.Key key =
              gson.fromJson(paramJson.get(Constants.PARAMETER_KEY), Parameter.Key.class);
          Type keyType = PARAM_CLASS_LUT[key.ordinal()];

          if (null != keyType) {
            value = gson.fromJson(paramJson.get(Constants.PARAMETER_VALUE), keyType);
          }

          mandatoryParams.add(new Parameter<>(key, value));
        }
      } else {
        mandatoryParams = Collections.emptyList();
      }

      if (null != optionalParamsJson) {
        optionalParams = new ArrayList<>(optionalParamsJson.size());

        for (JsonElement paramJsonElement : optionalParamsJson) {
          Value value = null;
          JsonObject paramJson = paramJsonElement.getAsJsonObject();
          Parameter.Key key =
              gson.fromJson(paramJson.get(Constants.PARAMETER_KEY), Parameter.Key.class);
          Type keyType = PARAM_CLASS_LUT[key.ordinal()];

          if (null != keyType) {
            value = gson.fromJson(paramJson.get(Constants.PARAMETER_VALUE), keyType);
          }

          optionalParams.add(new Parameter<>(key, value));
        }
      } else {
        optionalParams = Collections.emptyList();
      }

      return new InfoRequest(json.getAsJsonPrimitive(Constants.MESSAGE_TEXT).getAsString(),
          mandatoryParams, optionalParams);
    }
  }

  private static final class ContentSearchParser
      implements JsonSerializer<ContentSearch>, JsonDeserializer<ContentSearch> {
    private static final boolean TIME_AS_STRING = true;

    @Override
    public JsonElement serialize(ContentSearch src, Type typeOfSrc,
                                 JsonSerializationContext context) {

      JsonObject jsonObject = new JsonObject();
      Instant startTime = src.startTime;
      Instant endTime = src.endTime;

      SortOrder sortOrder = src.sortOrder;
      SortType sortType = src.sortType;

      jsonObject.addProperty(Constants.REQ_SEARCH_COUNT, src.reqSearchCount);
      jsonObject.addProperty(Constants.CONTENT_TYPE, src.contentType.getInt());
      jsonObject.addProperty(Constants.UPLOAD_TRIGGER_TYPE, src.triggerType.getInt());
      jsonObject.addProperty(Constants.OFFSET_INDEX, src.offsetIndex);
      jsonObject.add(Constants.SORT_BY_ORDER, context.serialize(sortOrder));
      jsonObject.add(Constants.SORT_BY_TYPE, context.serialize(sortType));

      Identifier uploadTriggeredUri = src.uploadTriggeredUri;
      Identifier sourceUri = src.sourceUri;

      if (null != uploadTriggeredUri) {
        jsonObject.add(Constants.UPLOAD_TRIGGERED_URI, uploadTriggeredUri.toJsonTree());
      }

      if (null != sourceUri) {
        jsonObject.add(Constants.SOURCE_URI, sourceUri.toJsonTree());
      }

      if (TIME_AS_STRING) {
        String startTimeStr;
        String endTimeStr;

        startTimeStr =
            (null == startTime) ? null : Formatters.getDateTimeFormatter().format(startTime);
        endTimeStr = (null == endTime) ? null : Formatters.getDateTimeFormatter().format(endTime);

        jsonObject.addProperty(Constants.START_EPOCH_TIME, startTimeStr);
        jsonObject.addProperty(Constants.END_EPOCH_TIME, endTimeStr);
      } else {
        long startTimeLong = (null == startTime) ? 0 : startTime.getEpochSecond();
        long endTimeLong = (null == endTime) ? 0 : endTime.getEpochSecond();

        jsonObject.addProperty(Constants.START_EPOCH_TIME, startTimeLong);
        jsonObject.addProperty(Constants.END_EPOCH_TIME, endTimeLong);
      }

      return jsonObject;
    }

    @Override
    public ContentSearch deserialize(JsonElement jsonElement, Type typeOfT,
                                     JsonDeserializationContext context) throws JsonParseException {
      ContentSearch contentSearch;
      JsonObject json = jsonElement.getAsJsonObject();

      int contentType = json.get(Constants.CONTENT_TYPE).getAsInt();
      int triggerType = json.get(Constants.UPLOAD_TRIGGER_TYPE).getAsInt();

      contentSearch = new ContentSearch(StorageContentMetadata.ContentType.getEnum(contentType),
          StorageContentMetadata.TriggerType.getEnum(triggerType));

      JsonElement startTime = json.get(Constants.START_EPOCH_TIME);
      JsonElement endTime = json.get(Constants.END_EPOCH_TIME);

      contentSearch.startTime = getEpochTime(startTime);
      contentSearch.endTime = getEpochTime(endTime);

      JsonElement reqSearchCount = json.get(Constants.REQ_SEARCH_COUNT);
      JsonElement offsetIndex = json.get(Constants.OFFSET_INDEX);

      contentSearch.reqSearchCount =
          (null == reqSearchCount || !reqSearchCount.isJsonPrimitive()) ? 0 :
              reqSearchCount.getAsInt();
      contentSearch.offsetIndex = (null == offsetIndex || !offsetIndex.isJsonPrimitive()) ? 0 :
          offsetIndex.getAsJsonPrimitive().getAsInt();

      JsonElement triggerUri = json.get(Constants.UPLOAD_TRIGGERED_URI);
      JsonElement sourceUri = json.get(Constants.SOURCE_URI);

      contentSearch.sourceUri = new Identifier.SourceIdentifier(
          (null == sourceUri) ? null : Identifier.fromJsonTree(sourceUri));
      contentSearch.uploadTriggeredUri = new Identifier.TriggerIdentifier(
          (null == triggerUri) ? null : Identifier.fromJsonTree(triggerUri));

      JsonElement sortOrder = json.get(Constants.SORT_BY_ORDER);
      JsonElement sortType = json.get(Constants.SORT_BY_TYPE);

      contentSearch.sortOrder = context.deserialize(sortOrder, SortOrder.class);
      contentSearch.sortType = context.deserialize(sortType, SortType.class);

      return contentSearch;
    }

    private Instant getEpochTime(JsonElement json) {
      if (null != json && json.isJsonPrimitive()) {
        JsonPrimitive timePrim = json.getAsJsonPrimitive();

        if (timePrim.isString()) {
          String endTimeStr = timePrim.getAsString();

          return Instant.from(Formatters.getDateTimeFormatter().parse(endTimeStr));
        }

        if (timePrim.isNumber()) {
          long endTimeLong = timePrim.getAsLong();

          return Instant.ofEpochSecond(endTimeLong);
        }
      }

      return Instant.EPOCH;
    }
  }

  private static final class SortTypeParser
      implements JsonSerializer<SortType>, JsonDeserializer<SortType> {

    @Override
    public SortType deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context)
        throws JsonParseException {
      return SortType.getEnum(json.getAsInt());
    }

    @Override
    public JsonElement serialize(SortType src, Type typeOfSrc, JsonSerializationContext context) {
      return new JsonPrimitive(src.getInt());
    }
  }

  private static final class SortOrderParser
      implements JsonSerializer<SortOrder>, JsonDeserializer<SortOrder> {

    @Override
    public SortOrder deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context)
        throws JsonParseException {
      return SortOrder.getEnum(json.getAsInt());
    }

    @Override
    public JsonElement serialize(SortOrder src, Type typeOfSrc, JsonSerializationContext context) {
      return new JsonPrimitive(src.getInt());
    }
  }
}
